//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.args;

import de.caff.annotation.NotNull;

/**
 * A boolean switch is for a property that can be switched on.
 * <p>
 *   So either this switch appears in the command line (switched on),
 *   or it does not appear (switched off).
 * </p>
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 27, 2019
 */
public class SimpleOnSwitch
        extends AbstractSwitch
{
  private boolean value;

  /**
   * Constructor.
   * @param shortOn the character used for switching on, the switching
   *                off character will be automatically done by inverting
   *                the case (eg. for {@code k} it will become {@code K}),
   *                so the character should be in upper or lower case
   * @param longOn  the long form used for switching on, the long form
   *                for switching off will either be created by prepending
   *                {@code no-} of, if the long form begins with the prefix
   *                {@code with-}, by replacing this prefix with {@code without-}
   * @param description description of this switch
   */
  public SimpleOnSwitch(@NotNull Character shortOn, @NotNull String longOn,
                        @NotNull String description)
  {
    super(shortOn, longOn, description);
  }

  @Override
  public boolean isVarLength() {
    return false;
  }

  @NotNull
  @Override
  public String getAppearance()
  {
    return "-" + shortForm + "|--" + longForm;
  }

  @Override
  public void found(char shortSwitch)
  {
    value = true;
  }

  @Override
  public void found(@NotNull String longSwitch)
  {
    value = true;
  }

  /**
   * Get the value of this switch.
   * @return {@code true}: switched on<br>
   *         {@code false}: switched off
   */
  public boolean isSwitchedOn()
  {
    return value;
  }
}
