//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.args;

import de.caff.annotation.NotNull;
import de.caff.generics.Types;

import java.util.Collection;
import java.util.Collections;

/**
 * A boolean switch is for a property that can be switched on or off.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 27, 2019
 */
public class SimpleOnOffSwitch
        extends AbstractSwitch
{
  public static final String LONG_PREFIX_WITH = "with-";
  public static final String LONG_PREFIX_WITHOUT = "without-";
  public static final String LONG_PREFIX_NO = "no-";

  @NotNull
  private final Character shortOff;
  @NotNull
  private final String longOff;

  private boolean value;

  /**
   * Constructor.
   * @param shortOn the character used for switching on, the switching
   *                off character will be automatically done by inverting
   *                the case (eg. for {@code k} it will become {@code K}),
   *                so the character should be in upper or lower case
   * @param longOn  the long form used for switching on, the long form
   *                for switching off will either be created by prepending
   *                {@code no-} of, if the long form begins with the prefix
   *                {@code with-}, by replacing this prefix with {@code without-}
   * @param defaultValue default value use when this switch does not appear on the command line
   * @param description description of this switch
   */
  public SimpleOnOffSwitch(@NotNull Character shortOn, @NotNull String longOn, boolean defaultValue,
                           @NotNull String description)
  {
    super(shortOn, longOn, description);
    if (Character.isLowerCase(shortOn)) {
      shortOff = Character.toUpperCase(shortOn);
    }
    else if (Character.isUpperCase(shortOn)) {
      shortOff = Character.toLowerCase(shortOn);
    }
    else {
      throw new IllegalArgumentException("Character '"+shortOn+"' does not have a case!");
    }
    if (shortOff.equals(shortForm)) {
      throw new IllegalArgumentException("Character '"+shortOn+"' is the same in upper and lower case!");
    }
    if (longOn.startsWith(LONG_PREFIX_WITH)) {
      longOff = LONG_PREFIX_WITHOUT + longOn.substring(LONG_PREFIX_WITH.length());
    }
    else {
      longOff = LONG_PREFIX_NO+longOn;
    }
    value = defaultValue;
  }

  @NotNull
  private static Switch view(@NotNull Switch baseSwitch,
                             @NotNull Character shortSwitch,
                             @NotNull String longSwitch,
                             @NotNull String description)
  {
    return new Switch()
    {
      @NotNull
      @Override
      public Collection<Character> getShortForms()
      {
        return Collections.singleton(shortSwitch);
      }

      @NotNull
      @Override
      public Collection<String> getLongForms()
      {
        return Collections.singleton(longSwitch);
      }

      @Override
      public boolean isVarLength() {
        return false;
      }

      @Override
      public void found(char shortSwitch)
      {
        baseSwitch.found(shortSwitch);
      }

      @Override
      public void found(@NotNull String longSwitch)
      {
        baseSwitch.found(longSwitch);
      }

      @NotNull
      @Override
      public String getAppearance()
      {
        return String.format("-%s|--%s", shortSwitch, longSwitch);
      }

      @NotNull
      @Override
      public String getDescription()
      {
        return description;
      }
    };

  }

  @Override
  public boolean isVarLength() {
    return false;
  }

  /**
   * View the 'on' state as a single switch state.
   * @param description description of the on state
   * @return switch view of only the on state switches
   */
  @NotNull
  public Switch viewOn(@NotNull String description)
  {
    return view(this, shortForm, longForm, description);
  }

  /**
   * View the 'off' state as a single switch state.
   * @param description description of the off state
   * @return switch view of only the off state switches
   */
  @NotNull
  public Switch viewOff(@NotNull String description)
  {
    return view(this, shortOff, longOff, description);
  }

  @NotNull
  @Override
  public Collection<Character> getShortForms()
  {
    return Types.asList(shortForm, shortOff);
  }

  @NotNull
  @Override
  public Collection<String> getLongForms()
  {
    return Types.asList(longForm, longOff);
  }

  @NotNull
  @Override
  public String getAppearance()
  {
    return String.format("-%s|-%s|--%s|--%s", shortForm, shortOff, longForm, longOff);
  }

  @Override
  public void found(char shortSwitch)
  {
    value = shortSwitch == shortForm;
  }

  @Override
  public void found(@NotNull String longSwitch)
  {
    value = longSwitch.equals(longForm);
  }

  /**
   * Get the value of this switch.
   * @return {@code true}: switched on<br>
   *         {@code false}: switched off
   */
  public boolean isSwitchedOn()
  {
    return value;
  }
}
