// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.args;

import de.caff.annotation.NotNull;

import java.util.ArrayList;
import java.util.List;

/**
 * A multiple argument.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 27, 2019
 */
public class MultipleArgument
        extends AbstractBasicMultipleArgument {
  @NotNull
  private final List<String> collectedArguments = new ArrayList<>();

  /**
   * Constructor.
   * @param name  argument name
   * @param description argument description
   * @param minCount minimal expected count
   * @param maxCount maximal expected count
   */
  public MultipleArgument(@NotNull String name,
                          @NotNull String description,
                          int minCount, int maxCount)
  {
    super(name, description, minCount, maxCount);
  }
  /**
   * Constructor.
   * This will allow an unlimited number of arguments.
   * @param name  argument name
   * @param description argument description
   * @param minCount minimal expected count
   */
  public MultipleArgument(@NotNull String name,
                          @NotNull String description,
                          int minCount)
  {
    this(name, description, minCount, Integer.MAX_VALUE);
  }

  /**
   * Are there a correct number of collected arguments?
   * @return {@code true} if the number is in the expected range,<br>
   *         {@code false} otherwise
   */
  public boolean isCorrect()
  {
    return minCount <= collectedArguments.size()  &&
           maxCount >= collectedArguments.size();
  }

  /**
   * Get the collected arguments.
   * @return collected arguments
   */
  @NotNull
  public List<String> getCollectedArguments()
  {
    return collectedArguments;
  }

  @Override
  public boolean consumeArgument(@NotNull String arg)
  {
    collectedArguments.add(arg);
    return collectedArguments.size() < maxCount;
  }

}

