//=============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util.args;

import de.caff.annotation.NotNull;
import de.caff.generics.Pair;
import de.caff.generics.Types;
import de.caff.util.Utility;

import java.io.PrintStream;
import java.util.*;

/**
 * Application command-line helper.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 27, 2019
 */
public class CommandLine
{
  private static final boolean DEBUG = Utility.getBooleanParameter("cmdline.debug", false);

  public static final String PREFIX_SHORT = "-";
  public static final String PREFIX_LONG = "--";
  public static final String LONG_ARG_SEP = "=";
  public static final char NON_BREAKING_SPACE = '\u00a0';

  @NotNull
  private final Map<Character, Switch> shortForms = new HashMap<>();
  @NotNull
  private final Map<String, Switch> longForms = new HashMap<>();
  @NotNull
  private final List<Argument> arguments = new LinkedList<>();
  @NotNull
  private final String line;
  @NotNull
  private final List<Pair<String>> descriptions = new LinkedList<>();

  /**
   * Constructor.
   * @param switchesOrArguments switches or arguments used on the command line
   */
  public CommandLine(@NotNull SwitchOrArgument ... switchesOrArguments)
  {
    this(Types.asList(switchesOrArguments));
  }

  /**
   * Constructor.
   * @param switchesOrArguments switches or arguments used on the command line
   */
  public CommandLine(@NotNull Iterable<SwitchOrArgument> switchesOrArguments)
  {
    boolean hadOptionalArg = false;
    final StringBuilder lineBuilder = new StringBuilder();
    boolean hadVarArgs = false;
    for (SwitchOrArgument soa : switchesOrArguments) {
      if (hadVarArgs) {
        throw new IllegalArgumentException("There is only one var-arg argument or switch allowed, and it has to appear as last item!");
      }
      hadVarArgs = soa.isVarLength();

      if (soa.isSwitch()) {
        final Switch s = (Switch)soa;
        for (Character ch : s.getShortForms()) {
          final Switch other = shortForms.get(ch);
          if (other != null) {
            throw new IllegalArgumentException(
                    String.format("Duplicate usage of short switch '%s%s'!", PREFIX_SHORT, ch));
          }
          shortForms.put(ch, s);
        }
        for (String str : s.getLongForms()) {
          checkLongForm(str);
          final Switch other = longForms.get(str);
          if (other != null) {
            throw new IllegalArgumentException(
                    String.format("Duplicate usage of long switch '%s%s'!", PREFIX_LONG, str));
          }
          longForms.put(str, s);
        }
        lineBuilder.append(forCommandLine(s));
      }
      else {
        final Argument arg = (Argument)soa;
        if (arg.getMinimalCount() < 0) {
          throw new IllegalArgumentException(String.format("Invalid minimal count %d for argument %s!", arg.getMinimalCount(), arg.getAppearance()));
        }
        if (arg.getMaximalCount() <= 0) {
          throw new IllegalArgumentException(String.format("Invalid maximal count %d for argument %s!", arg.getMaximalCount(), arg.getAppearance()));
        }
        if (arg.isOptional()) {
          hadOptionalArg = true;
        }
        else {
          if (hadOptionalArg) {
            throw new IllegalArgumentException(String.format("Optional arguments followed by non-optional argument %s is not supported!", arg.getAppearance()));
          }
        }
        lineBuilder.append(forCommandLine(arg));
        arguments.add(arg);
      }
      lineBuilder.append(' ');
      descriptions.add(Pair.createPair(soa.getAppearance(), soa.getDescription()));
    }
    line = lineBuilder.toString();

    // todo: check fulfillment of restrictions
  }

  @NotNull
  private static String forCommandLine(@NotNull SwitchOrArgument soa)
  {
    String appearance = soa.getAppearance();
    if (!soa.isSwitch()) {
      int max = ((Argument)soa).getMaximalCount();
      if (max > 3) {
        appearance += "...";
      }
      else {
        while (--max > 1) {
          appearance += " " + soa.getAppearance();
        }
      }
    }
    if (soa.isOptional()) {
      appearance = "[" + appearance + "]";
    }
    return appearance;
  }

  /**
   * Evaluate the command line.
   * @param args command line arguments
   * @throws UnknownSwitchException  if an unknown switch appears
   * @throws UnexpectedArgumentException if a long switch not requiring an argument comes with an argument
   * @throws TooManyArgumentsException if there are no more switches or arguments for consuming the command line args
   * @throws MissingSwitchArgumentException if there are not enough arguments for a switch
   */
  public void evaluate(@NotNull String[] args)
          throws UnknownSwitchException, UnexpectedArgumentException, TooManyArgumentsException, MissingSwitchArgumentException {
    evaluate(new LinkedList<>(Types.asList(args)));  // list will be modified!
  }

  /**
   * Evaluate the command line.
   * @param args command line arguments
   * @throws UnknownSwitchException  if an unknown switch appears
   * @throws UnexpectedArgumentException if a long switch not requiring an argument comes with an argument
   * @throws TooManyArgumentsException if there are no more switches or arguments for consuming the command line args
   * @throws MissingSwitchArgumentException ifa required switch is not present
   */
  public void evaluate(@NotNull List<String> args)
          throws UnknownSwitchException, UnexpectedArgumentException, TooManyArgumentsException, MissingSwitchArgumentException
  {
    boolean noMoreSwitches = false;
    final Iterator<Argument> argIt = arguments.iterator();
    while (!args.isEmpty()) {
      String arg = args.remove(0);
      if (!noMoreSwitches) {
        if (arg.startsWith(PREFIX_LONG)) {
          if (arg.equals(PREFIX_LONG)) {
            if (DEBUG) {
              System.out.println("CMD: <no-more-switches>");
            }
            noMoreSwitches = true;
          }
          else {
            String longForm = arg.substring(2);
            String appendedArg = null;
            final int eq = longForm.indexOf(LONG_ARG_SEP);
            if (eq > 0) {
              appendedArg = longForm.substring(eq + 1);
              longForm = longForm.substring(0, eq);
            }
            final Switch sw = longForms.get(longForm);
            if (sw == null) {
              throw new UnknownSwitchException(PREFIX_LONG + longForm);
            }
            if (DEBUG) {
              System.out.println("Long switch: --"+longForm);
            }
            sw.found(longForm);
            if (appendedArg != null) {
              if (!sw.needsArguments()) {
                throw new UnexpectedArgumentException(PREFIX_LONG + longForm, appendedArg);
              }
              // make append arg appear in the standard args
              args.add(0, appendedArg);
            }
            if (sw.needsArguments()) {
              while (!args.isEmpty()  &&  sw.consumeArgument(args.remove(0))) {
              }
            }
          }
          continue;
        }
        else if (arg.startsWith(PREFIX_SHORT)) {
          String shortForm = arg.substring(1);
          if (shortForm.isEmpty()) {
            throw new UnknownSwitchException("-");
          }
          while (!shortForm.isEmpty()) {
            final char switchChar = shortForm.charAt(0);
            shortForm = shortForm.substring(1);
            final Switch sw = shortForms.get(switchChar);
            if (sw == null) {
              throw new UnknownSwitchException(PREFIX_SHORT + switchChar);
            }
            if (DEBUG) {
              System.out.print("CMD: Short switch: -" + switchChar);
            }
            sw.found(switchChar);
            if (sw.needsArguments()) {
              if (args.isEmpty()) {
                throw new MissingSwitchArgumentException(shortForm);
              }
              arg = args.remove(0);
              if (DEBUG) {
                System.out.print(" " + arg);
              }
              while (sw.consumeArgument(arg)  &&  !args.isEmpty()) {
                arg = args.remove(0);
                if (DEBUG) {
                  System.out.print(" " + arg);
                }
              }
            }
            if (DEBUG) {
              System.out.println();
            }
          }
          continue;
        }
        // no switch
        if (!argIt.hasNext()) {
          // no argument available
          throw new TooManyArgumentsException(arg, args);
        }
        final Argument argument = argIt.next();
        if (argument.consumeArgument(arg)) {
          while (!args.isEmpty()) {
            final String a = args.remove(0);
            if (DEBUG) {
              System.out.println("CMD: Argument: " + a);
            }
            if (!argument.consumeArgument(a)) {
              break;
            }
          }
        }
      }
    }
  }

  public static void checkLongForm(@NotNull String longForm)
  {
    if (longForm.isEmpty()) {
      throw new IllegalArgumentException("Empty long form not allowed!");
    }
    for (char ch : longForm.toCharArray()) {
      if (!Character.isLetterOrDigit(ch)  &&  ch != '-') {
        throw new IllegalArgumentException("Illegal character '"+ch+"' in long form!");
      }
    }
  }

  @NotNull
  private static String spaces(int num)
  {
    final StringBuilder sb = new StringBuilder(num);
    while (num-- > 0) {
      sb.append(' ');
    }
    return sb.toString();
  }

  @NotNull
  private static Pair<String> splitBefore(@NotNull String str, final int pos)
  {
    int p = pos;
    while (p >= 0) {
      final char ch = str.charAt(p);
      if (Character.isSpaceChar(ch)  &&  ch != NON_BREAKING_SPACE) {
        return Pair.createPair(str.substring(0, p),
                               str.substring(p + 1));
      }
      --p;
    }
    // ugly, just split at the given position
    return Pair.createPair(str.substring(0, pos),
                           str.substring(pos));
  }

  @NotNull
  public String getUsage(final int lineLength, final int indent)
  {
    final int lineLen1 = lineLength - indent;
    final int lineLen2 = lineLen1 - indent;
    final String indent1 = spaces(indent);
    final String indent2 = indent1 + indent1;

    final StringBuilder sb = new StringBuilder("Command Line:\n");
    String out = line;

    if (out.length() > lineLength) {
      Pair<String> parts = splitBefore(out, lineLength);
      sb.append(parts.first).append('\n');
      out = parts.second;
      while (out.length() > lineLen1) {
        parts = splitBefore(out, lineLen1);
        sb.append(indent1).append(parts.first).append('\n');
        out = parts.second;
      }
      if (!out.isEmpty()) {
        sb.append(indent1).append(out).append('\n');
      }
    }

    sb.append("\nSwitches and Arguments:");
    for (Pair<String> descr : descriptions) {
      sb.append('\n');
      sb.append(indent1).append(descr.first).append('\n');
      for (String line : descr.second.split("\n")) {
        while (line.length() > lineLen2) {
          final Pair<String> parts = splitBefore(line, lineLen2);
          sb.append(indent2).append(parts.first).append('\n');
          line = parts.second;
        }
        sb.append(indent2).append(line).append('\n');
      }
    }
    return sb.toString();
  }

  public void printUsageAndExit(@NotNull PrintStream out,
                                int exitCode)
  {
    out.print(getUsage(80, 4));
    System.exit(exitCode);
  }

  public void printUsageAndExit(int exitCode)
  {
    printUsageAndExit(System.out, exitCode);
  }
}
