// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2003-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import javax.swing.*;
import java.util.*;

/**
 *  A Runnable which knows of exceptions.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public abstract class Worker
        implements Runnable
{
  private static final ThreadGroup WORKER_THREAD_GROUP = new ThreadGroup("de.caff.util.Worker") {

  };
  /** The exception catched (if any). */
  private Throwable catched;
  /** The listeners for finished work. */
  private final List<KnockOffListener>  knockOffListeners = new LinkedList<>();
  /** Shall this worker stop? */
  private boolean   stopped = false;

  /**
   *  Default constructor.
   */
  public Worker()
  {
  }

  /**
   *  Constructor taking an initial knock off listener.
   *  @param listener knock off listener
   */
  public Worker(@NotNull KnockOffListener listener)
  {
    addKnockOffListener(listener);
  }

  /**
   * When an object implementing interface {@code Runnable} is used
   * to create a thread, starting the thread causes the object's
   * {@code run} method to be called in that separately executing
   * thread.
   * <p>
   * The general contract of the method {@code run} is that it may
   * take any action whatsoever.
   *
   * @see     Thread#run()
   */
  @Override
  public synchronized void run()
  {
    try {
      execute();
    } catch (Throwable x) {
      catched = x;
    } finally {
      informKnockOffListeners();
    }
  }

  /**
   *  Implement this in extending classes to do the work.
   *  @throws Throwable any exception thrown during work
   */
  protected abstract void execute() throws Throwable;

  /**
   *  Get the throwable caught during work, if there happened one.
   *  @return the throwable or {@code null} if there didn't occur any
   */
  @Nullable
  public Throwable getCaughtThrowable()
  {
    return catched;
  }

  /**
   *  This throws the caught exception if there occurred one.
   *  @throws Throwable exception caught during execute of work
   */
  public void rethrow()
          throws Throwable
  {
    if (catched != null) {
      throw catched;
    }
  }

  /**
   *  Add a knock off listener.
   *  The knock off listener will be called in the AWT event thread after the worker has finished.
   *  @param listener new listener
   */
  public void addKnockOffListener(@NotNull KnockOffListener listener)
  {
    synchronized (knockOffListeners) {
      knockOffListeners.add(listener);
    }
  }

  /**
   *  Remove a knock off listener.
   *  @param listener listener to remove
   */
  public void removeKnockOffListener(@NotNull KnockOffListener listener)
  {
    synchronized (knockOffListeners) {
      knockOffListeners.remove(listener);
    }
  }

  /**
   *  Call all knockoff listeners.
   */
  protected void informKnockOffListeners()
  {
    invokeInEventDispatchThread(() -> {
      Collection<KnockOffListener> knockOffListenersCopy;
      synchronized (knockOffListeners) {
        knockOffListenersCopy = new ArrayList<>(knockOffListeners);
      }
      for (KnockOffListener listener: knockOffListenersCopy) {
        listener.knockedOff(Worker.this);
      }
    });
  }

  /**
   *  Removes all KnockOffListeners and sets the internal state to stopped.
   *  Implementing classes may check the {@link #isStopped()} method regularly
   *  and stop executing when it returns true.
   *  <p>
   *  This is a workaround for the broken behavior of the Thread.stop() method.
   */
  public void stop()
  {
    stop(true);
  }

  /**
   *  Sets the internal state to stopped.
   *  Implementing classes may check the {@link #isStopped()} method regularly
   *  and stop executing when the stop state is true.
   *  <p>
   *  This is a workaround for the broken behavior of the Thread.stop() method.
   *  @param removeListeners remove the listeners before stopping so they are not called when the worker finishes?
   *                         This is usually a good idea.
   */
  public void stop(boolean removeListeners)
  {
    if (removeListeners) {
      synchronized (knockOffListeners) {
        knockOffListeners.clear();
      }
    }
    synchronized (this) {
      stopped = true;
    }
  }

  /**
   *  Was the worker stopped?
   *  Implementing classes may check this method regularly
   *  and stop executing when it returns {@code true}.
   *  <p>
   *  This is a workaround for the broken behavior of the Thread.stop() method.
   *  @return was on of the stop methods called?
   *  @see #stop()
   *  @see #stop(boolean) 
   *
   */
  public synchronized boolean isStopped()
  {
    return stopped;
  }

  /**
   *  Invoke in the AWT thread.
   *  @param run runnable which has to be invoked in the AWT event thread
   */
  public static void invokeInEventDispatchThread(@NotNull Runnable run)
  {
    if (SwingUtilities.isEventDispatchThread()) {
      run.run();
    }
    else {
      SwingUtilities.invokeLater(run);
    }
  }

}
