// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.*;
import de.caff.generics.function.Function1;
import de.caff.util.debug.Debug;
import de.caff.util.measure.IllegalPhysicalLengthFormatException;
import de.caff.util.measure.PhysicalLength;

import javax.swing.*;
import java.awt.*;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;
import java.awt.image.VolatileImage;
import java.io.*;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.AccessControlException;
import java.text.CharacterIterator;
import java.text.StringCharacterIterator;
import java.util.List;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import static de.caff.util.ParameterTypes.Bool;

/**
 *  Utility contains various helpful functionality.
 * <p>
 *  Sorry, basically this is mess which cries for cleanup.
 *  @author rammi@caff.de
 */  
public final class Utility
{
  /** The current version of the library. */
  public static final String RELEASE_DATE = ModuleVersion.getReleaseDate();

  /** Physiolocigal brightness value for red. */
  public static final float PHYS_RED_SCALE   = 0.30f;
  /** Physiolocigal brightness value for green. */
  public static final float PHYS_GREEN_SCALE = 0.53f;
  /** Physiolocigal brightness value for blue. */
  public static final float PHYS_BLUE_SCALE  = 1.0f-PHYS_RED_SCALE-PHYS_GREEN_SCALE;
  /** Image observer which ignores everything. */
  public static final ImageObserver NULL_OBSERVER = (img, infoflags, x, y, width, height) -> (infoflags & (ImageObserver.ALLBITS | ImageObserver.ABORT)) == 0;
  /** Default resource path. */
  public static final String DEFAULT_RESOURCE_PATH = "de.caff.gimmicks.resources.IconConstants|/de/caff/gimmicks/resources/";
  /** Built date of this class. */
  public static final String BUILT_DATE = getBuildDate("???");
  /** Component used for access to a prepared image. */
  private static final Component preparer = new Canvas();
  /** Are we on Windows? */
  private static final boolean weAreOnWindows = (File.separatorChar == '\\'); // (too?) simple
  /** Empty URL array. */
  public static final URL[] EMPTY_URL_ARRAY = {};
  /** The default resource directory.. */
  private static String    resourceDir   = DEFAULT_RESOURCE_PATH;
  /** The event queue exception wrapper if one is used. */
  private static EventQueueExceptionWrapper exceptionWrapper = null;
  /** Extension class loader. */
  @NotNull
  private static ClassLoader extensionClassLoader = Utility.class.getClassLoader();
  /** Map collecting requests for properties. */
  @NotNull
  private static final ConcurrentMap<String, ParameterTypes> requestedParameters = new ConcurrentHashMap<>();

  /**
   * General comparator for strings, which ignores case.
   */
  @NotNull
  public static final Comparator<String> CASE_IGNORING_STRING_COMPARATOR = String::compareToIgnoreCase;

  /** Debugging mode. */
  private static boolean   debugging = false;

  /**
   *  Set the debugging mode.
   *  @param mode new mode
   */
  public static void setDebug(boolean mode) {
    debugging = mode;
  }

  /**
   *  Get the debug mode.
   *  @return debug mode
   */
  public static boolean isDebug() {
    return debugging;
  }

  /**
   * Get the class loader for extension classes.
   *
   * @return extension class loader
   * @see #updateExtensionClassLoader(Function1)
   */
  @NotNull
  public static ClassLoader getExtensionClassLoader()
  {
    synchronized (preparer) {
      return extensionClassLoader;
    }
  }

  /**
   * Update the extension class loader.
   * <p>
   * This special way of setting the {@link #getExtensionClassLoader() extension class loader}
   * is chosen to support clean synchronization.
   * <p>
   * Changing the class load mechanism requires certain rights and might not work in restricted
   * environments like applets.
   *
   * @param updater updater function which is called with the current extension class loader as argument
   *                and has to return the new extension class loader. The new class loader should use the
   *                old extension class loader as its parent to allow for nested extensions. The updater
   *                function must not return {@code null}
   */
  public static void updateExtensionClassLoader(@NotNull Function1<ClassLoader, ClassLoader> updater)
  {
    synchronized (preparer) {
      ClassLoader newLoader = updater.apply(extensionClassLoader);
      if (newLoader == null) {
        throw new IllegalArgumentException("updater returns null although it is not allowed to do so!");
      }
      extensionClassLoader = newLoader;
    }
  }

  /**
   * Add URLs tro the system class loader.
   * <p>
   * This is an evil hack, but allows to add libraries at runtime.
   * <p>
   * This method takes care of loading additional JARs which are linked via the
   * Class_path Manifest property, but not recursively.
   * <p>
   * It will not work in secure restricted environments like applets.
   *
   * @param urls JAR or class directory URLs to add to the system class loader
   * @return {@code true}: if this worked for all URLs<br>
   *         {@code false}: if it failed, probably due to security restrictions
   */
  public static boolean addToSystemClassLoader(@NotNull URL ... urls)
  {
    if (urls.length == 0) {
      return true;
    }
    try {
      final URLClassLoader sysLoader = (URLClassLoader) ClassLoader.getSystemClassLoader();
      final Class<?> sysClass = URLClassLoader.class;
      Method method = sysClass.getDeclaredMethod("addURL", URL.class);
      try {
        method.setAccessible(true);
        for (URL url : expandLinkedJars(urls)) {
          method.invoke(sysLoader, url);
        }
        return true;
      } finally {
        method.setAccessible(false);
      }
    } catch (Throwable x) {
      Debug.error(x);
    }
    return false;
  }

  /**
   * Update the extension class load mechanism by adding JAR files and file locations
   * to load classes.
   * <p>
   * This method takes care of loading additional JARs which are linked via the
   * Class_path Manifest property, but not recursively.
   * <p>
   * Changing the class load mechanism requires certain rights and might not work in restricted
   * environments like applets.
   *
   * @param urls URLs of local JAR files and class directories, net URLs are not accepted by the underlying
   *             Java class loader extension mechanism
   */
  public static void setExtensionClassLoadUrls(@NotNull final URL ... urls)
  {
    if (urls.length == 0) {
      return;
    }
    final Set<URL> urlSet = expandLinkedJars(urls);
    Debug.message("Added URLs: %0", urlSet);
    updateExtensionClassLoader(arg -> URLClassLoader.newInstance(urlSet.toArray(EMPTY_URL_ARRAY), arg));
  }

  /**
   * Expand all linked jars in URL jars.
   * @param urls URLs of local JAR files and class directories, net URLs are not accepted by the underlying
   *             Java class loader extension mechanism
   * @return set including linked jars
   */
  @NotNull
  private static Set<URL> expandLinkedJars(@NotNull URL[] urls)
  {
    final Set<URL> urlSet = new HashSet<URL>(Types.asList(urls));
    for (URL url : new ArrayList<URL>(urlSet)) {
      if (url.getFile().toLowerCase().endsWith(".jar")) {
        // check jar file for linked jars
        Debug.message("Checking %0 with protocol %1", url, url.getProtocol());
        if ("file".equals(url.getProtocol())) {
          try {
            final File file = new File(url.getPath());
            final File dir = file.getParentFile();
            JarFile jar = new JarFile(file);
            Manifest manifest = jar.getManifest();
            String linkedPaths = manifest.getMainAttributes().getValue(Attributes.Name.CLASS_PATH);
            Debug.message("linkedPaths=%0", linkedPaths);
            if (linkedPaths != null && !linkedPaths.isEmpty()) {
              for (String linkedPath: linkedPaths.split(" ")) {
                String[] parts = linkedPath.split("/");
                File linked = dir;
                for (String part : parts) {
                  linked = new File(linked, part);
                }
                Debug.message("linked=%0", linked);
                if (linked.exists()) {
                  Debug.message("Adding linked JAR %0 to extension classes.", linked);
                  urlSet.add(linked.toURI().toURL());
                }
              }
            }
          } catch (IOException e) {
            Debug.error(e);
          }
        }
      }
    }
    return urlSet;
  }

  /**
   *  Load an image and prepare a representation. Used for static images to be loaded
   *  in an very early stage of program execution.
   *  @param   path      path of the image file (may include loader class prepended with a {@code |} (pipe symbol) separator
   *  @return  the loaded image, or {@code null} if the image was not found or not accessible
   */
  @Nullable
  public static Image loadImage(@NotNull String path) {
    return loadImage(path, preparer);
  }

  /**
   *  Load an image and prepare a representation. Used for static images to be loaded
   *  in an very early stage of program execution.
   *  @param   path      path of the image file (may include loader class prepended with a {@code |} (pipe symbol) separator
   *  @param   observer image observer
   *  @return  the loaded image, or {@code null} if the image was not found or not accessible
   */
  @Nullable
  public static Image loadImage(@NotNull String path, @Nullable ImageObserver observer) {
    return loadImage(path, preparer, observer);
  }

  /**
   *  Load an image and prepare a representation. Used for static images to be loaded
   *  in an very early stage of program execution.
   *  @param   path      path of the image file (may include loader class prepended with a {@code |} (pipe symbol) separator
   *  @param   renderer  renderer component for image creation
   *  @return  the loaded image, or {@code null} if the image was not found or not accessible
   */
  @Nullable
  public static Image loadImage(@NotNull String path, @NotNull Component renderer) {
    return loadImage(path, renderer, null);
  }

  /**
   *  Load an image and prepare a representation. Used for static images to be loaded
   *  in an very early stage of program execution.
   *  @param   path      path of the image file (may include loader class prepended with a {@code |} (pipe symbol) separator
   *  @param   renderer  renderer component for image creation
   *  @param   observer image observer
   *  @return  the loaded image, or {@code null} if the image was not found or not accessible
   */
  @Nullable
  public static Image loadImage(@NotNull String path,
                                @NotNull Component renderer,
                                @Nullable ImageObserver observer) {
    if (resourceDir != null  &&  path.indexOf('|') < 0  &&  !path.startsWith("/")) {
      path = resourceDir + path;
    }
    return loadAnImage(path, renderer, observer);
  }

  /**
   *  Create a stack dump from a Throwable.
   *  @param x  the Throwable
   *  @return stack dump
   *  @see #getErrorMessage(Throwable)
   * @deprecated use {@link Debug#getStackDump(Throwable)} instead
   */
  @NotNull
  @Deprecated
  public static String getStackDump(@NotNull Throwable x)
  {
    return Debug.getStackDump(x);
  }

  /**
   * Get a (halfways) useful message from an exception.
   * @param x exception
   * @return error message
   * @see #getStackDump(Throwable)
   * @deprecated use {@link Debug#getErrorMessage(Throwable)} instead
   */
  @NotNull
  @Deprecated
  public static String getErrorMessage(@NotNull Throwable x)
  {
    return Debug.getErrorMessage(x);
  }

  /**
   *  Loads an image from a jar file. Be careful to always use /
   *  for dirs packed in jar!
   *  @param   path   path of file (e.g.. images/icon.gif)
   *  @param   renderer  component used for image rendering
   *  @return  the image
   */
  @Nullable
  private static Image loadAnImage(@NotNull String path,
                                   @NotNull Component renderer,
                                   @Nullable ImageObserver imageObserver) {
    Image img = null;
    try {
      URL url = openResourceViaClass(path);
      if (url == null) {
        // workaround for netscape problem
        return null;
      }

      img = Toolkit.getDefaultToolkit().createImage(url);
    } catch (Exception x) {
      debug(x);
    }
          
    if (img != null) {
      /* --- load it NOW --- */
      renderer.prepareImage(img, imageObserver);
    }
    else {
      Debug.error("Failed to load image %0!", path);
    }

    return img;
  } 

  /**
   *  Load a text file into a string. 
   *  @param   path   name of the text file
   *  @return  the loaded text
   */
  @NotNull
  public static String loadText(@NotNull String path) {
    if (resourceDir != null  &&  !path.startsWith("/")) {
      path = resourceDir + /*File.separator +*/ path;
    }
    return loadAText(path);
  }

  /**
   *  Loads a text file from a jar file. Be careful to always use /
   *  for dirs packed in jar!
   *  This silently assumes that the text is encoded in UTF-8.
   *  @param   path   path of file (e.g.. images/foo.txt)
   *  @return  the text
   */
  @NotNull
  private static String loadAText(@NotNull String path) {
    String txt = "";
    try {
      String line;
      //      System.out.println("Loading "+path);
      
      //      System.out.println("URL = "+url);
      BufferedReader reader = new BufferedReader(new InputStreamReader(Utility.class.getResourceAsStream(path),
                                                                       StandardCharsets.UTF_8));
      StringBuilder sb = new StringBuilder(0x10000);
      while ((line = reader.readLine()) != null) {
        sb.append(line).append('\n');
      }
      txt = sb.toString();
      reader.close();
    } catch (Exception x) {
      debug(x);
    }
          
    return txt;
  } 

  /**
   *  Test whether our System is DOS or Windows-based.
   *  @return   {@code true}   we are on Windows/DOS<br>
   *            {@code false} we are elsewhere
   * @deprecated name is outdated, use {@link #areWeOnWindows()} instead
   */
  @Deprecated
  public static boolean areWeOnDOS() {
    return weAreOnWindows;
  }

  /**
   *  Test whether our System Windows-based.
   *  @return   {@code true}  we are running on Windows<br>
   *            {@code false} we are running elsewhere
   */
  public static boolean areWeOnWindows() {
    return weAreOnWindows;
  }

  /**
   *  Set the resource directory.
   *  @param   dir   the image directory
   */
  public static void setResourceDir(@Nullable String dir) {
    resourceDir = dir;
  }

  /**
   *  Compile a formatted string with maximum 10 args.
   *  <pre>
   *  Special signs:
   *     %#  where hash is a digit from 0 to 9 means insert arg #
   *     &#64;#  where hash is a digit from 0 to 9 means insert localized arg #
   *     %%  means %
   *     &#64;&#64;  means &#64;
   *  </pre>
   *  @param   tag    resource tag for format string
   *  @param   args   arguments for insertion
   *  @param   res    active resource bundle
   *  @return  String with inserted args.
   */
  @NotNull
  public static String compileString(@NotNull String tag,
                                     @NotNull Object[] args,
                                     @NotNull ResourceBundle res) {
    String       format  = res.getString(tag);
    StringBuilder ret     = new StringBuilder(format.length());
    int          i;
    char         c;

    for (i = 0;   i < format.length();   i++) {
      c = format.charAt(i);
      
      if (c == '%'   ||   c == '@') {
        int argNum = -1;

        if (i < format.length()-1) {
          // this implies that there are never more than 10 args
          switch (format.charAt(i+1)) {
          case '%':
            if (c == '%') { // "%%" means "%"
              ret.append('%');
              i++;
            }
            break;

          case '@':
            if (c == '@') { // "@@" means "@"
              ret.append("@");
              i++;
            }
            break;

          case '0':
            argNum = 0;
            break;

          case '1':
            argNum = 1;
            break;

          case '2':
            argNum = 2;
            break;

          case '3':
            argNum = 3;
            break;

          case '4':
            argNum = 4;
            break;

          case '5':
            argNum = 5;
            break;

          case '6':
            argNum = 6;
            break;

          case '7':
            argNum = 7;
            break;

          case '8':
            argNum = 8;
            break;

          case '9':
            argNum = 9;
            break;

          default:
            break;
          }
        }
        if (argNum >= 0   &&   argNum < args.length) {
          if (c == '%') {
            // arg is a non-localized string
            ret.append(args[argNum]);
          }
          else { // c == '@'
            // arg is a tag for localization
            ret.append(res.getString(args[argNum].toString()));
          }
          i++;
        }
      }
      else {
        ret.append(c);
      }
    }

    return ret.toString();
  }

  /**
   *  Compile a formatted string with maximum 10 args.
   *  <pre>
   *  Special signs:
   *     %#  where hash is a digit from 0 to 9 means insert arg #
   *     %%  means %
   *  </pre>
   *  @param   format format string
   *  @param   args   arguments for insertion
   *  @return  String with inserted args.
   */
  public static String compileString(@NotNull String format, Object ... args)
  {
    StringBuilder ret     = new StringBuilder(format.length());
    int          i;
    char         c;

    for (i = 0;   i < format.length();   i++) {
      c = format.charAt(i);

      if (c == '%') {
        int argNum = -1;

        if (i < format.length()-1) {
          // this implies that there are never more than 10 args
          switch (format.charAt(i+1)) {
          case '%':
            if (c == '%') { // "%%" means "%"
              ret.append('%');
              i++;
            }
            break;

          case '0':
            argNum = 0;
            break;

          case '1':
            argNum = 1;
            break;

          case '2':
            argNum = 2;
            break;

          case '3':
            argNum = 3;
            break;

          case '4':
            argNum = 4;
            break;

          case '5':
            argNum = 5;
            break;

          case '6':
            argNum = 6;
            break;

          case '7':
            argNum = 7;
            break;

          case '8':
            argNum = 8;
            break;

          case '9':
            argNum = 9;
            break;

          default:
            break;
          }
        }
        if (argNum >= 0   &&   argNum < args.length) {
          // arg is a non-localized string
          ret.append(args[argNum]);
          i++;
        }
      }
      else {
        ret.append(c);
      }
    }

    return new String(ret);
  }

  /**
   *  Method to get the frame parent of any component.
   *  @param   comp   the component to search the frame for
   *  @return  the frame parent of the component
   */
  @NotNull
  public static Frame getFrame(@Nullable Component comp) {
    for (   ;  comp != null;   comp = comp.getParent()) {
      if (comp instanceof Frame) {
        return (Frame)comp;
      }
    }
    /* --- Not found. Ugly workaround: --- */
    return new Frame();
  }

  /**
   *  Compare two byte arrays.
   *  Compare {@code len} bytes from array 1 starting with offset 1
   *  with {@code len} bytes from array 2 starting with offset 2.
   *  Will return always {@code true} for {@code len &le;= 0}
   *  @param arr1    array 1
   *  @param off1    offset 1
   *  @param arr2    array 2
   *  @param off2    offset 2
   *  @param len     length to compare
   *  @return {@code true} if both chunks are equal<br>
   *          {@code false} otherwise
   */
  public static boolean equalBytes(@NotNull byte[] arr1, int off1,
                                   @NotNull byte[] arr2, int off2,
                                   int len) {
    while (len-- > 0) {
      //      System.out.println(arr1[off1] + " == "+arr2[off2]);
      if (arr1[off1++] != arr2[off2++]) {
        //	System.out.println();
        return false;                // not equal
      }
    }
    return true;                // equal
  }

  /**
   *  Look for a boolean applet parameter or application property.
   *  The expected value for {@code true} is {@code &quot;true&quot;}.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static boolean getBooleanParameter(@NotNull String key, boolean def) {
    return getBooleanParameter(key, def, "true");
  }

  /**
   * Look for a boolean applet parameter or application property.
   * This allows for more than one or other values for {@code true}.
   * @param key        parameter key
   * @param def        default value
   * @param trueValues values to be used for {@code true} return, compared case-insensitively.
   * @return the parameter value (if set) or the default
   */
  public static boolean getBooleanParameter(@NotNull String key, boolean def,
                                            @NotNull String ... trueValues)
  {
    final String value = getParameter(key);
    requestedParameters.put(key, Bool);
    if (value != null) {
      for (String tv : trueValues) {
        if (value.equalsIgnoreCase(tv)) {
          return true;
        }
      }
      return false;
    }
    else {
      return def;
    }
  }

  /**
   * Get an enum value from an applet parameter or application property.
   * @param key key for the value, expected to evaluate to the name of an enum of the required type
   * @param def default value, also defining the type of enum return by this method
   * @return enum value from property/parameter, or {@code def} if unset or value is invalid
   * @param <T> enum type
   */
  @NotNull
  @SuppressWarnings("unchecked") // Enum class T is expected to have values of type T.
  public static <T extends Enum<?>> T getEnumParameter(@NotNull String key,
                                                       @NotNull T def)
  {
    final String value = getParameter(key);
    requestedParameters.put(key, ParameterTypes.Enum);
    if (value != null) {
      final Enum<?>[] enumConstants = def.getClass().getEnumConstants();
      if (enumConstants != null) {
        for (Enum<?> enumValue : enumConstants) {
          if (value.equalsIgnoreCase(enumValue.name())) {
            return (T)enumValue;
          }
        }
      }
    }
    return def;
  }

  @Nullable
  private static String getParameter(@NotNull String key)
  {
    try {
      final String value = System.getProperty(key);
      if (Debug.getTraceMode()) {
        Debug.trace("getParameter(\"%0\")=%1", key, value);
      }
      return value;
    } catch (Exception x) {  // possible security exception
      // do nothing
      debug(x);
    }
    return null;
  }

  /**
   *  Look for a String applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static String getStringParameter(@NotNull String key, @Nullable String def) {
    final String value = getParameter(key);
    requestedParameters.put(key, ParameterTypes.String);
    return value != null
            ? value
            : def;
  }

  /**
   * Format a string for output.
   * @param str string, possibly {@code null}
   * @return string surrounded by double quotes, or special null representation
   */
  @NotNull
  private static String formatString(@Nullable String str)
  {
    return str != null
            ? String.format("\"%s\"", str)
            : "<null>";
  }

  /**
   *  Look for a color applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  @Nullable
  public static Color getColorParameter(@NotNull String key, @Nullable Color def)
  {
    requestedParameters.put(key, ParameterTypes.Color);
    final String value = getParameter(key);
    if (value != null) {
      // try to decode color
      try {
        return ColorDecoder.decode(value);
      } catch (Exception x) {
        // nothing
        debug(x);
      }
    }

    return def;
  }

  /**
   *  Look for an integer applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set and valid) or the default
   */
  public static int getIntParameter(@NotNull String key, int def) {
    return getIntParameter(key, def, 10);
  }

  /**
   *  Look for an integer applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @param  base number base
   *  @return the parameter value (if set and valid) or the default
   */
  public static int getIntParameter(@NotNull String key, int def, int base) {
    requestedParameters.put(key, ParameterTypes.Int);
    final String value = getParameter(key);
    if (value != null) {
      try {
        return Integer.parseInt(value, base);
      } catch (NumberFormatException x) {
        // nothing
        debug(x);
      }
    }
    return def;
  }

  /**
   * Look for a long applet parameter of application property.
   * @param key parameter key
   * @param def default value
   * @return the parameter value (if set and valid) or the default value
   */
  public static long getLongParameter(@NotNull String key, long def)
  {
    return getLongParameter(key, def, 10);
  }

  /**
   * Look for a long applet parameter of application property.
   * @param key parameter key
   * @param def default value
   * @param base number base
   * @return the parameter value (if set and valid) or the default value
   */
  public static long getLongParameter(@NotNull String key, long def, int base)
  {
    final String value = getParameter(key);
    requestedParameters.put(key, ParameterTypes.Long);
    if (value != null) {
      try {
        return Long.parseLong(value, base);
      } catch (NumberFormatException x) {
        debug(x);
      }
    }
    return def;
  }

  /**
   *  Look for a double applet parameter or application property.
   * <p>
   *  Uses percentage if ending with %.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static double getDoubleParameter(@NotNull String key, double def) {
    String value = getParameter(key);
    requestedParameters.put(key, ParameterTypes.Double);
    if (value != null) {
      value = value.trim();
      try {
        double factor = 1;
        if (value.endsWith("%")) {
          value = value.substring(0, value.length() - 1);
          factor = 0.01;
        }
        return Double.parseDouble(value) * factor;
      } catch (NumberFormatException x) {
        // nothing
        debug(x);
      }
    }

    return def;
  }

  /**
   *  Look for a float applet parameter or application property.
   * <p>
   *  Uses percentage if ending with %.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static float getFloatParameter(@NotNull String key, float def) {
    String value = getParameter(key);
    requestedParameters.put(key, ParameterTypes.Float);
    if (value != null) {
      value = value.trim();
      try {
        float factor = 1;
        if (value.endsWith("%")) {
          value = value.substring(0, value.length() - 1);
          factor = 0.01f;
        }
        return Float.parseFloat(value) * factor;
      } catch (NumberFormatException x) {
        // nothing
        debug(x);
      }
    }

    return def;
  }


  /**
   *  Look for a length applet parameter or application property.
   * <p>
   *  Uses percentage if ending with %.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static PhysicalLength getLengthParameter(@NotNull String key, PhysicalLength def) {
    String value = getParameter(key);
    requestedParameters.put(key, ParameterTypes.Length);
    if (value != null) {
      value = value.trim();
      try {
        return PhysicalLength.fromString(value, Locale.US);
      } catch (IllegalPhysicalLengthFormatException x) {
        // nothing
        debug(x);
      }
    }

    return def;
  }

  /**
   *  Print message if debug mode is on.
   *  @param  x  object which's toString is called
   */
  public static void debug(@Nullable Object x) {
    if (debugging) {
      System.out.println(x == null  ?  "<null>"  :  x.toString());
    }
  }

  /**
   *  Print the stack trace if debug mode is on.
   *  @param  x  exception
   */
  public static void debug(@NotNull Throwable x) {
    if (debugging) {
      x.printStackTrace();
    }
  }

  /**
   *  Print a given property to the console. Catch possible Security exceptions.
   *  Does nothing if not in debug mode.
   *  @param prop poperty name
   */
  public static void printProperty(@NotNull String prop) {
    try {
      debug(prop+"="+System.getProperty(prop));
    } catch (Throwable x) {
      // empty
    }
  }

  /**
   *  In debug mode: print properties to console.
   */
  public static void printProperties() {
    if (Utility.isDebug()) {
      String[] useful = {
              "java.version",
              "java.vendor",
              "os.name",
              "os.arch",
              "os.version"
      };

      for (String prop : useful) {
        printProperty(prop);
      }
    }
  }

  /**
   *  An equal function which accepts globbing.
   *  This method accepts the glob chars <tt>'*'</tt>
   *  (for any number of chars) and <tt>'?'</tt> (any single char).
   *  @param  mask   glob mask (containing special chars)
   *  @param  str    string to be checked against mask
   *  @return whether the string matches the mask
   */
  public static boolean globEquals(@NotNull String mask, @NotNull String str)
  {
    int maskLen = mask.length();
    int strLen  = str.length();
    if (maskLen > 0) {
      char first = mask.charAt(0);
      switch (first) {
      case '*':
        return
                globEquals(mask.substring(1), str) ||
                (strLen > 0  &&  globEquals(mask, str.substring(1)));

      case '?':
        return strLen > 0  &&
               globEquals(mask.substring(1), str.substring(1));

      default:
        return strLen > 0   &&
               first == str.charAt(0)   &&
               globEquals(mask.substring(1), str.substring(1));
      }
    }
    else {
      return maskLen == strLen;
    }
  }

  /**
   *  Add an exception listener which is called when an exception occurs during the
   *  dispatch of an AWT event.
   *  Note that depending on the environment this method may throw a security exception.
   *  @param listener listener to add
   */
  public static void addEventQueueExceptionListener(@NotNull EventQueueExceptionListener listener)
  {
    if (exceptionWrapper == null) {
      exceptionWrapper = new EventQueueExceptionWrapper();
    }
    exceptionWrapper.addEventQueueExceptionListener(listener);
  }

  /**
   *  Remove an exception listener which was called when an exception occurs during the
   *  dispatch of an AWT event.
   *  @param listener listener to remove
   */
  public static void removeEventQueueExceptionListener(@NotNull EventQueueExceptionListener listener)
  {
    if (exceptionWrapper != null) {
      exceptionWrapper.removeEventQueueExceptionListener(listener);
    }
  }

  /**
   *  Try to create a custom cursor from an icon. The hot spot is set to the icon center.
   *  @param icon      icon from which to create the cursor
   *  @param bgColor   background color for cursor
   *  @param name      name for accessibility
   *  @param fallback  fallback cursor taken if the image size is not supported
   *  @return the new cursor if it was possible to create one with the required size,
   *          or the fallback cursor
   */
  @NotNull
  public static Cursor createCustomCursor(@NotNull Icon icon,
                                          @NotNull Color bgColor,
                                          @NotNull String name,
                                          @NotNull Cursor fallback)
  {
    return createCustomCursor(icon, bgColor, null, name, fallback);
  }

  /**
   *  Try to create a custom cursor from an icon.
   *  @param icon      icon from which to create the cursor
   *  @param bgColor   background color for cursor
   *  @param hotspot   hot spot of cursor (if {@code null} the center of the icon is taken)
   *  @param name      name for accessibility
   *  @param fallback  fallback cursor taken if the image size is not supported
   *  @return the new cursor if it was possible to create one with the required size,
   *          or the fallback cursor
   */
  @NotNull
  public static Cursor createCustomCursor(@NotNull Icon icon,
                                          @NotNull Color bgColor,
                                          @Nullable Point hotspot,
                                          @NotNull String name,
                                          @NotNull Cursor fallback)
  {
    Toolkit toolkit = Toolkit.getDefaultToolkit();
    Dimension size = toolkit.getBestCursorSize(icon.getIconWidth(), icon.getIconHeight());
    if (size.width >= icon.getIconWidth()  &&   size.height >= icon.getIconHeight()) {
      int x = (size.width -icon.getIconWidth())/2;
      int y = (size.height-icon.getIconHeight())/2;
      BufferedImage image = createPlatformImage(size.width, size.height, Transparency.TRANSLUCENT);
      preparer.setBackground(bgColor);
      icon.paintIcon(preparer, image.getGraphics(), x, y);
      if (hotspot == null) {
        hotspot = new Point(icon.getIconWidth()/2, icon.getIconHeight()/2);
      }
      Cursor cursor = toolkit.createCustomCursor(image, new Point(hotspot.x+x, hotspot.y+y), name);
      if (cursor != null) {
        return cursor;
      }
    }
    return fallback;
  }

  /**
   * Get a font metrics.
   * @param font font for which a metrics is requested
   * @return font metrics
   */
  public static FontMetrics getFontMetrics(@NotNull Font font)
  {
    return preparer.getFontMetrics(font);
  }

  /**
   * Create an opaque buffered image which is optimized for the current platform.
   * <p>
   * Please note that it's always a good idea to call the image's {@code flush()} method
   * before it is discarded.
   * @param width        image width
   * @param height       image height
   * @return image of given size, optimized for the platform
   */
  @NotNull
  public static BufferedImage createPlatformImage(int width, int height)
  {
    return createPlatformImage(width, height, Transparency.OPAQUE);
  }

  /**
   * Create a buffered image which is optimized for the current platform.
   * <p>
   * Please note that the transparency type usually has an impact on performance.
   * Best is opaque, followed by bitmap, and translucent images may be a drawn a lot slower.
   * <p>
   * Regardless of that it's always a good idea to call the image's {@code flush()} method
   * before it is discarded.
   * @param width        image width
   * @param height       image height
   * @param transparency transparency type, either {@link java.awt.Transparency#OPAQUE},
   *                     {@link java.awt.Transparency#BITMASK}, or {@link java.awt.Transparency#TRANSLUCENT}
   * @return image of given size, optimized for the platform
   */
  @NotNull
  public static BufferedImage createPlatformImage(int width, int height, int transparency)
  {
    final GraphicsEnvironment localGraphicsEnvironment = GraphicsEnvironment.getLocalGraphicsEnvironment();
    if (localGraphicsEnvironment.isHeadlessInstance()) {
      return new BufferedImage(width, height,
                               transparency == Transparency.OPAQUE
                                       ? BufferedImage.TYPE_INT_RGB
                                       : BufferedImage.TYPE_INT_ARGB);
    }
    else {
      GraphicsDevice gd = localGraphicsEnvironment.getDefaultScreenDevice();
      GraphicsConfiguration gc = gd.getDefaultConfiguration();

      return gc.createCompatibleImage(width, height, transparency);
    }
  }

  /**
   * Try to create an image on which graphics operations have high performance.
   * This will try to create a PerfBufferedImage as is provided by the PerfGraphics library of
   * Werner Randelshofer.
   * <p>
   * Note that it's always a good idea to call the image's {@code flush()} method
   * before it is discarded.
   * @param width  image width
   * @param height image height
   * @return fast image or {@code null} if the PerfGraphics lib is not available
   * @see <a href="http://www.randelshofer.ch/oop/graphics/index.html">PerfGraphics page</a>
   */
  @Nullable
  public static BufferedImage createFastImage(int width, int height)
  {
    try {
      Class<?> pbiClass = Class.forName("ch.randelshofer.awt.graphics.PerfBufferedImage");
      Constructor<?> constructor = pbiClass.getConstructor(Integer.TYPE, Integer.TYPE);
      return (BufferedImage)constructor.newInstance(width, height);
    } catch (Throwable t) {
      // if something is thrown, this will happen on each invocation
      Debug.trace(t);
    }
    return null;
  }

  /**
   * This will create a {@code VolatileImage} and try to increase its acceleration priority.
   * <p>
   * Note that it's always a good idea to call the image's {@code flush()} method
   * before it is discarded.
   * @param comp   component from which to create image
   * @param width  image width
   * @param height image height
   * @return volatile image, maybe accelerated
   */
  @NotNull
  public static VolatileImage createFastVolatileImage(@NotNull Component comp, int width, int height)
  {
    VolatileImage image = comp.createVolatileImage(width, height);
    try {
      Method m = image.getClass().getMethod("setAccelerationPriority", Float.TYPE);
      m.invoke(image, 1f);
    } catch (NoSuchMethodException e) {
      // No Sun VM, or too old, or not supported on
    } catch (Throwable t) {
      // invocation failure
      Debug.trace(t);
    }
    return image;
  }

  /**
   * Get the brightness of a color.
   * Humans do perceive the different color components differently in respect of their brightness,
   * which this method tries to take care off.
   * @param color color which brightness is needed
   * @return brightness value between {@code 0} (black) and {@code 1.0} (white)
   */
  public static float getPhysiologicalBrightness(@NotNull Color color)
  {
    return getPhysiologicalBrightness(color.getRed(),
                                      color.getGreen(),
                                      color.getBlue());
  }

  /**
   * Get the physiological brightness of a RGB color.
   * The color values are expected to be in a range from {@code 0} to {@code 255},
   * otherwise the return of this method is useless.
   * @param red    red color value
   * @param green  green color value
   * @param blue   blue color value
   * @return brigtness value between {@code 0} (black) and {@code 1} (white)
   */
  public static float getPhysiologicalBrightness(int red, int green, int blue)
  {
    return (PHYS_RED_SCALE   * red   / 255.0f +
            PHYS_GREEN_SCALE * green / 255.0f +
            PHYS_BLUE_SCALE  * blue  / 255.0f);
  }

  /**
   * Get a gray with the same brightness as a given color.
   * Humans do perceive the different color components differently in respect of their brightness,
   * which this method tries to take care off.
   * @param color color to convert to gray
   * @return gray color with the same brightness as the given color
   */
  @NotNull
  public static Color getPhysiologicalGray(@NotNull Color color)
  {
    float brightness = getPhysiologicalBrightness(color);
    return new Color(brightness,
                     brightness,
                     brightness);
  }

  /**
   *  Get the distance between two colors by comparing their brightness.
   *  @param color1 first color
   *  @param color2 second color
   *  @return the distance of the colors as the absolute difference of their colors,
   *          a number between {@code 0} (same brightness) and {@code 1}
   *          (completely different brightness)
   */
  public static float getColorBrightnessDistance(@NotNull Color color1,
                                                 @NotNull Color color2)
  {
    return Math.abs(getPhysiologicalBrightness(color1) - getPhysiologicalBrightness(color2));
  }

  /** Properties to try if we are in a restricted environment. */
  private static final String[] FALLBACK_PROPERTIES = {
          "file.encoding",
          "file.encoding.pkg",
          "file.separator",
          "java.awt.graphicsenv",
          "java.awt.printerjob",
          "java.class.path",
          "java.class.version",
          "java.endorsed.dirs",
          "java.ext.dirs",
          "java.home",
          "java.io.tmpdir",
          "java.library.path",
          "java.runtime.name",
          "java.runtime.version",
          "java.specification.name",
          "java.specification.vendor",
          "java.specification.version",
          "java.vendor",
          "java.vendor.url",
          "java.vendor.url.bug",
          "java.version",
          "java.vm.info",
          "java.vm.name",
          "java.vm.specification.name",
          "java.vm.specification.vendor",
          "java.vm.specification.version",
          "java.vm.vendor",
          "java.vm.version",
          "os.arch",
          "os.name",
          "os.version",
          "path.separator",
          "sun.arch.data.model",
          "sun.boot.class.path",
          "sun.boot.library.path",
          "sun.cpu.endian",
          "sun.cpu.isalist",
          "sun.io.unicode.encoding",
          "sun.java.launcher",
          "sun.jnu.encoding",
          "sun.management.compiler",
          "sun.os.patch.level",
          "user.country",
          "user.dir",
          "user.home",
          "user.language",
          "user.name",
          "user.timezone"
  };

  /**
   * Get a list of system properties and their values, sorted alphabetically by property name.
   * <p>
   * Note that in restricted environments it's possible that not all properties are accessible.
   * In that case only some properties are returned.
   * @return list of pairs with property name as first and property value as second entry
   */
  @NotNull
  public static List<Pair<String>> getSortedSystemProperties()
  {
    List<Pair<String>> result = new LinkedList<>();
    try {
      Properties properties = System.getProperties();
      for (Enumeration<?> e = properties.propertyNames();  e.hasMoreElements(); ) {
        String name = (String)e.nextElement();
        result.add(new Pair<>(name, properties.getProperty(name)));
      }
    } catch (AccessControlException x) {
      for (String name: FALLBACK_PROPERTIES) {
        try {
          final String value = System.getProperty(name);
          if (value != null) {
            result.add(new Pair<>(name, value));
          }
        } catch (AccessControlException e) {
          // restricted, too
        }
      }
    }
    result.sort(Pair.getFirstEntryPairComparator(String.CASE_INSENSITIVE_ORDER));
    return result;
  }

  /**
   * Get a string where all non-ASCII and non-printable characters are escaped by their Unicode escapes.
   * @param str string
   * @return string where all non-ASCII and non-printable characters are escaped
   */
  @NotNull
  public static String toASCII(@NotNull String str)
  {
    StringBuilder sb = new StringBuilder();
    CharacterIterator iterator = new StringCharacterIterator(str);
    for(char c = iterator.first(); c != CharacterIterator.DONE; c = iterator.next()) {
      if (c >= 32 && c < 127) {
        sb.append(c);
      }
      else {
        sb.append(String.format("\\U+%04x", (int)c));
      }
    }
    return sb.toString();
  }

  /**
   * Is a string {@code null} or empty?
   * @param str string to check
   * @return {@code true} if the string is {@code null} or empty<br>
   *         {@code false} otherwise
   */
  public static boolean isNullOrEmpty(@Nullable String str)
  {
    return str == null  || str.isEmpty();
  }

  /**
   * Copy an image to the system clipboard.
   * @param image   the image to copy
   * @throws IllegalStateException if the clipboard is currently unavailable
   */
  public static void copyImageToClipboard(@NotNull final Image image)
  {
    final Transferable transferable = new Transferable()
    {
      @Override
      public DataFlavor[] getTransferDataFlavors()
      {
        return new DataFlavor[] { DataFlavor.imageFlavor };
      }

      @Override
      public boolean isDataFlavorSupported(DataFlavor flavor)
      {
        return DataFlavor.imageFlavor.equals(flavor);
      }

      @NotNull
      @Override
      public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException
      {
        if (!DataFlavor.imageFlavor.equals(flavor)) {
            throw new UnsupportedFlavorException(flavor);
        }
        return image;
      }
    };
    Toolkit.getDefaultToolkit().getSystemClipboard().setContents(transferable, null);
  }

  /**
   * Copy text to the system clipboard.
   * @param text   the text to copy
   * @throws IllegalStateException if the clipboard is currently unavailable
   */
  public static void copyTextToClipboard(@NotNull String text)
  {
    final Transferable transferable = new Transferable()
    {
      @Override
      public DataFlavor[] getTransferDataFlavors()
      {
        return new DataFlavor[] { DataFlavor.stringFlavor };
      }

      @Override
      public boolean isDataFlavorSupported(DataFlavor flavor)
      {
        return DataFlavor.stringFlavor.equals(flavor);
      }

      @NotNull
      @Override
      public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException
      {
        if (!DataFlavor.stringFlavor.equals(flavor)) {
          throw new UnsupportedFlavorException(flavor);
        }
        return text;
      }
    };
    Toolkit.getDefaultToolkit().getSystemClipboard().setContents(transferable, null);

  }

  /**
   * A more effective implementation of {@code Arrays.asList()},
   * useful if only an unmodifiable list is needed.
   * <p>
   * Changes of the array will be reflected in the list return from this method,
   * so the safest use is as a temporary object.
   *
   * @param array the array to wrap into a list view
   * @param <T> element type of array and list
   * @return unmodifiable list accessing the array
   * @deprecated Use {@link Types#asList(Object[])} instead.
   */
  @Deprecated
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T> List<T> asList(@NotNull T ... array)
  {
    return Types.asList(array);
  }

  /**
   * A more effective implementation of {@code Arrays.asList()},
   * useful if only an unmodifiable list is needed.
   * <p>
   * Changes of the array will be reflected in the list returned from this method,
   * so the safest use is as a temporary object.
   *
   * @param array the array to wrap into a list view
   * @param length the number of elements to be used from the array
   * @param <T> element type of array and list
   * @return unmodifiable list accessing the array
   * @deprecated Use {@link Types#asList(Object[], int)} instead.
   */
  @Deprecated
  @NotNull
  public static <T> List<T> asList(@NotNull T[] array, int length)
  {
    return Types.asList(array, length);
  }

  /**
   * A more effective implementation of {@code Arrays.asList()},
   * useful if only an unmodifiable list is needed.
   * <p>
   * Changes of the array will be reflected in the list returned from this method.
   *
   * @param array the array to wrap into a list view
   * @param start index of first used element of the array
   * @param length the number of elements to be used from the array
   * @param <T> element type of array and list
   * @return unmodifiable list accessing the array
   * @deprecated Use {@link Types#asList(Object[], int, int)} instead.
   */
  @Deprecated
  @NotNull
  public static <T> List<T> asList(@NotNull T[] array, int start, int length)
  {
    return Types.asList(array, start, length);
  }

  /**
   * Get a screen shot of a frame or dialog.
   * @param component component in the frame or dialog
   * @return screen shot image or {@code null} if there have been problems with the graphical setup
   */
  @Nullable
  public static BufferedImage getRootComponentImage(@NotNull Component component)
  {
    Component parent = component.getParent();
    if (parent != null) {
      return getRootComponentImage(parent);
    }
    GraphicsDevice device = component.getGraphicsConfiguration().getDevice();
    try {
      Robot robot = new Robot(device);
      return robot.createScreenCapture(component.getBounds());
    } catch (AWTException e) {
      Debug.error(e);
    }
    return null;
  }

  /**
   * Get the build date of a given class.
   * This assumes that the {@code Built-Date} property is set in the Manifest file
   * of the jar which provides the class.
   *
   * @param clazz class for accessing the associated manifest
   * @return built date from manifest file, or {@code null} if no manifest is found
   */
  @Nullable
  public static String getBuildDate(@NotNull Class<?> clazz)
  {
    Package pkg = clazz.getPackage();
    return pkg != null
            ? pkg.getImplementationVersion()
            : null;
  }

  /**
   * Get the build date of a given class.
   * This assumes that the {@code Built-Date} property is set in the Manifest file
   * of the jar which provides the class.
   *
   * @param clazz class for accessing the associated manifest
   * @param defaultValue default value returned if no manifest is found
   * @return built date from manifest file, or {@code defaultValue} if no manifest is found
   */
  @NotNull
  public static String getBuildDate(@NotNull Class<?> clazz,
                                    @NotNull String defaultValue)
  {
    final String result = getBuildDate(clazz);
    return result != null
            ? result
            : defaultValue;
  }

  /**
   * Use {@link #getBuildDate(Class)} using this class to get a built date.
   *
   * @return built date from Manifest file, or {@code null} if no Manifest is found
   */
  @Nullable
  public static String getBuildDate()
  {
    return getBuildDate(Utility.class);
  }

  /**
   * Use {@link #getBuildDate(Class)} using this class to get a built date.

   * @param defaultValue default value returned if no manifest is found
   * @return built date from manifest file, or {@code defaultValue} if no manifest is found
   */
  @NotNull
  public static String getBuildDate(@NotNull String defaultValue)
  {
    return getBuildDate(Utility.class, defaultValue);
  }


  /**
   * Create a stream which possibly points into a newly created ZIP file.
   * This does not check for prior existence of given file, so any existing file will be overwritten.
   * @param outPath  Output path.
   *                 If it ends with a {@code .zip} extension the output stream goes into a zip file
   *                 with the given name.
   * @param substExt if a zip file is created, the first entry will get this extension instead of {@code .zip}
   * @return either a stream into a zip or a simple file output stream
   * @throws IOException on errors opening the file or creating the zip header
   */
  @NotNull
  public static OutputStream createOptionalZip(@NotNull String outPath,
                                               @NotNull String substExt) throws IOException
  {
    OutputStream out = Files.newOutputStream(Paths.get(outPath));
    if (outPath.toLowerCase().endsWith(".zip")) {
      // do on-the-fly packing
      ZipOutputStream zout = new ZipOutputStream(out);
      if (!substExt.startsWith(".")) {
        substExt = "." + substExt;
      }
      String dxfname = outPath.substring(0, outPath.length()-4)+substExt;
      int slash = dxfname.lastIndexOf(File.separator);
      if (slash >= 0) {
        dxfname = dxfname.substring(slash+1);
      }
      zout.setMethod(ZipOutputStream.DEFLATED);
      zout.setLevel(9);
      zout.putNextEntry(new ZipEntry(dxfname));
      out = zout;
    }
    return out;
  }

  /**
   * Expand a byte array into a hexduimp text structure.
   * @param data byte array
   * @return hex editor formatted text, needs monospace font for correct display
   */
  @NotNull
  public static String toHexDump(@NotNull byte[] data)
  {
    String posMask;
    if (data.length <= 0x100) {
      posMask = "%02X";
    }
    else if (data.length <= 0x10000) {
      posMask = "%04X";
    }
    else if (data.length <= 0x1000000) {
      posMask = "%06X";
    }
    else {
      posMask = "%08X";
    }

    final int lineLength = 16;
    final StringBuilder sb = new StringBuilder();
    final int nrFullLines = data.length / lineLength;

    for (int l = 0;  l < nrFullLines;  ++l) {
      final int pos = lineLength * l;
      sb.append(String.format(posMask, pos)).append(": " );
      for (int p = 0;  p < lineLength;  ++p) {
        sb.append(String.format("%02X ", data[pos + p]));
      }
      sb.append(" ");
      for (int p = 0;  p < lineLength;  ++p) {
        sb.append(toChar(data[pos + p]));
      }
      sb.append('\n');
    }
    final int rest = data.length % lineLength;
    if (rest > 0) {
      final int pos = lineLength * nrFullLines;
      sb.append(String.format(posMask, pos)).append(": ");
      for (int p = 0;  p < lineLength;  ++p) {
        sb.append(p < rest
                          ? String.format("%02X ", data[pos + p])
                          : "   ");
      }
      for (int p = 0;  p < rest;  ++p) {
        sb.append(toChar(data[pos + p]));
      }
      sb.append('\n');
    }
    return sb.toString();
  }

  /**
   * Get a useful char for a byte.
   * @param b byte
   * @return associated char, or a point for unprintable chars
   */
  private static char toChar(byte b)
  {
    final int ch = b & 0xff;
    if (ch < 32 ||
        (ch >= 127 && ch < 160)) {
      return '.';
    }
    return (char)ch;
  }

  /**
   * Convert a hex data stream to a byte array.
   * <p>
   * This extracts hexadecimal digits from the string, converts each pair to
   * a byte, and collects these into a byte array.
   * <p>
   * White space is accepted, but non-hexadecimal digits will result in a NumberFormatException.
   *
   * @param hexData hexadecimal digit text
   * @return byte array
   */
  @NotNull
  public static byte[] fromHexData(@NotNull String hexData)
  {
    final int length = hexData.length();
    final StringBuilder sb = new StringBuilder(length);
    for (int i = 0;  i < length;  ++i) {
      final char ch = hexData.charAt(i);
      if (!Character.isWhitespace(ch)) {
        switch (ch) {
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
        case 'A':
        case 'B':
        case 'C':
        case 'D':
        case 'E':
        case 'F':
        case 'a':
        case 'b':
        case 'c':
        case 'd':
        case 'e':
        case 'f':
          sb.append(ch);
          break;
        default:
          throw new NumberFormatException("Not a hexadecimal digit: '"+ch+"'!");
        }
      }
    }
    if (sb.length() == 0) {
      return Empty.BYTE_ARRAY;
    }
    if (sb.length() % 2 == 1) {
      throw new NumberFormatException("Uneven number of hexadecimal digits!");
    }
    final String hex = sb.toString();
    final int arrayLength = sb.length() / 2;
    final byte[] array = new byte[arrayLength];
    for (int i = 0;  i < array.length;  ++i) {
      array[i] = (byte)Integer.parseInt(hex.substring(2*i, 2*i + 2), 16);
    }
    return array;
  }

  /**
   * Open a resource via a class.
   * @param classResource class name and resource URL string separated by a {@code |} character
   * @return resource URL or {@code null} if the resource is invalid or couldn't be opened
   */
  @Nullable
  public static URL openResourceViaClass(@NotNull String classResource)
  {
    final int bar = classResource.indexOf('|');
    if (bar > 0) {
      final String className = classResource.substring(0, bar);
      final String resUrl    = classResource.substring(bar + 1);
      try {
        final Class<?> loadClass  = Class.forName(className);
        return loadClass.getResource(resUrl);
      } catch (ClassNotFoundException e) {
        Debug.error(e);
      }
    }
    // if not prepended by loader class or loading fails try to load via this class
    return Utility.class.getResource(classResource);
  }

  /** 64 spaces. */
  private static final String SPACES = "                                                                ";

  /**
   * Get a string consisting of a given number of space characters.
   * @param num number of spaces
   * @return string with the given number of spaces
   */
  @NotNull
  public static String spaces(int num)
  {
    if (num < 0) {
      throw new IllegalArgumentException("num has to be non-negative, but is "+num+"!");
    }
    if (num == 0) {
      return Empty.STRING;
    }
    if (num > SPACES.length()) {
      String result = SPACES;
      final int half = (num + 1) / 2;
      while (result.length() < half) {
        result += result;
      }
      return result + result.substring(0, num - result.length());
    }
    else {
      return SPACES.substring(0, num);
    }
  }

  /**
   * Get the requested parameters.
   * This is helpful to find out which parameters were requested during
   * the current run so far.
   * <p>
   * Parameters are requested using the following static
   * methods of this class:
   * <ul>
   *   <li>{@link #getStringParameter(String, String)}</li>
   *   <li>{@link #getBooleanParameter(String, boolean)}</li>
   *   <li>{@link #getBooleanParameter(String, boolean, String...)}</li>
   *   <li>{@link #getIntParameter(String, int)}</li>
   *   <li>{@link #getIntParameter(String, int, int)}</li>
   *   <li>{@link #getLongParameter(String, long)}</li>
   *   <li>{@link #getLongParameter(String, long, int)}</li>
   *   <li>{@link #getDoubleParameter(String, double)}</li>
   *   <li>{@link #getFloatParameter(String, float)}</li>
   *   <li>{@link #getEnumParameter(String, Enum)}</li>
   *   <li>{@link #getColorParameter(String, Color)}</li>
   *   <li>{@link #getLengthParameter(String, PhysicalLength)}</li>
   * </ul>
   * @return dictionary with parameter names as keys, and the requested parameter type as values,
   *         giving an overview of the parameters requested so far
   */
  @NotNull
  public static Dict<String, ParameterTypes> getRequestedParameters()
  {
    return Dict.viewMap(requestedParameters);
  }

  /**
   * Print out revisions and built dates.
   * This prints out the build date of the jar containing this class
   * (this needs special jar creation where an attribute
   * {@code Implementation-Version} is included in the Manifest)
   * and the {@link #RELEASE_DATE release date constant of this class}
   * and of all classes which complete names are found in the arguments.
   * @param args complete class names like {@code de.caff.util.Utility}
   */
  public static void main(@NotNull String[] args)
  {
    System.out.println("Jar Build Date: " + getBuildDate("No build date!"));
    System.out.println("Release Date:   " + RELEASE_DATE);
    for (String arg : args) {
      try {
        final Class<?> cls = Class.forName(arg);
        final String date = (String)cls.getDeclaredField("RELEASE_DATE").get(null);
        System.out.println("--");
        System.out.println("Class:          " + cls.getName());
        System.out.println("Release Date:   " + date);
      } catch (ClassNotFoundException | IllegalAccessException | NoSuchFieldException e) {
        e.printStackTrace();
      }
    }
  }
}

