// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2022-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Indexable;
import de.caff.generics.Enums;

/**
 * A tri-state which is either on, off, or undefined.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 19, 2022
 */
public enum TriState
{
  /** On state. */
  On(true),
  /** Off state. */
  Off(false),
  /** Undefined state. */
  Undefined(null);

  /**
   * The boolean equivalent of this state.
   */
  @Nullable
  public final Boolean bool;

  /**
   * The values of this enum.
   * Use this instead of {@link #values()} because it does not require internal array copying.
   */
  @NotNull
  public static final Indexable<TriState> VALUES = Indexable.viewArray(values());

  /**
   * Constructor.
   * @param bool boolean equivalent of this state
   */
  TriState(@Nullable Boolean bool)
  {
    this.bool = bool;
  }

  /**
   * Convert a nullable boolean into a tri-state.
   * @param bool incoming boolean, possibly {@code null}
   * @return tri-state equivalent of {@code bool}
   */
  @NotNull
  public static TriState fromBoolean(@Nullable Boolean bool)
  {
    return bool == null
            ? Undefined
            : (bool ? On : Off);
  }

  /**
   * Get the TriState enum associated with the given internal value.
   *
   * @param internalValue internal value
   * @return associated enum value, or {@code null} if there is none
   */
  @Nullable
  public static TriState fromInternal(int internalValue)
  {
    return Enums.getEnumFromOrdinal(VALUES, internalValue);
  }

  /**
   * Get the TriState enum associated with the given internal value.
   *
   * @param internalValue internal value
   * @param defaultValue  default value
   * @return associated enum value, or {@code defaultValue} if there is none
   */
  @NotNull
  public static TriState fromInternal(int internalValue, @NotNull TriState defaultValue)
  {
    return Enums.getEnumFromOrdinal(VALUES, internalValue, defaultValue);
  }
}
