// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.generics.OrderedPair;

import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Simple template handler allowing to expand special strings
 * in texts.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August 12, 2019
 */
public class TextTemplateHandler<T>
{
  /**
   * Context-less expander.
   * For usage create a {@link #withoutContext()} context-less
   * text handler and use its {@link TextTemplateHandler.WithoutContext#register(Pattern, TextTemplateHandler.Expander)},
   * {@link TextTemplateHandler.WithoutContext#register(String, TextTemplateHandler.Expander)} and
   * {@link TextTemplateHandler.WithoutContext#expand(String)} methods.
   */
  @FunctionalInterface
  public interface Expander
  {
    /**
     * Expand the given match.
     *
     * @param match   match result to expand
     * @param handler the text template handler which found the match,
     *                may be used for recursive expansion
     * @return string with completely expanded place holders
     * @throws IOException on expansion errors
     */
    @NotNull
    String expand(@NotNull MatchResult match, @NotNull TextTemplateHandler.WithoutContext handler)
            throws IOException;
  }

  /**
   * Expander which uses a context.
   */
  @FunctionalInterface
  public interface CtxExpander<T>
  {
    /**
     * Expand the given match.
     * @param match   match result to expand
     * @param handler the text template handler which found the match,
     *                may be used for recursive expansion
     * @param context context of the expander
     * @return string with completely expanded place holders
     * @throws IOException on expansion errors
     */
    @NotNull
    String expand(@NotNull MatchResult match,
                  @NotNull TextTemplateHandler<T> handler,
                  @NotNull T context)
            throws IOException;
  }



  @NotNull
  private final List<OrderedPair<Pattern, CtxExpander<T>>> registeredExpanders = new CopyOnWriteArrayList<>();

  /**
   * Register an expander for a regular expression pattern.
   * @param pattern  pattern to look for
   * @param expander expander to be called if pattern is found
   */
  public void register(@NotNull Pattern pattern,
                       @NotNull CtxExpander<T> expander)
  {
    registeredExpanders.add(OrderedPair.create(pattern, expander));
  }

  /**
   * Register an expander for a simple string pattern.
   * @param sequence  string sequence to look for
   * @param expander  expander to be called if sequence is found
   */
  public void register(@NotNull String sequence,
                       @NotNull CtxExpander<T> expander)
  {
    register(Pattern.compile(Pattern.quote(sequence)), expander);
  }

  /**
   * Expand a string with the given context.
   * @param text     text string
   * @param context  context to use for expansion
   * @return expanded string
   * @throws IOException on expansion errors
   */
  @NotNull
  public String expand(@NotNull String text,
                       @NotNull T context)
          throws IOException
  {
    final StringBuilder sb = new StringBuilder();
    for (OrderedPair<Pattern, CtxExpander<T>> pair : registeredExpanders) {
      final Matcher matcher = pair.first.matcher(text);
      int end = 0;
      sb.setLength(0);
      while (matcher.find()) {
        sb.append(text.substring(end, matcher.start()));
        sb.append(pair.second.expand(matcher.toMatchResult(), this, context));
        end = matcher.end();
      }
      if (end > 0) {
        sb.append(text.substring(end));
        text = sb.toString();
      }
    }
    return text;
  }

  /**
   * Helper class.
   * This allows to use {@link Expander} without context.
   */
  public static class WithoutContext
          extends TextTemplateHandler<WithoutContext>
  {
    /** Internally used converter to use standard TextTemplateHandler. */
    private static class ToCtxExpander
            implements CtxExpander<WithoutContext>
    {
      @NotNull
      private final Expander expander;

      ToCtxExpander(@NotNull Expander expander)
      {
        this.expander = expander;
      }

      @NotNull
      @Override
      public String expand(@NotNull MatchResult match, @NotNull TextTemplateHandler<WithoutContext> handler, @NotNull WithoutContext context)
              throws IOException
      {
        return expander.expand(match, context);
      }

      @Override
      public boolean equals(Object o)
      {
        if (this == o) {
          return true;
        }
        if (o == null || getClass() != o.getClass()) {
          return false;
        }
        ToCtxExpander that = (ToCtxExpander)o;
        return expander.equals(that.expander);
      }

      @Override
      public int hashCode()
      {
        return Objects.hash(expander);
      }
    }

    /**
     * Register an expander for a regular expression pattern.
     * @param pattern  pattern to look for
     * @param expander expander to be called if pattern is found
     */
    public void register(@NotNull Pattern pattern,
                         @NotNull Expander expander)
    {
      register(pattern, new ToCtxExpander(expander));
    }

    /**
     * Register an expander for a simple string pattern.
     * @param sequence  string sequence to look for
     * @param expander  expander to be called if sequence is found
     */
    public void register(@NotNull String sequence,
                         @NotNull Expander expander)
    {
      register(sequence, new ToCtxExpander(expander));
    }

    /**
     * Expand a string with the given context.
     * @param text     text string
     * @return expanded string
     * @throws IOException on expansion errors
     */
    @NotNull
    public String expand(@NotNull String text)
            throws IOException
    {
      return expand(text, this);
    }
  }

  /**
   * Factory method for a text template handler which does not use a context.
   * @return context-less handler
   */
  @NotNull
  public static WithoutContext withoutContext()
  {
    return new WithoutContext();
  }
}
