// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.generics.Empty;
import de.caff.generics.Indexable;
import de.caff.generics.Types;
import de.caff.util.debug.Debug;

import java.io.*;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.util.List;

/**
 * A text line ring buffer working as a byte collector.
 * <p>
 * This is especially useful for collecting text output.
 * <p>
 * If the buffer overflows you can only retrieve the latest
 * {@linkplain #getLimit()} lines.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August 10, 2022
 */
public class TextLineRingBuffer
        implements ByteCollector
{
  /** Local eol sequence. */
  private static final String EOL = String.format("%n");

  /** Initial limit: 64 Mi-lines. */
  static final int INITIAL_LIMIT = 0x1000;

  private static final int PIPE_SIZE = 4096;

  /** Ring buffer for text lines. */
  @NotNull
  private final RingBuffer<String> ringBuffer;
  /** Helper for adding 1 byte. */
  private final byte[] tmp = new byte[1];
  /** Char buffer created from byte buffer. */
  @NotNull
  private final CharBuffer pendingCharBuffer = CharBuffer.allocate(4096);
  @NotNull
  private final PipedOutputStream pos;
  @NotNull
  private final BufferedReader textReader;

  private boolean lastWasCR;

  private long totalNumberOfBytes;

  /**
   * Create a ring buffer with the given size.
   * @param limit size limit of this text line buffer
   * @param charset charset used to convert bytes into strings
   */
  public TextLineRingBuffer(int limit,
                            @NotNull Charset charset)
  {
    this(limit, charset, INITIAL_LIMIT);
  }

  /**
   * Create a ring buffer with the given size.
   * This allows to set the initial limit for testing.
   * @param limit size limit of this buffer
   * @param charset charset for decoding text from bytes
   * @param initialLimit initial limit
   */
  TextLineRingBuffer(int limit, @NotNull Charset charset, int initialLimit)
  {
    ringBuffer = new RingBuffer<>(limit, initialLimit);
    final PipedInputStream pis = new PipedInputStream(PIPE_SIZE);
    try {
      pos = new PipedOutputStream(pis);
      textReader = new BufferedReader(new InputStreamReader(pis, charset));
    } catch (IOException e) {
      // only thrown if pis would already by connected
      Debug.error(e);
      throw new RuntimeException("Unexpected!", e);
    }
  }

  /**
   * Get the limit of this ring buffer.
   * @return buffer limit
   */
  public int getLimit()
  {
    return ringBuffer.getLimit();
  }

  /**
   * Get the number of valid lines cached in this buffer.
   * Due to implementation the returned number might
   * be 1 more than the {@link #getLimit() cache limit}.
   * @return number of text lines cached, possibly 1 more than the limit if
   *         the last line added is not ending with an eol mark
   */
  public synchronized int size()
  {
    return pendingCharBuffer.position() > 0
            ? ringBuffer.size() + 1
            : ringBuffer.size();
  }

  /**
   * Is the text line ring buffer empty.
   * @return if there are no cached lines
   */
  public boolean isEmpty()
  {
    return ringBuffer.isEmpty()  &&  pendingCharBuffer.position() == 0;
  }

  /**
   * Append one byte
   * @param value byte value. Accepts integer for convenience, but will cast the given value to byte.
   */
  @Override
  public synchronized void append(int value)
  {
    tmp[0] = (byte)value;
    append(tmp, 0, 1);
  }

  /**
   * Append the given byte array part to this buffer.
   * @param bytes  basic byte array
   * @param offset start offset of the bytes to append
   * @param length number of bytes to append
   * @throws IndexOutOfBoundsException if the parameters define a part which is at least partially outside the {@code bytes}
   */
  @Override
  public synchronized void append(@NotNull byte[] bytes, int offset, int length)
  {
    if ((offset < 0) || (offset > bytes.length) || (length < 0) ||
        ((offset + length) - bytes.length > 0)) {
      throw new IndexOutOfBoundsException();
    }
    try {
      while (length > 0) {
        final int bytesWritten = Math.min(length, PIPE_SIZE);
        pos.write(bytes, offset, bytesWritten);
        while (textReader.ready()) {
          final int read = textReader.read();
          switch (read) {
          case 0x0d: // carriage return
            addPendingLine();
            lastWasCR = true;
            break;
          case 0x0a: // line feed
            if (lastWasCR) {
              lastWasCR = false;
            }
            else {
              addPendingLine();
            }
            break;
          default:
            pendingCharBuffer.append((char)read);
            lastWasCR = false;
            break;
          }
        }
        offset += bytesWritten;
        length -= bytesWritten;
      }
      totalNumberOfBytes += length;
    } catch (IOException x) {
      Debug.error(x);
      throw new RuntimeException("Unexpected!", x);
    }
  }

  /**
   * Append the line in the
   */
  private void addPendingLine()
  {
    pendingCharBuffer.flip();
    ringBuffer.append(pendingCharBuffer.toString());
    pendingCharBuffer.clear();
  }

  /**
   * Clear all data.
   * This will clear everything but {@link #getNumberOfCollectedBytes() the total number of bytes handled by this buffer}.
   */
  @Override
  public synchronized void clear()
  {
    ringBuffer.clear();
    pendingCharBuffer.clear();
    try {
      pos.flush();
      while (textReader.ready()) {
        textReader.read();
      }
    } catch (IOException x) {
      Debug.error(x);
      throw new RuntimeException("Unexpected!", x);
    }
  }

  /**
   * Get a temporary view of the lines in this ring buffer.
   * Use only on finished buffer: the returned view will become
   * invalid if any data is appended to this ring buffer.
   * The advantage compared to {@link #toLineIndexable()},
   * {@link #toLineList()}, and {@link #toString()} is
   * that no data is copied.
   *
   * @return lines view
   */
  @NotNull
  public synchronized Indexable<String> viewLines()
  {
    if (pendingCharBuffer.position() > 0) {
      // have to include last line
      final int pos = pendingCharBuffer.position();
      try {
        pendingCharBuffer.flip();
        return Indexable.combine(ringBuffer.view(),
                                 Indexable.singleton(pendingCharBuffer.toString()));
      } finally {
        // reset buffer state
        pendingCharBuffer.limit(pendingCharBuffer.capacity());
        pendingCharBuffer.position(pos);
      }
    }
    return ringBuffer.view();
  }

  /**
   * Get the lines of this ring buffer as an indexable of strings.
   * This is safe, the returned indexable is independent of this buffer
   * with the price of copying the data.
   * @return indexable of size {@linkplain #size()} with the lines of this ring buffer
   */
  @NotNull
  public synchronized Indexable<String> toLineIndexable()
  {
    return viewLines().frozen();
  }

  /**
   * Get the lines of this ring buffer as a list of strings.
   * This is safe, the returned list is independent of this buffer
   * with the price of copying the data.
   * @return list with the lines of this ring buffer
   */
  @NotNull
  public synchronized List<String> toLineList()
  {
    return viewLines().toList();
  }

  /**
   * Get the lines of this ring buffer as an array of strings.
   * This is safe, the returned array is independent of this buffer
   * with the price of copying the data.
   * @return array with the lines of this ring buffer
   */
  @NotNull
  public synchronized String[] toLineArray()
  {
    return viewLines().toArray(String.class);
  }

  /**
   * Get the lines of this line buffer as a concatenated string.
   * @return string representation of the lines, connected with the local
   *         end-of-line sequence
   */
  @NotNull
  public synchronized String toString()
  {
    if (isEmpty()) {
      return Empty.STRING;
    }
    final String result = Types.join(EOL, viewLines());
    return pendingCharBuffer.position() == 0
            ? result + EOL
            : result;
  }

  /**
   * Get the total number of bytes appended to this buffer.
   * @return total number of appended bytes
   */
  @Override
  public synchronized long getNumberOfCollectedBytes()
  {
    return totalNumberOfBytes;
  }
}
