// ============================================================================
// File:               StringTool
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2021-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            27.05.21 10:46
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.generics.Indexable;

import java.util.regex.Pattern;

/**
 * Helper class with tools for string handling.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since May 27, 2021
 */
public final class StringTool
{
  /** Don't create. */
  private StringTool()
  {
  }

  /**
   * Split a string at a given separator character.
   * @param str       string to split
   * @param separator separator character (not handled as a regular expression)
   * @return parts of the string created by splitting it at the given simple separator
   */
  @NotNull
  public static Indexable<String> splitRaw(@NotNull String str, char separator)
  {
    return splitRaw(str, Character.toString(separator));
  }

  /**
   * Split a string at a given raw string pattern.
   * @param str       string to split
   * @param pattern separator pattern (not handled as a regular expression)
   * @return parts of the string created by splitting it at the given simple separator pattern
   */
  @NotNull
  public static Indexable<String> splitRaw(@NotNull String str, @NotNull String pattern)
  {
    return splitRegex(str, Pattern.quote(pattern));
  }

  /**
   * Split a string at a given regular expression pattern.
   * @param str      string to split
   * @param pattern  separator pattern
   * @return parts of the string created by splitting it at the given simple separator pattern
   */
  @NotNull
  public static Indexable<String> splitRegex(@NotNull String str, @NotNull String pattern)
  {
    return split(str, Pattern.compile(pattern));
  }

  /**
   * Split a string at a given regular expression pattern.
   * @param str      string to split
   * @param pattern  separator pattern
   * @return parts of the string created by splitting it at the given simple separator pattern
   */
  @NotNull
  public static Indexable<String> split(@NotNull String str, @NotNull Pattern pattern)
  {
    return Indexable.viewArray(pattern.split(str));
  }

  /**
   * Calculate the length of the longest line in the given text.
   * @param text text to check
   * @return the length of the longest line, w/o the linefeed character(s)
   */
  public static int calcLongestLineLength(@NotNull String text)
  {
    int longest = 0;
    int lineLength = 0;
    final int textLength = text.length();
    for (int c = 0;  c < textLength;  ++c) {
      switch (text.charAt(c)) {
      case '\n':
      case '\r':
        longest = Math.max(longest, lineLength);
        lineLength = 0;
        break;
      default:
        ++lineLength;
        break;
      }
    }
    return Math.max(longest, lineLength);
  }
}
