// ============================================================================
// File:               StringHelper
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            11/23/23 2:19 PM
//=============================================================================
package de.caff.util;

import java.util.regex.Pattern;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 23, 2023
 */
public class StringHelper
{
  /**
   * The end-of-line pattern.
   * This is most useful for splitting multi-line texts into lines.
   */
  public static final Pattern EOL_PATTERN = Pattern.compile("\\R");

  /**
   * Canonize multi-line texts so that each individual line is trimmed.
   * @param text text possibly consisting of more than one line, a {@code null} text will return {@code null}
   * @return text with trimmed lines
   */
  public static String trimMultiLineText(String text)
  {
    return trimMultiLineText(text, false);
  }

  /**
   * Canonize multi-line texts so that each individual line is trimmed.
   * @param text text possibly consisting of more than one line, a {@code null} text will return {@code null}
   * @param trimEnds trim away empty lines at the beginning and the end?
   * @return text with trimmed lines
   */
  public static String trimMultiLineText(String text, boolean trimEnds)
  {
    if (text == null) {
      return null;
    }
    final StringBuilder sb = new StringBuilder(text.length());
    boolean nl = false;
    // note: Integer.MAX_VALUE is necessary, otherwise a last empty line is lost
    for (String line : EOL_PATTERN.split(text, Integer.MAX_VALUE)) {
      if (nl) {
        sb.append('\n');
      }
      else {
        nl = true;
      }
      sb.append(line.trim());
    }
    return trimEnds
            ? sb.toString().trim()
            : sb.toString();
  }
}
