// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.i18n.I18n;

import java.util.Locale;

/**
 *  Concrete implementation of a status flag, using int values.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class StatusFlag
        extends AbstractBasicStatusFlag
        implements BitFlag
{
  /** The bitvalue. */
  private final BitFlag flag;

  /**
   * Constructor.
   * @param p   power of bit (0 to 31)
   * @param basetag i18n tag start. Depending on the value of the flag this will be expanded
   */
  public StatusFlag(int p, @NotNull String basetag)
  {
    super(basetag);
    flag = SimpleBitFlag.get(p);
  }

  /**
   *  Get the i18n tag of this flag depending
   *  whether it is set in the given value or not.
   *  @param mask bit mask value
   *  @return complete i18n tag
   */
  public String getI18nTag(@NotNull BitMask mask)
  {
    return getI18nTag(isSetIn(mask));
  }

  /**
   *  Get the internationalized description of this flag depending
   *  whether it is set in the given value or not, using the default locale.
   *  @param mask bit mask value
   *  @return description text
   */
  public String getDescription(@NotNull BitMask mask)
  {
    return I18n.getString(getI18nTag(mask));
  }

  /**
   *  Get the internationalized description of this flag depending
   *  whether it is set or not.
   *  @param mask bit mask value
   *  @param locale locale to use
   *  @return description text
   */
  public String getDescription(@NotNull BitMask mask,Locale locale)
  {
    return I18n.getString(getI18nTag(mask), locale);
  }

  /**
   * Is this flag set in the given mask?
   *
   * @param mask bit mask
   * @return {@code true}: if the flag is set<br>
   * {@code false}: if the flag is not set
   */
  @Override
  public boolean isSetIn(@NotNull BitMask mask)
  {
    return flag.isSetIn(mask);
  }

  /**
   * Convert this flag into a bit mask.
   *
   * @return bit mask
   */
  @NotNull
  public BitMask toBitMask()
  {
    return BitMaskUtil.toBitMask(flag);
  }

  /**
   * Set this flag in the given bit mask.
   *
   * @param mask bit mask
   * @return new bit mask where this flag is set
   */
  @NotNull
  @Override
  public BitMask setIn(@NotNull BitMask mask)
  {
    return flag.setIn(mask);
  }

  /**
   * Clear this flag in the given bit mask.
   *
   * @param mask bit mask
   * @return new bit mask where this flag is cleared
   */
  @NotNull
  @Override
  public BitMask clearIn(@NotNull BitMask mask)
  {
    return flag.clearIn(mask);
  }

}
