// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;

/**
 * A simple bit flag implementation.
 * <p>
 * For efficiency reasons this class is not constructable,
 * use the {@link SimpleBitFlag#get(int)} factory methods instead.
 * <p>
 * This class is immutable.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class SimpleBitFlag
        implements BitFlag
{
  private static final SimpleBitFlag[] STANDARD_FLAGS = new SimpleBitFlag[64];
  static {
    for (int p = 0;  p < STANDARD_FLAGS.length;  ++p) {
      STANDARD_FLAGS[p] = new SimpleBitFlag(p);
    }
  }

  private final int bitPosition;

  /**
   * Constructor.
   * @param bitPosition  bit flag position, non-negative
   */
  private SimpleBitFlag(int bitPosition)
  {
    if (bitPosition < 0) {
      throw new IllegalArgumentException("pos has to be non-negative!");
    }
    this.bitPosition = bitPosition;
  }

  /**
   * Is this flag set in the given mask?
   *
   * @param mask bit mask
   * @return {@code true}: if the flag is set<br>
   * {@code false}: if the flag is not set
   */
  @Override
  public boolean isSetIn(@NotNull BitMask mask)
  {
    return mask.isSet(bitPosition);
  }

  /**
   * Set this flag in the given bit mask.
   *
   * @param mask bit mask
   * @return new bit mask where this flag is set
   */
  @NotNull
  @Override
  public BitMask setIn(@NotNull BitMask mask)
  {
    return mask.set(bitPosition);
  }

  /**
   * Clear this flag in the given bit mask.
   *
   * @param mask bit mask
   * @return new bit mask where this flag is cleared
   */
  @NotNull
  @Override
  public BitMask clearIn(@NotNull BitMask mask)
  {
    return mask.clear(bitPosition);
  }

  /**
   * Convert this flag into a bit mask.
   *
   * @return bit mask
   */
  @NotNull
  public BitMask toBitMask()
  {
    if (bitPosition < BitMask16.BIT_COUNT) {
      return setIn(BitMask16.ZERO);
    }
    if (bitPosition < BitMask32.BIT_COUNT) {
      return setIn(BitMask32.ZERO);
    }
    if (bitPosition < BitMask64.BIT_COUNT) {
      return setIn(BitMask64.ZERO);
    }
    return setIn(new BitSetBitMask(bitPosition + 1));
  }

  /**
   * Convert this flog into an integer bit mask.
   * @return integer bit mask, possibly {@code 0} if this flag is outside of 32 bit range range
   */
  public int toInt()
  {
    if (bitPosition < 32) {
      return 0x01 << bitPosition;
    }
    return 0;
  }

  /**
   * Convert this flog into a long bit mask.
   * @return long bit mask, possibly {@code 0} if this flag is outside of 64 bit range
   */
  public long toLong()
  {
    if (bitPosition < 64) {
      return 0x01L << bitPosition;
    }
    return 0;
  }

  /**
   * Get the bit position associated with this flag.
   * @return bit position
   */
  public int getBitPosition()
  {
    return bitPosition;
  }

  /**
   * Get the simple bit flag for which the given index is set.
   * @param index bit flag index
   * @return simple bit flag which has the given bit index set
   */
  @NotNull
  public static SimpleBitFlag get(int index)
  {
    if (index < 0  ||  index >= STANDARD_FLAGS.length) {
      return new SimpleBitFlag(index);
    }
    return STANDARD_FLAGS[index];
  }
}
