// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2010-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Empty;
import de.caff.generics.Pair;
import de.caff.generics.Types;
import de.caff.io.InputStreamSink;
import de.caff.util.debug.Debug;
import de.caff.version.SemVer;

import java.io.*;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.jar.JarFile;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * Check whether a new program version is available.
 * <p>
 * This is very similar to {@link VersionChecker}, but uses semantic versions.
 * <p>
 * If a newer one is available the new jar can be downloaded and
 * saved in a way that the old jar is replaced.
 * Possibly the program can also be restarted using the new jar.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SemanticVersionChecker
        implements KnockOffListener
{
  public static final String TAG_SEPARATOR = ":";
  public static final String TAG_COMMENT = "#";
  public static final String TAG_VERSION = "VERSION";
  public static final String TAG_MINJVM = "MINJVM";
  public static final String TAG_NOUPDATE = "NOUPDATE";
  private static final Pattern ZIP_FILE_SEPARATOR_PATTERN = Pattern.compile("/");

  /**
   * Program information necessary for automatic update.
   */
  public interface ProgramAccess
  {
    /**
     * Get a string describing the current version.
     * @return current version as string
     */
    @NotNull
    SemVer getCurrentVersion();

    /**
     * Tell whether a given version is newer.
     * @param version possibly newer semantic version
     * @return {@code true} version is newer than the current version<br>
     *         {@code false} version is not new than the current version
     */
    boolean isNewerVersion(@NotNull SemVer version);

    /**
     * Tell whether a given version is newer.
     * @param version possibly newer version, assumed to represent a semantic version
     * @return {@code true} version is newer than the current version<br>
     *         {@code false} version is not new than the current version
     */
    default boolean isNewerVersion(@NotNull String version)
    {
      final SemVer semVer = SemVer.parse(version);
      if (semVer == null) {
        Debug.error("Not a semantic version: %1", version);
        return false;
      }
      return isNewerVersion(semVer);
    }

    /**
     * Get the URL of the file containing version and possibly other information.
     * <p>
     * The file format is simple.
     * A line starting with a hash character ({@code #}) is starting a comment.
     * Empty lines are ignored.
     * All other lines have the format<br>
     * {@code TAG: CONTENT}<br>
     * where TAG is one of the following
     * <ul>
     *  <li><tt>VERSION</tt>: <tt>CONTENT</tt> is the current version number</li>
     *  <li><tt>MINJVM</tt>: <tt>CONTENT</tt> is the minimal version of the JVM necessary to run the program (eg <tt>1.5</tt>)</li>
     *  <li><tt>NOUPDATE[vnr]</tt>: with <tt>vnr</tt> being a version number (meaning exactly this version) or
     *      a version number prefixed with a &lt; sign (meaning all versions before the given version), and <tt>CONTENT</tt> being the URL
     *      of a HTML file describing the reason why an automatic update for the described versions is not possible.
     *  </li>
     * </ul>
     *
     * @return URL to info file
     */
    @NotNull
    String getInfoFileUrl();

    /**
     * Get the URL of an update zip.
     * This is useful when there is more than one file to update,
     * in which case this zip is downloaded and unpacked.
     *
     * @param versionNumber version number requested.
     * @param oldVersion  old version currently running
     * @return update URL or {@code null} if there is no update possible
     */
    @Nullable
    String getPackageUrl(@NotNull SemVer versionNumber, @NotNull SemVer oldVersion);

    /**
     * Get the URL of the program jar.
     * The program jar is the jar necessary for a complete new installation.
     * @param versionNumber version number requested
     * @param oldVersion number of installed version
     * @return URL of program jar, or {@code null} if there is no update location,
     *         in which case {@link #getPackageUrl(SemVer, SemVer)} should not return {@code null}
     */
    @Nullable
    String getUpdateUrl(@NotNull SemVer versionNumber, @NotNull SemVer oldVersion);

    /**
     * Get the base name of the installed jar file.
     * @return name of installed jar file
     */
    @NotNull
    String getProgramJarName();

    /**
     * Get the file to which the downloaded jar is written.
     * The default behavior (return {@code null}) is to
     * overwrite the currently running jar.
     * @return explicit jar output or {@code null} for the default behavior
     */
    @Nullable
    default File getOutputJar()
    {
      return null;
    }

    /**
     * Show the HTML content of an URL describing why an update is not possible.
     * @param url URL to show
     * @param currentVersion current version
     * @param newVersion new version
     */
    void showInfoNoUpdatePossible(@NotNull URL url, @NotNull SemVer currentVersion, @NotNull SemVer newVersion);

    /**
     * Informative callback called when it is determined that there is no newer version available.
     */
    default void versionIsCurrent()
    {
    }

    /**
     * Get an okay from the user to download and install a new version.
     * <p>
     * Typical question text:<br>
     * "A new version of XY is available: "+version+<br>
     * "Do you want to download and install it?"
     * @param currentVersion current version
     * @param newVersion new version
     *
     * @return {@code true}: install new version<br>
     *         {@code false}: don't install new version
     */
    boolean isUserOkayDownload(@NotNull SemVer currentVersion, @NotNull SemVer newVersion);

    /**
     * Get an okay from the user to restart the program.
     * <p>
     * Typical question text:<br>
     * "The new version "+version+" is now installed."+<br>
     * "Would you like to restart?"
     * @param currentVersion current version
     * @param newVersion new version
     *
     * @return {@code true}: restart into new version<br>
     *         {@code false}: keep running
     */
    boolean isUserOkayRestart(@NotNull SemVer currentVersion, @NotNull SemVer newVersion);

    /**
     * Show an info that a restart of the application is necessary in order to run the new version.
     * <p>
     * This is called when it is not possible to do an automatic restart.
     * Typical text:<br>
     " "The new version "+version+" is now installed."+<br>
     * "You'll need to restart the program to finish the installation."
     *
     * @param currentVersion current version
     * @param newVersion new version
     */
    void showInfoNeedRestart(@NotNull SemVer currentVersion, @NotNull SemVer newVersion);

    /**
     * Show an error method that downloading failed because the file was not accessible.
     * @param currentVersion current version
     * @param newVersion new version which was tried to be downloaded
     * @param url URL with which it was tried to download
     * @param exception the exception on which the error is based
     */
    void showErrorDownloadFailureNotFound(@NotNull SemVer currentVersion,
                                          @NotNull SemVer newVersion,
                                          @NotNull String url,
                                          @NotNull Exception exception);

    /**
     * Show an error method that downloading failed because the file could not be downloaded due to i/o problems.
     * @param currentVersion current version
     * @param newVersion new version which was tried to be downloaded
     * @param url URL with which it was tried to download
     * @param exception the exception on which the error is based
     */
    void showErrorDownloadFailureIO(@NotNull SemVer currentVersion,
                                    @NotNull SemVer newVersion,
                                    @NotNull String url,
                                    @NotNull IOException exception);

    /**
     * End the application, but don't exit.
     * <p>
     * This should write all necessary data to disk, and store the preferences,
     * stop all threads no longer necessary and close all windows.
     * @return array of commandline arguments to be provided for the newly started application in order to restore
     *         the current state
     */
    @Nullable
    String[] endWithoutExit();

    /**
     * Register a shutdown hook for renaming the temp file where the updated jar is stored to?
     * @param tmpFile temporary file
     * @return {@code true}: for automatically taking care<br>
     *         {@code false}: the program access itself cares for renaming
     */
    default boolean registerRenameShutdownHook(@NotNull File tmpFile)
    {
      return true;
    }
  }

  /** Access to program information and user. */
  @NotNull
  private final ProgramAccess programAccess;

  /**
   * Helper class for loading version info in background.
   */
  private static class VersionLoader
          extends Worker
  {
    /** Tag separator pattern. */
    private static final Pattern TAG_SEPARATOR_PATTERN = Pattern.compile(TAG_SEPARATOR);
    /** Version read. */
    private SemVer version;
    /** Minimal necessary JVM version. */
    private String minJVM;
    /** Program access. */
    @NotNull
    private final ProgramAccess programAccess;
    /** List of pairs of version and no update reason URL. */
    private final List<Pair<String>> noUpdateInfo = new LinkedList<>();

    /**
     * Default constructor.
     * @param programAccess program access
     */
    private VersionLoader(@NotNull ProgramAccess programAccess)
    {
      this.programAccess = programAccess;
    }

    /**
     * Implement this in extending classes to do the work.
     *
     * @throws Exception any exception thrown during work
     */
    @Override
    protected void execute() throws Exception
    {
      URL url = new URL(programAccess.getInfoFileUrl());
      try (InputStream input = url.openStream()) {
        BufferedReader reader = new BufferedReader(new InputStreamReader(input));
        String line;
        while ((line = reader.readLine()) != null) {
          line = line.trim();
          if (line.isEmpty()) {
            continue;
          }
          if (line.startsWith(TAG_COMMENT)) {
            Debug.message("Comment: " + line.substring(1));
          }
          else {
            String[] parts = TAG_SEPARATOR_PATTERN.split(line, 2);
            String tag = parts[0].trim().toUpperCase();
            if (parts.length == 2) {
              if (TAG_VERSION.equalsIgnoreCase(tag)) {
                version = SemVer.parse(parts[1].trim());
                if (version == null) {
                  Debug.error("Not a semantic version: %1", parts[1].trim());
                }
              }
              else if (TAG_MINJVM.equalsIgnoreCase(tag)) {
                minJVM = parts[1].trim();
              }
              else if (tag.startsWith(TAG_NOUPDATE) &&
                       tag.charAt(TAG_NOUPDATE.length()) == '[' &&
                       tag.endsWith("]")) {
                String versionInfo = tag.substring(TAG_NOUPDATE.length() + 1, tag.length() - 1);
                noUpdateInfo.add(Pair.createPair(versionInfo, parts[1].trim()));
              }
              else {
                throw new IOException("Unknown tag in version description in " + url + ":\n" + line);
              }
            }
            else {
              throw new IOException("Format error in version description in " + url + ":\n" + line);
            }
          }
        }
        reader.close();
        if (version == null) {
          throw new IOException("No version number found in " + url);
        }
        Debug.message("Current version:   " + programAccess.getCurrentVersion() +
                      "\nPublished version: " + version);
      }
    }

    /**
     * Get the loaded version.
     * @return version or {@code null}
     */
    @Nullable
    public SemVer getVersion()
    {
      return version;
    }

    /**
     * Get the minimal JVM necessary to run the new version.
     * @return necessary JVM or {@code null}
     */
    @Nullable
    public String getMinJVM()
    {
      return minJVM;
    }

    /**
     * Return an URL given access to a reason why a given program version cannot update automatically.
     * <p>
     * If there is a message URL for a given version, a reinstall is necessary.
     * The message URL allows to give information for the reason.
     * @return message URL for the given version, or {@code null} if there is not special handling necessary
     */
    @Nullable
    public URL getNoUpdateReasonUrl()
    {
      for (Pair<String> p : noUpdateInfo) {
        String v = p.first;
        if (v.startsWith("<")) {
          if (programAccess.isNewerVersion(v.substring(1))) {
            return getRelativeUrl(p.second);
          }
        }
        else if (v.equals(programAccess.getCurrentVersion())) {
          return getRelativeUrl(p.second);
        }
      }
      return null;
    }

    /**
     * Resolve a possibly relative URL.
     * @param subUrl possible relative URL
     * @return absolute URL
     */
    private URL getRelativeUrl(String subUrl)
    {
      try {
        URL baseUrl = new URL(programAccess.getInfoFileUrl());
        return new URL(baseUrl, subUrl);
      } catch (MalformedURLException e) {
        Debug.error(e);
        return null;
      }
    }
  }

  /**
   * Helper class for a simple version build from
   * numbers separated by points. It is assumed that
   * each part is just counted upwards, with the result
   * that version <tt>1.10</tt> is newer than version <tt>1.2</tt>.
   */
  public static class Version
          implements Comparable<Version>
  {
    private static final Pattern VERSION_SEPARATOR_PATTERN = Pattern.compile("[._]");
    private final int[] parts;

    /**
     * Constructor.
     * @param version version number
     * @throws NumberFormatException if version contains anything else than digits and points (or underscores)
     */
    public Version(String version)
    {
      String[] arr = VERSION_SEPARATOR_PATTERN.split(version);
      parts = new int[arr.length];
      for (int a = 0;  a < arr.length;  ++a) {
        parts[a] = Integer.parseInt(arr[a]);
      }
    }

    /**
     * Compares this object with the specified object for order.  Returns a
     * negative integer, zero, or a positive integer as this object is less
     * than, equal to, or greater than the specified object.
     * @param version the object to be compared.
     * @return a negative integer, zero, or a positive integer as this object
     *         is less than, equal to, or greater than the specified object.
     * @throws ClassCastException if the specified object's type prevents it
     *                            from being compared to this object.
     */
    @Override
    public int compareTo(@NotNull Version version)
    {
      int len = Math.min(parts.length, version.parts.length);
      for (int p = 0;  p < len;  ++p) {
        int d = parts[p] - version.parts[p];
        if (d != 0) {
          return d;
        }
      }
      return parts.length - version.parts.length;
    }
  }

  /**
   * Create version checker.
   * @param programAccess access to checked program
   */
  private SemanticVersionChecker(@NotNull ProgramAccess programAccess)
  {
    this.programAccess = programAccess;
  }

  /**
   * Invoked when the worker has finished.
   * @param worker worker which finished
   */
  @Override
  public void knockedOff(@NotNull Worker worker)
  {
    try {
      worker.rethrow();
      SemVer version = ((VersionLoader)worker).getVersion();
      String minJVM = ((VersionLoader)worker).getMinJVM();
      if (version != null) {
        if (!programAccess.getCurrentVersion().equals(version)) {
          if (programAccess.isNewerVersion(version)) {
            // newer version available
            Debug.message("New version %0 available (current: %1)", version, programAccess.getCurrentVersion());
            URL noUpdateUrl = ((VersionLoader)worker).getNoUpdateReasonUrl();
            if (noUpdateUrl != null) {
              Debug.message("Have no-update URL: "+noUpdateUrl);
              programAccess.showInfoNoUpdatePossible(noUpdateUrl, programAccess.getCurrentVersion() , version);
              return;
            }
            if (minJVM != null) {
              if (new Version(minJVM).compareTo(new Version(System.getProperty("java.version"))) > 0) {
                // cannot be sure that program will be able to run after update, so stop here
                Debug.error("Update not possible: JVM is too old :%0 (needed: %1)!",
                            System.getProperty("java.version"),
                            minJVM);
                return;
              }
              Debug.message("Current JVM %0 is good enough (min: %1)",
                            System.getProperty("java.version"),
                            minJVM);
            }
            final File jarFile = getJarFile();
            final File parentFile = jarFile != null ? jarFile.getParentFile() : null;
            if (jarFile != null  &&  (jarFile.exists()
                                              ? jarFile.canWrite()
                                              : parentFile != null && parentFile.canWrite())) {
              boolean yes = programAccess.isUserOkayDownload(programAccess.getCurrentVersion(),
                                                             version);

              if (yes) {
                final File temp = downloadAndInstall(programAccess.getCurrentVersion(), version, jarFile);
                if (temp != null) {
                  List<String> commandLine = getStartCommand(jarFile);
                  if (commandLine == null) {
                    programAccess.showInfoNeedRestart(programAccess.getCurrentVersion(), version);
                  }
                  else {
                    boolean answer = programAccess.isUserOkayRestart(programAccess.getCurrentVersion(), version);
                    if (answer) {
                      if (writeStreamToFile(Files.newInputStream(temp.toPath()), jarFile)) {
                        temp.deleteOnExit();
                        String[] args = programAccess.endWithoutExit();
                        if (args != null) {
                          commandLine.addAll(Types.asList(args));
                          String[] command = commandLine.toArray(Empty.STRING_ARRAY);
                          Process process = new ProcessBuilder(command).start();
                          InputStreamSink.startOn(process.getInputStream());
                          InputStreamSink.startOn(process.getErrorStream());
                          process.waitFor();
                          System.exit(0);
                        }
                        return;
                      }
                    }
                  }
                  if (programAccess.registerRenameShutdownHook(temp)) {
                    // if we come here, we have to take care that temp is copied over
                    // Under Windows it's no good idea to copy over the jar file while we are running
                    Runtime.getRuntime().addShutdownHook(new Thread(() -> {
                      try {
                        writeStreamToFile(Files.newInputStream(temp.toPath()), jarFile);
                      } catch (IOException e) {
                        Debug.error(e);
                      }
                      temp.delete();
                    }));
                  }
                }
              }
            }
            else {
              Debug.warn("No jar file access, cannot update!");
            }
          }
          else {
            Debug.message("No newer version available (current: %0)", programAccess.getCurrentVersion());
            programAccess.versionIsCurrent();  // indeed it is even newer
          }
        }
        else {
          programAccess.versionIsCurrent();
        }
      }
    } catch (Throwable t) {
      Debug.warn(t);
    }
  }

  private static List<String> getStartCommand(File jarFile)
  {
    try {
      List<String> result = new LinkedList<>();
      String javaHome = System.getProperty("java.home");
      if (javaHome != null) {
        result.add(javaHome+File.separator+"bin"+File.separator+"java");
        result.add("-mx"+Runtime.getRuntime().maxMemory());
        result.add("-jar");
        result.add(jarFile.toString());
        return result;
      }
    } catch (Throwable t) {
      Debug.error(t);
    }
    return null;
  }

  /**
   * Write the complete content of a stream to a file.
   * @param is input stream
   * @param target target
   * @return {@code true} if anything was written<br>
   *         {@code false}: if nothing was written
   * @throws IOException on read and write errors
   */
  public static boolean writeStreamToFile(@NotNull InputStream is,
                                          @NotNull File target) throws IOException
  {
    Debug.trace("writeStreamToFile(???, "+target+")");
    if (target.getParentFile().isDirectory()  ||
        target.getParentFile().mkdirs()) {
      byte[] buffer = new byte[4096];
      int bytes = 0;
      int len;
      try (OutputStream os = Files.newOutputStream(target.toPath())) {
        while ((len = is.read(buffer)) > 0) {
          os.write(buffer, 0, len);
          bytes += len;
        }
      } finally {
        is.close();
      }
      return bytes > 0;
    }
    return false;
  }

  private static void unpackZip(File zipFile, File targetDir) throws IOException
  {
    try (ZipFile zip = new ZipFile(zipFile, ZipFile.OPEN_READ)) {
      for (Enumeration<?> e = zip.entries(); e.hasMoreElements(); ) {
        ZipEntry entry = (ZipEntry)e.nextElement();
        File target = targetDir;
        String[] parts = ZIP_FILE_SEPARATOR_PATTERN.split(entry.getName());
        for (String part : parts) {
          target = new File(target, part);
        }
        if (entry.isDirectory()) {
          if (!target.mkdirs()) {
            Debug.error("Failed to create dir: %0", target);
          }
        }
        else {
          writeStreamToFile(zip.getInputStream(entry), target);
        }
      }
    }
  }

  private static boolean isJar(File file)
  {
    try {
      try (final JarFile ignored = new JarFile(file)) {
        return true;
      }
    } catch (IOException e) {
      Debug.warn("Not a jar: "+file);
      return false;
    }
  }

  private File downloadAndInstall(@NotNull SemVer oldVersion, @NotNull SemVer version, @NotNull  File jarFile)
  {
    String urlAsString = programAccess.getUpdateUrl(version, oldVersion);
    Debug.message("updateUrl="+urlAsString);
    try {
      if (urlAsString != null) {
        URL url = new URL(urlAsString);
        Debug.message("Trying update of jar file "+jarFile+"...");

        InputStream is = url.openStream();
        File temp = File.createTempFile(jarFile.getName(), "jar");
        //temp.deleteOnExit();
        Debug.message("tmp="+temp);
        if (writeStreamToFile(is, temp)  &&  isJar(temp)) {
          // writeStreamToFile(new FileInputStream(temp), jarFile);
          return temp;
        }
      }
      // Update failed, try complete install
      Debug.message("Trying complete download...");
      urlAsString = programAccess.getPackageUrl(version, programAccess.getCurrentVersion());
      Debug.message("packageUrl="+urlAsString);
      if (urlAsString != null) {
        URL url = new URL(urlAsString);
        InputStream is = url.openStream();
        File temp = File.createTempFile("kroak", "zip");
        // temp.deleteOnExit();
        if (writeStreamToFile(is, temp)) {
          Debug.message("Download complete, unpacking ZIP...");
          unpackZip(temp, jarFile.getParentFile().getParentFile());
          Debug.message("Ready.");
          return temp;
        }
      }
    } catch (MalformedURLException | FileNotFoundException e) {
      programAccess.showErrorDownloadFailureNotFound(programAccess.getCurrentVersion(),
                                                     version,
                                                     urlAsString,
                                                     e);
    } catch (IOException e) {
      programAccess.showErrorDownloadFailureIO(programAccess.getCurrentVersion(),
                                               version,
                                               urlAsString,
                                               e);
    }
    return null;
  }

  /**
   *  Returns the path of the jar file from which this program was started.
   *  @return jar file or {@code null} when jar is not found
   */
  private File getJarFile()
  {
    final File f = programAccess.getOutputJar();
    if (f != null) {
      try {
        return f.getCanonicalFile();
      } catch (IOException e) {
        Debug.error("Cannot canonize file %0: %1", f, e);
        return null;  // better safe than sorry
      }
    }
    try {
      final String classpath = System.getProperty("java.class.path");
      if (classpath != null) {
        final String[] jars = classpath.split(Pattern.quote(File.pathSeparator));
        for (String jar : jars) {
          if (jar.endsWith(programAccess.getProgramJarName())) {
            File file = new File(jar);
            if (file.exists()) {
              return file.getCanonicalFile();
            }
          }
        }
      }
    } catch (Throwable t) {
      Debug.error(t);
    }
    return null;
  }

  /**
   *  Check whether a new version is available.
   *  @param pa access to already running program
   */
  public static void checkForNewVersion(@NotNull ProgramAccess pa)
  {
    final VersionLoader loader = new VersionLoader(pa);
    loader.addKnockOffListener(new SemanticVersionChecker(pa));
    new Thread(loader).start();
  }
}
