// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.generics.ByteIndexable;
import de.caff.generics.Indexable;

import java.util.*;

/**
 * A byte buffer to which you can add data up to a maximum size.
 * <p>
 * If the buffer overflows you can only retrieve the latest
 * {@linkplain #getLimit()} bytes.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August 10, 2022
 */
public class RingBuffer<T>
{
  /** Initial limit: 1 MiB. */
  static final int INITIAL_LIMIT = 0x100000;

  /** Buffer limit. */
  private final int limit;
  /**
   * Internal ring buffer.
   * May change size until it overflows.
   */
  private T[] ringBuffer;
  /** Offset in the active buffer where the next byte will be added. */
  private int bufferOffset;
  /** Number of cached items. */
  private int itemsCached;
  /** Total number of items appended to this ring buffer. */
  private long totalNumberOfItems;

  /**
   * Create a ring buffer with the given size.
   * @param limit size limit of this buffer
   */
  public RingBuffer(int limit)
  {
    this(limit, INITIAL_LIMIT);

  }
  /**
   * Create a ring buffer with the given size.
   * This allows to set the initial limit for testing.
   * @param limit size limit of this buffer
   * @param initialLimit initial limit
   */
  @SuppressWarnings("unchecked")
  RingBuffer(int limit, int initialLimit)
  {
    if (limit <= 0) {
      throw new IllegalArgumentException("limit has to be positive, but is " + limit);
    }
    if (limit < initialLimit) {
      ringBuffer = (T[])new Object[limit];
    }
    else {
      ringBuffer = (T[])new Object[initialLimit];
    }
    this.limit = limit;
  }

  /**
   * Get the limit of this ring buffer.
   * @return buffer limit
   */
  public int getLimit()
  {
    return limit;
  }

  /**
   * Get the number of valid bytes cached in this buffer.
   * @return number of bytes cached
   */
  public synchronized int size()
  {
    return itemsCached;
  }

  /**
   * Is this ring buffer empty?
   * @return {@code true} if the buffer is empty<br>
   *         {@code false} if not
   */
  public synchronized boolean isEmpty()
  {
    return itemsCached == 0;
  }

  /**
   * Append one item.
   * @param item item to append
   */
  @SuppressWarnings("unchecked")
  public synchronized void append(T item)
  {
    if (bufferOffset < ringBuffer.length) {
      ringBuffer[bufferOffset++] = item;
      if (itemsCached < limit) {
        ++itemsCached;
      }
      ++totalNumberOfItems;
    }
    else {
      if (ringBuffer.length == limit) {
        ringBuffer[0] = item;
        bufferOffset = 1;
        ++totalNumberOfItems;
      }
      else {
        append((T[])(new Object[] { item }));
      }
    }
  }

  /**
   * Append the given items to the buffer.
   * @param items items to append
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  public synchronized final void append(@NotNull T... items)
  {
    append(items, 0, items.length);
  }

  /**
   * Append the given byte array part to this buffer.
   * @param items  items to append
   * @param offset start offset of the bytes to append
   * @param length number of bytes to append
   * @throws IndexOutOfBoundsException if the parameters define a part which is at least partially outside the {@code bytes}
   */
  public synchronized void append(@NotNull T[] items, int offset, int length)
  {
    if ((offset < 0) || (offset > items.length) || (length < 0) ||
        ((offset + length) - items.length > 0)) {
      throw new IndexOutOfBoundsException();
    }
    appendInternally(items, offset, length);
  }

  /**
   * Internal implementation of appending.
   * @param items  items to append
   * @param offset start offset of the bytes to append
   * @param length number of bytes to append
   */
  private void appendInternally(@NotNull T[] items, int offset, int length)
  {
    if (length > ringBuffer.length - bufferOffset) {
      if (ringBuffer.length < limit) {
        // expand buffer
        final int newSize;
        if (ringBuffer.length > Integer.MAX_VALUE / 2) {
          newSize = limit;
        }
        else {
          newSize = Math.min(limit, 2 * ringBuffer.length);
        }
        ringBuffer = Arrays.copyOf(ringBuffer, newSize);
        appendInternally(items, offset, length);
        return;
      }
      if (length > limit) {
        clear();
        final int delta = length - limit;
        appendInternally(items, offset + delta, limit);
        return;
      }
      assert ringBuffer.length == limit;
      final int atEnd = limit - bufferOffset;
      if (atEnd > 0) {
        System.arraycopy(items, offset, ringBuffer, bufferOffset, atEnd);
        offset += atEnd;
        length -= atEnd;
        totalNumberOfItems += atEnd;
      }
      bufferOffset = 0;
      itemsCached = limit;
    }
    System.arraycopy(items, offset, ringBuffer, bufferOffset, length);
    bufferOffset += length;
    if (itemsCached < limit) {
      itemsCached += Math.min(length, limit - itemsCached);
    }
    totalNumberOfItems += length;
  }

  /**
   * Create a list with the items in this buffer.
   * @return list of size {@linkplain #size()} containing the buffered items
   */
  @NotNull
  public synchronized List<T> toList()
  {
    if (itemsCached == 0) {
      return Collections.emptyList();
    }

    final List<T> result = new ArrayList<>(itemsCached);
    if (bufferOffset < itemsCached) {
      final int atEnd = itemsCached - bufferOffset;
      copyTo(limit - atEnd, atEnd, result);
      copyTo( 0, bufferOffset, result);
    }
    else {
      copyTo( bufferOffset - itemsCached, itemsCached, result);
    }
    return result;
  }

  private void copyTo(int from, int len, @NotNull Collection<T> target)
  {
    final int end = from + len;
    for (int i = from;  i < end;  ++i) {
      target.add(ringBuffer[i]);
    }
  }

  /**
   * Get an indexable view of this ring buffer.
   * <p>
   * This is more efficient than copying all bytes in {@linkplain #toList()},
   * especially if only a few items of a larger content are of interest. But the
   * price is high: any change to this buffer can break the returned view. So this
   * is best used on a finished buffer.
   *
   * @return indexable view of this buffer
   * @see #toIndexable()
   * @see #toList()
   */
  @NotNull
  public synchronized Indexable<T> view()
  {
    if (itemsCached == 0) {
      return Indexable.emptyIndexable();
    }
    if (bufferOffset >= itemsCached) {
      return Indexable.viewArray(ringBuffer).subSet(bufferOffset - itemsCached, bufferOffset);
    }
    final int size = itemsCached;
    final int start = bufferOffset;
    final int boundary = ringBuffer.length - start;
    return new Indexable.Base<T>()
    {
      @Override
      public T get(int index)
      {
        if (index < boundary) {
          return ringBuffer[start + index];
        }
        return ringBuffer[index - boundary];
      }

      @Override
      public int size()
      {
        return size;
      }
    };
  }

  /**
   * Get a stable byte indexable from this buffer.
   * <p>
   * Compared to {@link #view()} the returned indexable
   * is decoupled from this buffer, with the price
   * of a copy of the bytes. Compared to the result of
   * <pre>{@code
   * ByteIndexable.viewArray(ringBuffer.toByteArray())
   * }</pre>
   * this method provides an already {@linkplain ByteIndexable#frozen() frozen}
   * byte indexable.
   * @return stable frozen byte indexable
   * @see #toList()
   * @see #view()
   */
  @NotNull
  public synchronized Indexable<T> toIndexable()
  {
    return view().frozen();
  }

  /**
   * Clear all data.
   * This will clear everything but {@link #getNumberOfCollectedItems() the total number of bytes handled by this buffer}.
   */
  public synchronized void clear()
  {
    bufferOffset = 0;
    itemsCached = 0;
  }

  /**
   * Get the total number of items appended to this buffer.
   * @return total number of appended items
   */
  public synchronized long getNumberOfCollectedItems()
  {
    return totalNumberOfItems;
  }
}
