// ============================================================================
// File:               ParameterType
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            6/16/23 9:58 AM
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Enums;
import de.caff.generics.Indexable;
import de.caff.util.measure.PhysicalLength;

/**
 * Types of parameters (Java properties) requested via utility class.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since June 16, 2023
 */
public enum ParameterTypes
{
  /**
   * String parameter.
   * Requested by {@link Utility#getStringParameter(java.lang.String, java.lang.String)}
   */
  String,
  /**
   * Boolean parameter.
   * Requested by {@link Utility#getBooleanParameter(java.lang.String, boolean)} and
   * {@link Utility#getBooleanParameter(java.lang.String, boolean, java.lang.String...)}.
   */
  Bool,
  /**
   * Integer parameter.
   * Requested by {@link Utility#getIntParameter(java.lang.String, int)} and
   * {@link Utility#getIntParameter(java.lang.String, int, int)}.
   */
  Int,
  /**
   * Long parameter.
   * Requested by {@link Utility#getLongParameter(java.lang.String, long)} and
   * {@link Utility#getLongParameter(java.lang.String, long, int)}.
   */
  Long,
  /**
   * Double parameter.
   * Requested by {@link Utility#getDoubleParameter(java.lang.String, double)}.
   */
  Double,
  /**
   * Float parameter.
   * Requested by {@link Utility#getFloatParameter(java.lang.String, float)}.
   */
  Float,
  /**
   * Enum parameter.
   * Requested by {@link Utility#getEnumParameter(java.lang.String, java.lang.Enum)}.
   */
  Enum,
  /**
   * Color parameter.
   * Requested by {@link Utility#getColorParameter(java.lang.String, java.awt.Color)}.
   */
  Color,
  /**
   * Length parameter.
   * Requested by {@link Utility#getLengthParameter(java.lang.String, PhysicalLength)}.
   */
  Length;

  /**
   * The values of this enum.
   * Use this instead of {@link #values()} because it does not require internal array copying.
   */
  @NotNull
  public static final Indexable<ParameterTypes> VALUES = Indexable.viewArray(values());

  /**
   * Get the ParameterType enum associated with the given internal value.
   *
   * @param internalValue internal value
   * @return associated enum value, or {@code null} if there is none
   */
  @Nullable
  public static ParameterTypes fromInternal(int internalValue)
  {
    return Enums.getEnumFromOrdinal(VALUES, internalValue);
  }

  /**
   * Get the ParameterType enum associated with the given internal value.
   *
   * @param internalValue internal value
   * @param defaultValue  default value
   * @return associated enum value, or {@code defaultValue} if there is none
   */
  @NotNull
  public static ParameterTypes fromInternal(int internalValue, @NotNull ParameterTypes defaultValue)
  {
    return Enums.getEnumFromOrdinal(VALUES, internalValue, defaultValue);
  }
}
