// ============================================================================
// File:               MemFormat
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2021-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            14.05.21 11:01
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Indexable;
import de.caff.generics.Types;

import java.math.BigInteger;

/**
 * Helper for formatting memory sizes in a more human-readable form.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since Mai 14, 2021
 */
public enum MemFormat
{
  /** Memory sizes are based on 10, i.e. a kilobyte is 1000 (10^3) bytes. */
  Base10(1000,
         "k", // kilo   (10^3)
         "M", // Mega   (10^6)
         "G", // Giga   (10^9)
         "T", // Tera   (10^12)
         "P", // Peta   (10^15)
         "E", // Exa    (10^18)
         "Z", // Zetta  (10^21)
         "Y", // Yotta  (10^24)
         "X", // Xona   (10^27)  // todo: this is different compared to SIPrefix. Why?
         "W", // Weka   (10^30)
         "V", // Vunda  (10^33)
         "U", // Uda    (10^36)
         "T", // Treda  (10^39)
         "S", // Sorta  (10^42)
         "R", // Rinta  (10^45)
         "Q"), // Quexa  (10^48)
         // going on with (but I don't know the prefixes): pepta, ocha, nena, minga, luma

  /** Memory sizes are based on 2, i.e. a kilobyte is 1024 (2^10) bytes. */
  Base2(1024,
        "ki", // kilo  (2^10)
        "Mi", // Mega  (2^20)
        "Gi", // Giga  (2^30)
        "Ti", // Tera  (2^40)
        "Pi", // Peta  (2^50)
        "Ei", // Exa   (2^60)
        "Zi", // Zetta (2^70)
        "Yi", // Yotta (2^80)
        "Xi", // Xona  (2^90)
        "Wi", // Weka  (2^100)
        "Vi", // Vunda (2^110)
        "Ui", // Uda   (2^120)
        "Ti", // Treda (2^130)
        "Si", // Sorta (2^140)
        "Ri", // Rinta (2^150)
        "Qi");  // Quexa (2^160)
        // going on with (but I don't know the prefixes): pepta, ocha, nena, minga, luma

  /** !00 as a big integer. */
  @NotNull
  private static final BigInteger HUNDRED = BigInteger.valueOf(100);

  /** Size of one kilobyte when using this base. */
  public final int kiloSize;
  /** Size of one kilobyte when using this base, as BigInteger. */
  public final BigInteger bigKiloSize;
  /** List of prefixes when using this base. */
  @NotNull
  public final Indexable<String> prefixes;

  /**
   * Constructor.
   * @param kiloSize size of one kilobyte when using this base
   * @param prefixes prefixes when using this base
   */
  MemFormat(int kiloSize, @NotNull String... prefixes)
  {
    this.kiloSize = kiloSize;
    bigKiloSize = BigInteger.valueOf(kiloSize);
    this.prefixes = Indexable.viewArray(prefixes);
  }

  /**
   * Get a human-readable string representation of a memory size.
   * This will ussually return a string with no more than 3 significant digits.
   * @param memSize memory size
   * @return human-readable string representation like 256M
   */
  @NotNull
  public String toString(long memSize)
  {
    return toString(BigInteger.valueOf(memSize));
  }

  /**
   * Get a human-readable string representation of a memory size.
   * This will usually return a string with no more than 3 significant digits.
   * @param memSize memory size
   * @return human-readable string representation like 256M
   */
  @NotNull
  public String toString(@NotNull BigInteger memSize)
  {
    return toString(memSize, null, null);
  }

  /**
   * Get a human-readable string representation of a memory size.
   * This will usually return a string with no more than 3 significant digits.
   * @param memSize memory size
   * @param sep     separator between number and memory prefix, none if {@code null}
   *                or empty
   * @param suf     additional suffix, e.g. {@code "Byte"}, none if {@code null}
   *                or empty
   * @return human-readable string representation like <tt>256 MB</tt>
   *         with a space character as separator and a {@code "B"} as suffix
   */
  @NotNull
  public String toString(long memSize,
                         @Nullable String sep,
                         @Nullable String suf)
  {
    return toString(BigInteger.valueOf(memSize), sep, suf);
  }

  /**
   * Get a human-readable string representation of a memory size.
   * This will usually return a string with no more than 3 significant digits.
   * @param memSize memory size
   * @param sep     separator between number and memory prefix, none if {@code null}
   *                or empty
   * @param suf     additional suffix, e.g. {@code "Bytes"}, none if {@code null}
   *                or empty
   * @return human-readable string representation like <tt>256 MB</tt>
   *         with a space character as separator and a {@code "B"} as suffix
   */
  @NotNull
  public String toString(@NotNull BigInteger memSize,
                         @Nullable String sep,
                         @Nullable String suf)
  {
    sep = Types.notNull(sep);
    suf = Types.notNull(suf);
    if (memSize.signum() == -1) {
      return "-" + toString(memSize.negate());
    }
    if (memSize.compareTo(bigKiloSize) < 0) {  // no prefix
      return memSize.toString();
    }
    else {
      long rest = 0;
      for (String prefix : prefixes) {
        rest = memSize.mod(bigKiloSize).longValue();
        memSize = memSize.divide(bigKiloSize);
        if (memSize.compareTo(bigKiloSize) < 0) {
          final StringBuilder sb = new StringBuilder(16);
          if (rest > 0) {
            final double fraction = rest / (double)kiloSize;
            if (memSize.compareTo(BigInteger.TEN) < 0) {
              final int round = (int)Math.round(100 * fraction);
              if (round == 100) {
                sb.append(memSize.add(BigInteger.ONE));
              }
              else {
                sb.append(memSize);
                if (round != 0) {
                  sb.append(String.format(".%d", round));
                }
              }
            }
            else if (memSize.compareTo(HUNDRED) < 0) {
              final int round = (int)Math.round(10 * fraction);
              if (round == 10) {
                sb.append(memSize.add(BigInteger.ONE));
              }
              else {
                sb.append(memSize);
                if (round != 0) {
                  sb.append(String.format(".%d", round));
                }
              }
            }
            else {
              sb.append(memSize);
            }
          }
          else {
            sb.append(memSize);
          }
          sb.append(sep).append(prefix).append(suf);
          return sb.toString();
        }
      }
      // not enough prefixes
      if (BigInteger.valueOf(2 * rest).compareTo(bigKiloSize) >= 0) {
        memSize = memSize.add(BigInteger.ONE);
      }
      return String.format("%s%s%s%s", memSize, sep, prefixes.gyt(-1), suf);
    }
  }
}
