// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;

import java.io.IOException;
import java.io.Writer;

/**
 * Basically the same as {@link java.io.StringWriter},
 * but using a more efficient {@link java.lang.StringBuilder}
 * instead of a {@link java.lang.StringBuffer}.
 * <p>
 * This means that this class is not thread-safe, but in most usage
 * scenarios this is not necessary.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ImprovedStringWriter
        extends Writer
{
  private final StringBuilder buffer;

  /**
   * Constructor.
   */
  public ImprovedStringWriter()
  {
    lock = buffer = new StringBuilder();
  }

  /**
   * Constructor.
   * @param initialCapacity initial buffer capacity
   * @throws java.lang.IllegalArgumentException if {@code initialCapacity} is negative
   */
  public ImprovedStringWriter(int initialCapacity)
  {
    if (initialCapacity < 0) {
      throw new IllegalArgumentException("Negative buffer size");
    }
    lock = buffer = new StringBuilder(initialCapacity);
  }

  /**
   * Writes a single character.  The character to be written is contained in
   * the 16 low-order bits of the given integer value; the 16 high-order bits
   * are ignored.
   *
   * <p> Subclasses that intend to support efficient single-character output
   * should override this method.
   *
   * @param c int specifying a character to be written
   * @throws java.io.IOException If an I/O error occurs
   */
  @Override
  public void write(int c) throws IOException
  {
    buffer.append((char)c);
  }

  /**
   * Writes a portion of an array of characters.
   *
   * @param cbuf Array of characters
   * @param off  Offset from which to start writing characters
   * @param len  Number of characters to write
   * @throws java.io.IOException If an I/O error occurs
   */
  @Override
  public void write(@NotNull char[] cbuf, int off, int len) throws IOException
  {
    buffer.append(cbuf, off, len);
  }

  /**
   * Writes a string.
   *
   * @param str String to be written
   * @throws java.io.IOException If an I/O error occurs
   */
  @Override
  public void write(@NotNull String str) throws IOException
  {
    buffer.append(str);
  }

  /**
   * Writes a portion of a string.
   *
   * @param str A String
   * @param off Offset from which to start writing characters
   * @param len Number of characters to write
   * @throws IndexOutOfBoundsException If <tt>off</tt> is negative, or <tt>len</tt> is negative,
   *                                   or <tt>off+len</tt> is negative or greater than the length
   *                                   of the given string
   * @throws java.io.IOException       If an I/O error occurs
   */
  @Override
  public void write(@NotNull String str, int off, int len) throws IOException
  {
    buffer.append(str, off, len);
  }

  /**
   * Appends the specified character sequence to this writer.
   *
   * <p> An invocation of this method of the form <tt>out.append(csq)</tt>
   * behaves in exactly the same way as the invocation
   *
   * <pre>
   *     out.write(csq.toString()) </pre>
   *
   * <p> Depending on the specification of <tt>toString</tt> for the
   * character sequence <tt>csq</tt>, the entire sequence may not be
   * appended. For instance, invoking the <tt>toString</tt> method of a
   * character buffer will return a subsequence whose content depends upon
   * the buffer's position and limit.
   *
   * @param csq The character sequence to append.  If <tt>csq</tt> is
   *            <tt>null</tt>, then the four characters <tt>"null"</tt> are
   *            appended to this writer.
   * @return This writer
   * @throws java.io.IOException If an I/O error occurs
   * @since 1.5
   */
  @Override
  public Writer append(CharSequence csq) throws IOException
  {
    if (csq == null) {
      buffer.append("null");
    }
    else {
      buffer.append(csq);
    }
    return this;
  }

  /**
   * Appends a subsequence of the specified character sequence to this writer.
   * <tt>Appendable</tt>.
   *
   * <p> An invocation of this method of the form <tt>out.append(csq, start,
   * end)</tt> when <tt>csq</tt> is not <tt>null</tt> behaves in exactly the
   * same way as the invocation
   *
   * <pre>
   *     out.write(csq.subSequence(start, end).toString()) </pre>
   *
   * @param csq   The character sequence from which a subsequence will be
   *              appended.  If <tt>csq</tt> is <tt>null</tt>, then characters
   *              will be appended as if <tt>csq</tt> contained the four
   *              characters <tt>"null"</tt>.
   * @param start The index of the first character in the subsequence
   * @param end   The index of the character following the last character in the
   *              subsequence
   * @return This writer
   * @throws IndexOutOfBoundsException If <tt>start</tt> or <tt>end</tt> are negative, <tt>start</tt>
   *                                   is greater than <tt>end</tt>, or <tt>end</tt> is greater than
   *                                   <tt>csq.length()</tt>
   * @throws java.io.IOException       If an I/O error occurs
   * @since 1.5
   */
  @Override
  public Writer append(CharSequence csq, int start, int end) throws IOException
  {
    buffer.append(csq, start, end);
    return this;
  }

  /**
   * Appends the specified character to this writer.
   *
   * <p> An invocation of this method of the form <tt>out.append(c)</tt>
   * behaves in exactly the same way as the invocation
   *
   * <pre>
   *     out.write(c) </pre>
   *
   * @param c The 16-bit character to append
   * @return This writer
   * @throws java.io.IOException If an I/O error occurs
   * @since 1.5
   */
  @Override
  public Writer append(char c) throws IOException
  {
    buffer.append(c);
    return this;
  }

  /**
   * Flushes the stream.  If the stream has saved any characters from the
   * various write() methods in a buffer, write them immediately to their
   * intended destination.  Then, if that destination is another character or
   * byte stream, flush it.  Thus one flush() invocation will flush all the
   * buffers in a chain of Writers and OutputStreams.
   *
   * <p> If the intended destination of this stream is an abstraction provided
   * by the underlying operating system, for example a file, then flushing the
   * stream guarantees only that bytes previously written to the stream are
   * passed to the operating system for writing; it does not guarantee that
   * they are actually written to a physical device such as a disk drive.
   *
   * @throws java.io.IOException If an I/O error occurs
   */
  @Override
  public void flush() throws IOException
  {
    // do nothing
  }

  /**
   * Closes the stream, flushing it first. Once the stream has been closed,
   * further write() or flush() invocations will cause an IOException to be
   * thrown. Closing a previously closed stream has no effect.
   *
   * @throws java.io.IOException If an I/O error occurs
   */
  @Override
  public void close() throws IOException
  {
    // nothing to do
  }

  /**
   * Get the underlying buffer.
   *
   * @return the underlying buffer holding the content of this writer
   * @see #toString()
   */
  public StringBuilder getBuffer()
  {
    return buffer;
  }

  /**
   * Returns the current content of this writer.
   * @return buffer content
   * @see #getBuffer()
   */
  @Override
  public String toString()
  {
    return buffer.toString();
  }
}
