// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;

/**
 * Class which draws an image centered to another image,
 * while taking care of deferred image loading.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ImageToImageDrawer
        implements ImageObserver
{
  /** The source image. */
  @NotNull
  private final Image sourceImage;
  /** The target image. */
  @NotNull
  private final BufferedImage targetImage;

  private final boolean doScale;

  /**
   * Constructor
   * @param sourceImage image to draw, possibly not yet completely loaded
   * @param targetImage target image to which the source image is drawn
   * @param scale       scale source image to fit to target image?
   */
  public ImageToImageDrawer(@NotNull Image sourceImage,
                            @NotNull BufferedImage targetImage,
                            boolean scale)
  {
    this.sourceImage = sourceImage;
    this.targetImage = targetImage;
    doScale = scale;
    doDraw();
  }

  /**
   * Get the source image.
   * @return source image
   */
  @NotNull
  public Image getSourceImage()
  {
    return sourceImage;
  }

  /**
   * Get the target image.
   * @return target image
   */
  @NotNull
  public BufferedImage getTargetImage()
  {
    return targetImage;
  }

  /**
   * Actually draw the image.
   */
  private void doDraw()
  {
    Graphics2D g2 = targetImage.createGraphics();
    try {
      if (doScale) {
        double scaleX = targetImage.getWidth() / (double)sourceImage.getWidth(this);
        double scaleY = targetImage.getHeight() / (double)sourceImage.getHeight(this);
        double scale = Math.min(scaleX, scaleY);
        AffineTransform at = AffineTransform.getTranslateInstance(-0.5 * sourceImage.getWidth(this),
                                                                  -0.5 * sourceImage.getHeight(this));
        at.preConcatenate(AffineTransform.getScaleInstance(scale, scale));
        at.preConcatenate(AffineTransform.getTranslateInstance(0.5 * targetImage.getWidth(),
                                                               0.5 * targetImage.getHeight()));
        g2.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
        g2.drawImage(sourceImage, at, this);
      }
      else {
        g2.drawImage(sourceImage,
                     (targetImage.getWidth() - sourceImage.getWidth(this)) / 2,
                     (targetImage.getHeight() - sourceImage.getHeight(this)) / 2,
                     this);
      }
    } finally {
      g2.dispose();
    }
  }

  /**
   * This method is called when information about an image which was
   * previously requested using an asynchronous interface becomes
   * available.  Asynchronous interfaces are method calls such as
   * getWidth(ImageObserver) and drawImage(img, x, y, ImageObserver)
   * which take an ImageObserver object as an argument.  Those methods
   * register the caller as interested either in information about
   * the overall image itself (in the case of getWidth(ImageObserver))
   * or about an output version of an image (in the case of the
   * drawImage(img, x, y, [w, h,] ImageObserver) call).
   *
   * <p>This method
   * should return true if further updates are needed or false if the
   * required information has been acquired.  The image which was being
   * tracked is passed in using the img argument.  Various constants
   * are combined to form the infoflags argument which indicates what
   * information about the image is now available.  The interpretation
   * of the x, y, width, and height arguments depends on the contents
   * of the infoflags argument.
   * <p>
   * The {@code infoflags} argument should be the bitwise inclusive
   * <b>OR</b> of the following flags: {@code WIDTH},
   * {@code HEIGHT}, {@code PROPERTIES}, {@code SOMEBITS},
   * {@code FRAMEBITS}, {@code ALLBITS}, {@code ERROR},
   * {@code ABORT}.
   *
   * @param img       the image being observed.
   * @param infoflags the bitwise inclusive OR of the following
   *                  flags:  {@code WIDTH}, {@code HEIGHT},
   *                  {@code PROPERTIES}, {@code SOMEBITS},
   *                  {@code FRAMEBITS}, {@code ALLBITS},
   *                  {@code ERROR}, {@code ABORT}.
   * @param x         the <i>x</i> coordinate.
   * @param y         the <i>y</i> coordinate.
   * @param width     the width.
   * @param height    the height.
   * @return {@code false} if the infoflags indicate that the
   * image is completely loaded; {@code true} otherwise.
   * @see #WIDTH
   * @see #HEIGHT
   * @see #PROPERTIES
   * @see #SOMEBITS
   * @see #FRAMEBITS
   * @see #ALLBITS
   * @see #ERROR
   * @see #ABORT
   * @see java.awt.Image#getWidth
   * @see java.awt.Image#getHeight
   * @see java.awt.Graphics#drawImage
   */
  @Override
  public boolean imageUpdate(Image img, int infoflags, int x, int y, int width, int height)
  {
    if ((infoflags & (ERROR | ABORT)) == 0) {
      doDraw();
    }
    return (infoflags & (ALLBITS|ABORT)) == 0;
  }
}
