// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.Predicate1;

import java.util.*;

/**
 *  This class handles <b>simple</b> glob matching:
 *  <ul>
 *    <li><tt>*</tt> stands for any number of chars (including zero)</li>
 *    <li><tt>?</tt> stands for exactly one char</li>
 *    <li><tt>[<em>AX-Z</em>]</tt> stands for characters <tt>A</tt>,
 *             <tt>X</tt>, <tt>Y</tt> and <tt>Z</tt></li>
 *    <li><tt>[^<em>AX-Z</em>]</tt> stands for all characters but <tt>A</tt>,
 *             <tt>X</tt>, <tt>Y</tt> and <tt>Z</tt></li>
 *  </ul>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class GlobMatcher
{
  private static final Matcher[] EMPTY_MATCHER_ARRAY = {};

  /** The glob pattern. */
  @NotNull
  private final String pattern;
  /** The corresponding matchers. */
  @NotNull
  private final Matcher[] matchers;

  /**
   * Constructor.
   * This will construct a case-sensitive matcher.
   * @param pattern pattern
   */
  public GlobMatcher(@NotNull String pattern)
  {
    this(pattern, false);
  }

  /**
   * Constructor.
   * @param pattern pattern
   * @param caseInsensitive make the matcher case-insensitive?
   */
  public GlobMatcher(@NotNull String pattern, boolean caseInsensitive)
  {
    this(pattern, extractMatchers(pattern, caseInsensitive));
  }

  /**
   * Internal constructor.
   * @param pattern   the pattern
   * @param matchers  the matchers associated with the given pattern
   */
  private GlobMatcher(@NotNull String pattern, @NotNull Matcher[] matchers)
  {
    this.pattern = pattern;
    this.matchers = matchers;
  }

  /**
   * Get the pattern.
   * @return the underlying pattern
   */
  @NotNull
  public String getPattern()
  {
    return pattern;
  }

  /**
   * Get the cardinality of this matcher.
   * The cardinality describes the number of different match areas.
   * @return the cardinality
   */
  public int getCardinality()
  {
    return matchers.length;
  }

  /**
   * Is this glob matcher using wildcards?
   * @return the answer
   */
  public boolean hasWildCards()
  {
    for (Matcher m : matchers) {
      if (!m.isPlain()) {
        return true;
      }
    }

    return false;
  }

  /**
   * Get this matcher as a predicate.
   * @return predicate returning {@code true} if this pattern matches
   */
  @NotNull
  public Predicate1<String> asPredicate()
  {
    return this::isMatching;
  }

  /**
   * Get a matcher if the given pattern uses globbing.
   * @param pattern the pattern
   * @param caseInsensitive make the matcher case-insensitive
   * @return a matcher for the given pattern, or {@code null} if the string does not contain wildcards
   */
  @Nullable
  public static GlobMatcher whenUsingGlobbing(@NotNull String pattern,
                                              boolean caseInsensitive)
  {
    final Matcher[] matchers = GlobMatcher.extractMatchers(pattern, caseInsensitive);
    if (matchers.length == 0) {
      return null;
    }
    for (Matcher matcher : matchers) {
      if (!matcher.isPlain()) {
        return new GlobMatcher(pattern, matchers);
      }
    }
    return null;
  }

  @NotNull
  private static Matcher[] extractMatchers(@NotNull String matchPattern,
                                           boolean caseInsensitive)
  {
    int pos = 0;
    List<Matcher> result = new LinkedList<>();
    boolean nextIsVerbatim = false;
    while (pos < matchPattern.length()) {
      char ch = matchPattern.charAt(pos);
      if (nextIsVerbatim) {
        nextIsVerbatim = false;
        result.add(caseInsensitive ? new CaseInsensitiveCharMatcher(ch) : new CharMatcher(ch));
        continue;
      }
      switch (ch) {
      case '\\':
        nextIsVerbatim = true;
        break;

      case '*':
        result.add(ASTERISK_MATCHER);
        break;

      case '?':
        result.add(QUESTIONMARK_MATCHER);
        break;

      case '[':
        int end = matchPattern.indexOf(']', pos + 1);
        if (end == -1) {
          result.add(new CharMatcher(ch));
        }
        else {
          if (end == pos + 1) {
            // allowing ] as first char in set
            end = matchPattern.indexOf(']', end + 1);
            if (end == -1) {
              break;
            }
          }
          else if (end == pos + 2  &&  matchPattern.charAt(pos + 1) == '^') {
            // allowing ] as first char in inverted set
            end = matchPattern.indexOf(']', end + 1);
            if (end == -1) {
              break;
            }
          }

          String set = matchPattern.substring(pos+1, end);
          pos = end;
          result.add(set.startsWith("^") ?
                             new InvertedSetMatcher(set.substring(1), caseInsensitive) :
                             new SetMatcher(set, caseInsensitive));
        }
        break;

      default:
        result.add(caseInsensitive
                           ? new CaseInsensitiveCharMatcher(ch)
                           : new CharMatcher(ch));
        break;
      }
      ++pos;
    }
    return result.toArray(EMPTY_MATCHER_ARRAY);
  }

  /**
   *  Is a given string matching this pattern?
   *  @param str the string
   *  @return {@code true} if the string is matching, {@code false} otherwise
   */
  public boolean isMatching(@NotNull String str)
  {
    return isMatching(str, 0);
  }

  /**
   *  Recursive match test.
   *  @param str     current string
   *  @param matcher index of matcher
   *  @return is the matcher and all following matching?
   */
  private boolean isMatching(@NotNull String str, int matcher)
  {
    if (matcher == matchers.length) {
      return str.isEmpty();
    }
    String[] result = matchers[matcher].getMatches(str);
    if (result == null) {
      return false;
    }
    for (int r = result.length - 1;  r >= 0;  --r) {
      if (isMatching(result[r], matcher + 1)) {
        return true;
      }
    }
    return false;
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) return true;
    if (!(o instanceof GlobMatcher)) return false;
    return Objects.equals(pattern, ((GlobMatcher)o).pattern);
  }

  @Override
  public int hashCode()
  {
    return Objects.hashCode(pattern);
  }

  /**
   * Match a character.
   */
  private interface Matcher
  {
    /** No match. */
    String[] NO_MATCH = null;

    /**
     *  Get the matches of a string.
     *  @param str String to match
     *  @return the answer
     */
    @Nullable
    String[] getMatches(@NotNull String str);

    /**
     * Is this matcher using plain string compare?
     * @return the answer
     */
    default boolean isPlain()
    {
      return false;
    }
  }

  /**
   *  A matcher which just matches a simple character.
   */
  private static class CharMatcher
          implements Matcher
  {
    /** The char to match. */
    private final char ch;

    /**
     * Constructor.
     * @param ch   char to match
     */
    private CharMatcher(char ch)
    {
      this.ch = ch;
    }

    /**
     * Get the matches of a string.
     *
     * @param str String to match
     * @return the answer
     */
    @Override
    @Nullable
    public String[] getMatches(@NotNull String str)
    {
      return !str.isEmpty() &&  str.charAt(0) == ch
              ? new String[] {
              str.substring(1)
      }
              : NO_MATCH;
    }

    @Override
    public boolean isPlain()
    {
      return true;
    }
  }

  /**
   *  A matcher which just matches a simple character.
   */
  private static class CaseInsensitiveCharMatcher
          implements Matcher
  {
    /** The char to match. */
    private final String ch;

    /**
     * Constructor.
     * @param ch   char to match
     */
    private CaseInsensitiveCharMatcher(char ch)
    {
      this.ch = Character.toString(ch);
    }

    /**
     * Get the matches of a string.
     *
     * @param str String to match
     * @return the answer
     */
    @Override
    @Nullable
    public String[] getMatches(@NotNull String str)
    {
      return !str.isEmpty() &&  ch.equalsIgnoreCase(str.substring(0, 1))
              ? new String[] {
              str.substring(1)
      }
              : NO_MATCH;
    }

    @Override
    public boolean isPlain()
    {
      return true;
    }
  }

  /** An asterisk matches any number of characters. */
  private static final Matcher ASTERISK_MATCHER = new Matcher()
  {
    /**
     * Get the matches of a string.
     *
     * @param str String to match
     * @return the answer
     */
    @Override
    public String[] getMatches(@NotNull String str)
    {
      final String[] result = new String[str.length() + 1];
      for (int l = 0;  l < result.length;  ++l) {
        result[l] = str.substring(l);
      }
      return result;
    }
  };

  /** A question mark matches any single characters. */
  private static final Matcher QUESTIONMARK_MATCHER = new Matcher()
  {
    /**
     * Get the matches of a string.
     *
     * @param str String to match
     * @return the answer
     */
    @Override
    @Nullable
    public String[] getMatches(@NotNull String str)
    {
      if (!str.isEmpty()) {
        return new String[] {
                str.substring(1)
        };
      }
      return NO_MATCH;
    }
  };

  /** A set matcher matches a given set of characters. */
  private static class SetMatcher
          implements Matcher
  {
    /** The characters which are matched. */
    @NotNull
    private final Set<Character> matching = new HashSet<>();

    /**
     * Constructor.
     * @param setDefinition set definition without enclosing brackets.
     * @param caseInsensitive use case-insensitive compare
     */
    private SetMatcher(@NotNull String setDefinition, boolean caseInsensitive)
    {
      char lastChar = '\0';
      for (int i = 0;  i < setDefinition.length();  ++i) {
        char ch = setDefinition.charAt(i);
        if (ch == '-') {
          if (matching.isEmpty()) {
            if (caseInsensitive) {
              addCaseInsensitive(ch);
            }
            else {
              matching.add(ch);
            }
          }
          else {
            if (++i == setDefinition.length()) {
              throw new IllegalArgumentException("Not a valid glob set: "+setDefinition);
            }
            final char startChar = lastChar;
            ch = setDefinition.charAt(i);
            if (ch <= lastChar) {
              throw new IllegalArgumentException("Not a valid glob set: "+setDefinition);
            }
            if (caseInsensitive) {
              for (char c = (char) (startChar + 1);  c <= ch; ++c) {
                addCaseInsensitive(c);
              }
            }
            else {
              for (char c = (char) (startChar + 1); c <= ch; ++c) {
                matching.add(c);
              }
            }
            lastChar = ch;
          }
        }
        else {
          if (caseInsensitive) {
            addCaseInsensitive(ch);
          }
          else {
            matching.add(ch);
          }
          lastChar = ch;
        }
      }
    }

    private void addCaseInsensitive(char ch)
    {
      matching.add(ch);
      matching.add(Character.toUpperCase(ch));
      matching.add(Character.toLowerCase(ch));
      matching.add(Character.toTitleCase(ch));
    }

    /**
     * Get the matches of a string.
     *
     * @param str String to match
     * @return the answer
     */
    @Override
    @Nullable
    public String[] getMatches(@NotNull String str)
    {
      return !str.isEmpty() &&  isMatching(str.charAt(0))  ?
              new String[] {
                      str.substring(1)
              }  :
              NO_MATCH;
    }

    /**
     *  Is a character matching?
     *  @param ch character
     *  @return the answer
     */
    public boolean isMatching(char ch)
    {
      return matching.contains(ch);
    }
  }

  /**
   *  Matcher for characters not in a set.
   */
  private static class InvertedSetMatcher
          extends SetMatcher
  {

    /**
     * Constructor.
     * @param setDefinition  set definition without enclosing brackets and inverter sign
     * @param caseInsensitive use case insensitive comparison?
     */
    private InvertedSetMatcher(@NotNull String setDefinition, boolean caseInsensitive)
    {
      super(setDefinition, caseInsensitive);
    }

    /**
     * Get the matches of a string.
     *
     * @param str String to match
     * @return the answer
     */
    @Nullable
    @Override
    public String[] getMatches(@NotNull String str)
    {
      return !str.isEmpty() &&  !isMatching(str.charAt(0))  ?
              new String[] {
                      str.substring(1)
              }  :
              NO_MATCH;
    }
  }

  /**
   * Returns a string representation of the object.
   * @return a string representation of the object.
   */
  @Override
  public String toString()
  {
    return getClass().getName()+"("+pattern+")";
  }

  private static class Tester
  {
    private static class Test
    {
      @NotNull
      private final String testString;
      private final boolean result;

      private Test(@NotNull String testString, boolean result)
      {
        this.testString = testString;
        this.result = result;
      }

      @NotNull
      public String getTestString()
      {
        return testString;
      }

      public boolean getResult()
      {
        return result;
      }

      public void check(@NotNull GlobMatcher globMatcher)
      {
        if (globMatcher.isMatching(testString) != result) {
          System.out.println("FAILED:\t"+globMatcher+"\t'"+testString+'\''+"\tExpected: "+result);
        }
      }
    }
    private final GlobMatcher matcher;
    private final Test[]      tests;

    private Tester(@NotNull String pattern, @NotNull Test[] tests)
    {
      this.matcher = new GlobMatcher(pattern);
      this.tests = tests;
    }

    public void runTests()
    {
      for (Test test : tests) {
        test.check(matcher);
      }
    }
  }

  public static void main(@NotNull String[] args)
  {
    if (args.length > 2) {
      // first arg is the pattern, further args are test strings
      final GlobMatcher matcher = new GlobMatcher(args[0]);
      for (int a = 1;  a < args.length;  ++a) {
        System.out.println(args[a]+"\t"+
                           (matcher.isMatching(args[a]) ? "MATCH" : "MISS"));
      }
    }
    else {
      final Tester[] tester = {
              new Tester("A*A", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AA", true),
                      new Tester.Test("ABA", true),
                      new Tester.Test("AAB", false),
                      new Tester.Test("BAA", false),
                      new Tester.Test("AAA", true),
                      new Tester.Test("AAABBBBBBBBBBBBBBBBBBBBBBBBB", false),
                      new Tester.Test("AAABBBBBBBBBBBBBBBBBBBBBBBBBA", true)
              }),
              new Tester("A?C", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("AC", false),
                      new Tester.Test("ABC", true),
                      new Tester.Test("AABC", false),
                      new Tester.Test("ABCC", false),
                      new Tester.Test("ABBC", false),
                      }),
              new Tester("A[BC]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", true),
                      new Tester.Test("ACD", true),
                      }),
              new Tester("A[^BC]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", false),
                      new Tester.Test("ACD", false),
                      new Tester.Test("AED", true),
                      }),
              new Tester("A[B^C]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", true),
                      new Tester.Test("ACD", true),
                      new Tester.Test("AED", false),
                      new Tester.Test("A^D", true),
                      }),
              new Tester("A[]BC]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", true),
                      new Tester.Test("ACD", true),
                      new Tester.Test("AED", false),
                      new Tester.Test("A]D", true),
                      new Tester.Test("A[D", false),
                      }),
              new Tester("A[^]BC]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", false),
                      new Tester.Test("ACD", false),
                      new Tester.Test("AED", true),
                      new Tester.Test("A]D", false),
                      new Tester.Test("A[D", true),
                      new Tester.Test("A^D", true),
                      }),
              new Tester("A[^]^BC]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", false),
                      new Tester.Test("ACD", false),
                      new Tester.Test("AED", true),
                      new Tester.Test("A]D", false),
                      new Tester.Test("A[D", true),
                      new Tester.Test("A^D", false),
                      }),
              new Tester("A[^^BC]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", false),
                      new Tester.Test("ACD", false),
                      new Tester.Test("AED", true),
                      new Tester.Test("A]D", true),
                      new Tester.Test("A^D", false),
                      }),
              new Tester("A[?*BC]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", true),
                      new Tester.Test("ACD", true),
                      new Tester.Test("AED", false),
                      new Tester.Test("A?D", true),
                      new Tester.Test("A*D", true),
                      new Tester.Test("A^D", false),
                      }),
              new Tester("A[^?*BC]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", false),
                      new Tester.Test("ACD", false),
                      new Tester.Test("AED", true),
                      new Tester.Test("A?D", false),
                      new Tester.Test("A*D", false),
                      new Tester.Test("A^D", true),
                      }),
              new Tester("A[-?*BC]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", true),
                      new Tester.Test("ACD", true),
                      new Tester.Test("AED", false),
                      new Tester.Test("A?D", true),
                      new Tester.Test("A*D", true),
                      new Tester.Test("A^D", false),
                      new Tester.Test("A-D", true),
                      }),
              new Tester("A[^-?*BC]D", new Tester.Test[] {
                      new Tester.Test("", false),
                      new Tester.Test("A", false),
                      new Tester.Test("AB", false),
                      new Tester.Test("ABC", false),
                      new Tester.Test("ABCD", false),
                      new Tester.Test("ABD", false),
                      new Tester.Test("ACD", false),
                      new Tester.Test("AED", true),
                      new Tester.Test("A?D", false),
                      new Tester.Test("A*D", false),
                      new Tester.Test("A^D", true),
                      new Tester.Test("A-D", false),
                      }),
              };
      for (Tester aTester : tester) {
        aTester.runTests();
      }
    }
  }
}
