// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Types;
import de.caff.generics.function.FragileProcedure1;
import de.caff.generics.function.Predicate1;
import de.caff.util.debug.Debug;

import java.io.*;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.regex.Pattern;

/**
 * Helper tools for handling files.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since March 05, 2019
 */
public final class FileTool
{
  /** Sort directories first.
   * <p>
   * Ignores names.
   * <p>
   * Use {@link Comparator#thenComparing(Comparator)} with
   * {@link #BY_NAME_CASE_INSENSITIVE} or
   * {@link #BY_NAME_CASE_SENSITIVE} to get directories
   * first, while names are ordered inside each group.
   * <p>
   * Use {@link Comparator#reversed()} for the reverse of this.
   */
  public static final Comparator<File> DIRECTORIES_FIRST =
          (f1, f2) -> f1.isDirectory() ^ f2.isDirectory()
                  ? (f1.isDirectory() ? -1 : 1)
                  : 0;

  /**
   * Sort by name (case insensitive).
   * <p>
   * Directories and files are mixed.
   * <p>
   * Use {@link Comparator#reversed()} to reverse this order.
   *
   * @see #DIRECTORIES_FIRST
   */
  public static final Comparator<File> BY_NAME_CASE_INSENSITIVE =
          (f1, f2) -> {
            final int result = f1.getName().compareToIgnoreCase(f2.getName());
            // Note:
            // File objects having names which only differ in case usually¹
            // still refer to two different files, so don't return 0 in
            // that case.
            //
            // ¹ at least when used during the walkTree() methods
            return result == 0
                    ?  f1.getName().compareTo(f2.getName())
                    :  result;
          };

  /**
   * Sort by name (case sensitive).
   * <p>
   * Directories and files are mixed.
   * <p>
   * Use {@link Comparator#reversed()} to reverse this order.
   *
   * @see #DIRECTORIES_FIRST
   */
  public static final Comparator<File> BY_NAME_CASE_SENSITIVE = Comparator.comparing(File::getName);

  /** Size of compare buffer for two files. */
  private static final int FILE_COMPARE_BUFFER_SIZE = 1 << 16;
  /** Return value of {@link #firstDiff(File, File)} when both contents are equal. */
  public static final long NO_DIFFERENCE = -1L;

  /** Don't construct. */
  private FileTool() {}

  /**
   * Walk a file tree.
   * This walks through the tree in depth-first order.
   * Symbolic links are not handled specially.
   * Files in a directory are encountered in 'natural' order.
   *
   * @param file         file to start with, usually a directory
   * @param stepIntoDir  Predicate allowing to avoid directories.
   *                     Always called with a directory, i.e.
   *                     {@code file.isDirectory()} is always {@code true}.
   * @param fileHandler  Handler called for each file.
   *                     File is never a directory, i.e.
   *                     {@code file.isDirectory()} is always {@code false}.
   */
  public static void walkTree(@NotNull File file,
                              @NotNull Predicate<? super File> stepIntoDir,
                              @NotNull Consumer<? super File> fileHandler)
  {
    if (!file.exists()) {
      Debug.error("walkTree() called w/ non-existent file: %0", file);
      return;
    }
    if (file.isDirectory()) {
      if (stepIntoDir.test(file)) {
        final File[] subFiles = file.listFiles();
        if (subFiles != null) {
          for (File subFile : subFiles) {
            walkTree(subFile, stepIntoDir, fileHandler);
          }
        }
      }
    }
    else {
      fileHandler.accept(file);
    }
  }

  /**
   * Walk a file tree.
   * This walks through the tree in depth first order.
   * Symbolic links are not handled specially.
   * This steps into every directory found.
   * Files in a directory are encountered in 'natural' order.
   *
   * @param file         file to start with, usually a directory
   * @param fileHandler  Handler called for each file.
   *                     File is never a directory, i.e.
   *                     {@code file.isDirectory()} is always {@code false}.
   */
  public static void walkTree(@NotNull File file,
                              @NotNull Consumer<? super File> fileHandler)
  {
    walkTree(file, Predicate1.alwaysTrue(), fileHandler);
  }

  /**
   * Walk a file tree in a defined order.
   * This walks through the tree in depth first order.
   * Symbolic links are not handled specially.
   * Files in a directory are encountered in the defined order.
   *
   * @param file         file to start with, usually a directory
   * @param stepIntoDir  Predicate allowing to avoid directories.
   *                     Always called with a directory, i.e.
   *                     {@code file.isDirectory()} is always {@code true}.
   * @param fileHandler  Handler called for each file.
   *                     File is never a directory, i.e.
   *                     {@code file.isDirectory()} is always {@code false}.
   * @param order        comparator defining the order in which files inside a
   *                     directory are handled. Common orders can be defined
   *                     by combining the comparators {@link #DIRECTORIES_FIRST},
   *                     {@link #BY_NAME_CASE_INSENSITIVE}, and
   *                     {@link #BY_NAME_CASE_SENSITIVE}.
   */
  public static void walkTreeOrdered(@NotNull File file,
                                     @NotNull Predicate<? super File> stepIntoDir,
                                     @NotNull Consumer<? super File> fileHandler,
                                     @NotNull Comparator<? super File> order)
  {
    if (!file.exists()) {
      Debug.error("walkTree() called w/ non-existent file: %0", file);
      return;
    }
    if (file.isDirectory()) {
      if (stepIntoDir.test(file)) {
        final File[] subFiles = file.listFiles();
        if (subFiles != null) {
          Arrays.sort(subFiles, order);
          for (File subFile : subFiles) {
            walkTree(subFile, stepIntoDir, fileHandler);
          }
        }
      }
    }
    else {
      fileHandler.accept(file);
    }
  }

  /**
   * Walk a file tree in a defined order.
   * This walks through the tree in depth first order.
   * Symbolic links are not handled specially.
   * This steps into every directory found.
   * Files in a directory are encountered in 'natural' order.
   *
   * @param file         file to start with, usually a directory
   * @param fileHandler  Handler called for each file.
   *                     File is never a directory, i.e.
   *                     {@code file.isDirectory()} is always {@code false}.
   * @param order        comparator defining the order in which files inside a
   *                     directory are handled. Common orders can be defined
   *                     by combining the comparators {@link #DIRECTORIES_FIRST},
   *                     {@link #BY_NAME_CASE_INSENSITIVE}, and
   *                     {@link #BY_NAME_CASE_SENSITIVE}.
   */
  public static void walkTreeOrdered(@NotNull File file,
                                     @NotNull Consumer<? super File> fileHandler,
                                     @NotNull Comparator<? super File> order)
  {
    walkTreeOrdered(file, Predicate1.alwaysTrue(), fileHandler, order);
  }

  /**
   * Walk a file tree and apply a consumer which might throw an exception.
   * This walks through the tree in depth-first order.
   * This allows to exclude or react on directories.
   * Symbolic links are not handled specially.
   * Files in a directory are encountered in 'natural' order.
   *
   * @param file         file to start with, usually a directory
   * @param stepIntoDir  Predicate allowing to avoid directories.
   *                     Always called with a directory, i.e.
   *                     {@code file.isDirectory()} is always {@code true},
   *                     before stepping into that directory.
   * @param fileHandler  Handler called for each file.
   *                     File is never a directory, i.e.
   *                     {@code file.isDirectory()} is always {@code false}.
   * @param <E> exception thrown by fragile procedure, forwarded by this method
   * @throws E when {@code fileHandler} throws it
   */
  public static <E extends Exception> void walkTreeFragile(@NotNull File file,
                                                           @NotNull Predicate<File> stepIntoDir,
                                                           @NotNull FragileProcedure1<E, ? super File> fileHandler)
          throws E
  {
    if (!file.exists()) {
      Debug.error("walkTreeFragile() called w/ non-existent file: %0", file);
      return;
    }
    if (file.isDirectory()) {
      if (stepIntoDir.test(file)) {
        final File[] subFiles = file.listFiles();
        if (subFiles != null) {
          for (File subFile : subFiles) {
            walkTreeFragile(subFile, stepIntoDir, fileHandler);
          }
        }
      }
    }
    else {
      fileHandler.apply(file);
    }
  }

  /**
   * Walk a file tree and apply a consumer which might throw an exception.
   * This walks through the tree in depth-first order.
   * Symbolic links are not handled specially.
   * Files in a directory are encountered in 'natural' order.
   *
   * @param file         file to start with, usually a directory
   * @param fileHandler  Handler called for each file.
   *                     File is never a directory, i.e.
   *                     {@code file.isDirectory()} is always {@code false}.
   * @param <E> exception thrown by fragile procedure, forwarded by this method
   * @throws E when {@code fileHandler} throws it
   */
  public static <E extends Exception> void walkTreeFragile(@NotNull File file,
                                                           @NotNull FragileProcedure1<E, ? super File> fileHandler)
          throws E
  {
    walkTreeFragile(file, Predicate1.alwaysTrue(), fileHandler);
  }

  /**
   * Get the extension of the file name.
   * @param file file with a possible extension
   * @return everything after the last dot of the filename, possibly empty it
   *         the file ends with a dot, or {@code null} if the name does not contain
   *         any dots
   */
  @Nullable
  public static String getExtension(@NotNull File file)
  {
    return getExtension(file.getName());
  }

  /**
   * Get the extension of the file name.
   * @param fileName file name with a possible extension
   * @return everything after the last dot of the filename, possibly empty it
   *         the file ends with a dot, or {@code null} if the name does not contain
   *         any dots
   */
  @Nullable
  public static String getExtension(@NotNull String fileName)
  {
    final int lastDot = fileName.lastIndexOf('.');
    return lastDot >= 0 ? fileName.substring(lastDot + 1) : null;
  }

  /**
   * Is the content of two files equal?
   * @param file1 first file
   * @param file2 second file
   * @return {@code true} if both files have the same content<br>
   *         {@code false} if both files have different content
   * @throws IOException on file open or read errors
   */
  public static boolean contentEquals(@NotNull File file1,
                                      @NotNull File file2)
          throws IOException
  {
    if (file1.length() != file2.length()) {
      return false;
    }
    return firstDiff(file1, file2) == NO_DIFFERENCE;
  }

  /**
   * Is the content of two files equal?
   * @param file1 first file
   * @param file2 second file
   * @return {@code true} if both files have the same content<br>
   *         {@code false} if both files have different content
   * @throws IOException on file open or read errors
   */
  public static boolean contentEquals(@NotNull Path file1,
                                      @NotNull Path file2)
          throws IOException
  {
    return contentEquals(file1.toFile(), file2.toFile());
  }

  /**
   * Is the content of two files equal?
   * @param filePath1 path of first file
   * @param filePath2 path of second file
   * @return {@code true} if both files have the same content<br>
   *         {@code false} if both files have different content
   * @throws IOException on file open or read errors
   */
  public static boolean contentEquals(@NotNull String filePath1,
                                      @NotNull String filePath2)
          throws IOException
  {
    return contentEquals(new File(filePath1),
                         new File(filePath2));
  }

  /**
   * Get the position of the first difference of the content of two files.
   * @param file1 first file
   * @param file2 second file
   * @return first position where both files differ, or {@link #NO_DIFFERENCE} if both files have the same content
   * @throws IOException on file open or read errors
   */
  public static long firstDiff(@NotNull File file1,
                               @NotNull File file2)
          throws IOException
  {
    if (file1.equals(file2)) {
      return NO_DIFFERENCE;
    }

    try (InputStream is1 = Files.newInputStream(file1.toPath())) {
      try (InputStream is2 = Files.newInputStream(file2.toPath())) {
        final byte[] buffer1 = new byte[FILE_COMPARE_BUFFER_SIZE];
        final byte[] buffer2 = new byte[FILE_COMPARE_BUFFER_SIZE];
        long pos = 0;
        while (true) {
          final int bytesRead1 = Math.max(is1.read(buffer1), 0);
          final int bytesRead2 = Math.max(is2.read(buffer2), 0);
          final int minLen = Math.min(bytesRead1, bytesRead2);
          final int diffPos = firstDiff(buffer1,
                                        buffer2,
                                        minLen);
          if (diffPos >= 0) {
            return pos + diffPos;
          }
          pos += minLen;
          if (bytesRead1 != bytesRead2) {
            return pos;
          }
          if (minLen == 0) {
            return NO_DIFFERENCE;
          }
        }
      }
    }
  }

  /**
   * Get the position of the first difference of the content of two files.
   * @param filePath1 path of first file
   * @param filePath2 path of second file
   * @return first position where both files differ, or {@link #NO_DIFFERENCE} if both files have the same content
   * @throws IOException on file open or read errors
   */
  public static long firstDiff(@NotNull String filePath1,
                               @NotNull String filePath2)
          throws IOException
  {
    return firstDiff(new File(filePath1),
                     new File(filePath2));
  }

  /**
   * Find the first difference in two byte arrays.
   * @param data1 first byte array
   * @param data2 second byte array
   * @param searchLen length to search through the arrays
   * @return position if first difference found, or {@code -1} if there was no difference
   */
  private static int firstDiff(@NotNull byte[] data1,
                               @NotNull byte[] data2,
                               final int searchLen)
  {
    for (int i = 0;  i < searchLen;  ++i) {
      if (data1[i] != data2[i]) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Load a file into memory.
   * @param file file to load
   * @return byte array with file content
   * @throws IOException on read errors or if the file is too large to fit into a byte array
   * @see #load(String) 
   */
  @NotNull
  public static byte[] load(@NotNull File file)
          throws IOException
  {
    final long length = file.length();
    if (length > Integer.MAX_VALUE) {
      throw new IOException("File too large: "+length);
    }
    try (InputStream fos = Files.newInputStream(file.toPath())) {
      final byte[] content = new byte[(int)length];
      IOUtil.readFully(fos, content);
      return content;
    }
  }

  /**
   * Load a file into memory.
   * @param file file to load
   * @return byte array with file content
   * @throws IOException on read errors or if the file is too large to fit into a byte array
   * @see #load(File) 
   */
  @NotNull
  public static byte[] load(@NotNull String file)
          throws IOException
  {
    return load(new File(file));
  }

  /**
   * Store the given bytes to a file.
   * There are no safety belts, the file will ve overwritten if it exists.
   * @param file file to write to
   * @param data data to write
   * @throws IOException on write errors
   */
  public static void store(@NotNull File file,
                           @NotNull byte[] data)
    throws IOException
  {
    try (OutputStream fos = Files.newOutputStream(file.toPath())) {
      fos.write(data);
    }
  }

  /**
   * Store the given bytes to a file.
   * There are no safety belts, the file will ve overwritten if it exists.
   * @param file file to write to
   * @param data data to write
   * @param offset start offset into the byte array
   * @param length number of bytes to be written
   * @throws IOException on write errors
   */
  public static void store(@NotNull File file,
                           @NotNull byte[] data,
                           int offset,
                           int length)
    throws IOException
  {
    if (offset < 0) {
      throw new IllegalArgumentException("Invalid negative offset: "+offset);
    }
    if (length < 0) {
      throw new IllegalArgumentException("Invalid negative length: "+length);
    }
    if (offset + length > data.length) {
      throw new IllegalArgumentException("Requested bytes will not fit into array!");
    }
    try (OutputStream fos = Files.newOutputStream(file.toPath())) {
      fos.write(data, offset, length);
    }
  }

  /**
   * Store the given bytes to a file.
   * There are no safety belts, the file will ve overwritten if it exists.
   * @param file file to write to
   * @param data data to write
   * @throws IOException on write errors
   */
  public static void store(@NotNull String file,
                           @NotNull byte[] data)
    throws IOException
  {
    store(new File(file), data);
  }

  /**
   * Store the given bytes to a file.
   * There are no safety belts, the file will ve overwritten if it exists.
   * @param file file to write to
   * @param data data to write
   * @param offset start offset into the byte array
   * @param length number of bytes to be written
   * @throws IOException on write errors
   */
  public static void store(@NotNull String file,
                           @NotNull byte[] data,
                           int offset,
                           int length)
    throws IOException
  {
    store(new File(file), data, offset, length);
  }

  /**
   * Append the given bytes to a file.
   * @param file file to append to
   * @param data data to append
   * @throws IOException on write errors
   */
  public static void append(@NotNull File file,
                            @NotNull byte[] data)
    throws IOException
  {
    try (OutputStream fos = Files.newOutputStream(file.toPath(), StandardOpenOption.APPEND, StandardOpenOption.CREATE)) {
      fos.write(data);
    }
  }

  /**
   * Append the given bytes to a file.
   * @param file file to append to
   * @param data data to append
   * @param offset start offset into the byte array
   * @param length number of bytes to be written
   * @throws IOException on write errors
   */
  public static void append(@NotNull File file,
                            @NotNull byte[] data,
                            int offset,
                            int length)
    throws IOException
  {
    if (offset < 0) {
      throw new IllegalArgumentException("Invalid negative offset: "+offset);
    }
    if (length < 0) {
      throw new IllegalArgumentException("Invalid negative length: "+length);
    }
    if (offset + length > data.length) {
      throw new IllegalArgumentException("Requested bytes will not fit into array!");
    }
    try (OutputStream fos = Files.newOutputStream(file.toPath(), StandardOpenOption.APPEND, StandardOpenOption.CREATE)) {
      fos.write(data, offset, length);
    }
  }

  /**
   * Append the given bytes to a file.
   * @param file file to append to
   * @param data data to append
   * @throws IOException on write errors
   */
  public static void append(@NotNull String file,
                            @NotNull byte[] data)
    throws IOException
  {
    append(new File(file), data);
  }

  /**
   * Append the given bytes to a file.
   * @param file file to append to
   * @param data data to append
   * @param offset the start offset into the byte array
   * @param length number of bytes to be written
   * @throws IOException on write errors
   */
  public static void append(@NotNull String file,
                            @NotNull byte[] data,
                            int offset,
                            int length)
    throws IOException
  {
    append(new File(file), data, offset, length);
  }

  /**
   * Delete a file or file tree.
   * <p>
   * If removing fails for whatever reason, further processing is stop
   * and {@code false} returned.
   *
   * @param file file or directory
   * @return {@code true} if deletion is complete<br>
   *         {@code false} if deletion of the file or the tree (partially) failed
   */
  public static boolean deleteTree(@NotNull File file)
  {
    cleanup(file);
    return file.delete();
  }

  /**
   * Cleanup a directory by removing its content.
   * If {@code dir} is not a directory, nothing will be deleted.
   * If removing fails for whatever reason, further processing is stop
   * and {@code false} returned.
   * @param dir directory
   * @return {@code true} if cleanup is complete<br>
   *         {@code false} if cleanup of the directory (partially) failed
   */
  public static boolean cleanup(@NotNull File dir)
  {
    if (dir.isDirectory()) {
      final File[] files = dir.listFiles();
      if (files != null) {
        for (File f : files) {
          if (!deleteTree(f)) {
            return false;
          }
        }
      }
    }
    return true;
  }

  /**
   * Append a complete path to a file.
   * This method expects the system's default as a file separator.
   * @param dir     directory where paths are
   * @param subPath sub path, has to be relative
   * @return evaluated file/directory
   */
  @NotNull
  public static File multiExpand(@NotNull File dir,
                                 @NotNull String subPath)
  {
    return multiExpand(dir, File.separatorChar, subPath);
  }

  /**
   * Append a complete path to a file.
   * @param dir           directory where paths are
   * @param fileSeparator file separator char, usually either {@code '/'} or {@code `\\`}
   * @param subPath       sub path, has to be relative
   * @return evaluated file/directory
   */
  public static File multiExpand(@NotNull File dir,
                                 char fileSeparator,
                                 @NotNull String subPath)
  {
    final String sep = Character.toString(fileSeparator);
    if (subPath.startsWith(sep)) {
      throw new IllegalArgumentException("Cannot expand with absolute subPath: "+subPath);
    }
    return expand(dir, subPath.split(Pattern.quote(sep)));
  }

  /**
   * Append multiple path items to a file.
   * @param dir           directory where paths are
   * @param subPath       The sub path. User has to take care that the sub paths are valid
   *                      file names and don't contain separator characters.
   * @return evaluated file/directory
   */
  public static File expand(@NotNull File dir,
                            @NotNull String ... subPath)
  {
    File result = dir;
    for (String sub : subPath) {
      switch (sub) {
      case ".":
        break;

      case "..":
        result = result.getParentFile();
        break;

      default:
        result = new File(result, sub);
        break;
      }
    }
    return result;
  }

  @NotNull
  public static List<File> performGlobbing(@NotNull File fileWithPossibleGlobbing)
          throws IOException
  {
    String pathWithPossibleGlobbing = fileWithPossibleGlobbing.toString();
    if (GlobMatcher.whenUsingGlobbing(pathWithPossibleGlobbing, true) == null) {
      // no globbing used
      if (fileWithPossibleGlobbing.exists()) {
        return Collections.singletonList(fileWithPossibleGlobbing);
      }
      return Collections.emptyList();
    }

    if (Utility.areWeOnWindows()) {
      pathWithPossibleGlobbing = pathWithPossibleGlobbing.replace('\\', '/'); // PatchMatcher will work fine
    }
    final PathMatcher matcher = FileSystems.getDefault().getPathMatcher("glob:" + pathWithPossibleGlobbing);
    // need a useful start
    final List<String> parts = Arrays.asList(pathWithPossibleGlobbing.split(Pattern.quote("/")));
    int used = 1;
    while (GlobMatcher.whenUsingGlobbing(Types.join('/', parts.subList(0, used)), true) == null) {
      ++used;
    }

    final Set<Path> paths = new HashSet<>();
    final Path searchRoot = Paths.get(Types.join('/', parts.subList(0, used - 1)));
    Files.walkFileTree(searchRoot,
            new SimpleFileVisitor<Path>() {
              @Override
              public FileVisitResult visitFile(Path path, BasicFileAttributes basicFileAttributes) throws IOException {
                if (matcher.matches(path)) {
                  paths.add(path);
                }
                return super.visitFile(path, basicFileAttributes);
              }

              @Override
              public FileVisitResult visitFileFailed(Path path, IOException e) throws IOException {
                return FileVisitResult.CONTINUE;
              }

              @Override
              public FileVisitResult postVisitDirectory(Path path, IOException e) throws IOException {
                if (e != null) {
                  throw e;
                }
                if (matcher.matches(path)) {
                  paths.add(path);
                }
                return FileVisitResult.CONTINUE;
              }
            });

    return Types.map(paths, Path::toFile);
  }
}
