// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.generics.SynchronizedCollection;

import java.awt.*;
import java.util.Collection;
import java.util.LinkedList;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 *  A wrapper for the event queue which catches all uncaught exceptions
 *  and distributes them to {@link EventQueueExceptionListener}s.
 *  This should only be used as a singleton, and it is indirectly used by
 *  the {@link Utility#addEventQueueExceptionListener(EventQueueExceptionListener)}
 *  method.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class EventQueueExceptionWrapper
        extends EventQueue
{
  /** The {@link EventQueueExceptionListener}s. */
  private final Collection<EventQueueExceptionListener> listeners = new SynchronizedCollection<>(new LinkedList<EventQueueExceptionListener>());
  private final AtomicBoolean recursionGuard = new AtomicBoolean(false);

  /**
   *  Constructor.
   */
  public EventQueueExceptionWrapper()
  {
    Toolkit.getDefaultToolkit().getSystemEventQueue().push(this);
  }

  /**
   * Dispatches an event. The manner in which the event is
   * dispatched depends upon the type of the event and the
   * type of the event's source object:
   * <table border=1 summary="Event types, source types, and dispatch methods">
   * <tr>
   * <th>Event Type</th>
   * <th>Source Type</th>
   * <th>Dispatched To</th>
   * </tr>
   * <tr>
   * <td>ActiveEvent</td>
   * <td>Any</td>
   * <td>event.dispatch()</td>
   * </tr>
   * <tr>
   * <td>Other</td>
   * <td>Component</td>
   * <td>source.dispatchEvent(AWTEvent)</td>
   * </tr>
   * <tr>
   * <td>Other</td>
   * <td>MenuComponent</td>
   * <td>source.dispatchEvent(AWTEvent)</td>
   * </tr>
   * <tr>
   * <td>Other</td>
   * <td>No action (ignored)</td>
   * </tr>
   * </table>
   *
   * @param event an instance of {@code java.awt.AWTEvent},
   *              or a subclass of it
   */
  @Override
  protected void dispatchEvent(AWTEvent event)
  {
    try {
      super.dispatchEvent(event);
    } catch (Throwable t) {
      if (!recursionGuard.getAndSet(true)) {
        for (EventQueueExceptionListener listener : listeners) {
          listener.exceptionOccured(event, t);
        }
        recursionGuard.set(false);
      }
    }
  }

  /**
   *  Add an exception listener which is called when an exception occurs during the
   *  dispatch of an AWT event.
   *  @param listener listener to add
   */
  public void addEventQueueExceptionListener(EventQueueExceptionListener listener)
  {
    synchronized (listeners) {
      listeners.add(listener);
    }
  }

  /**
   *  Remove an exception listener which was called when an exception occurs during the
   *  dispatch of an AWT event.
   *  @param listener listener to remove
   */
  public void removeEventQueueExceptionListener(EventQueueExceptionListener listener)
  {
    synchronized (listeners) {
      listeners.remove(listener);
    }
  }
}
