// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;

import java.util.*;
import java.util.function.BiPredicate;
import java.util.function.Consumer;
import java.util.function.Predicate;

/**
 * Type safe bit flags.
 * This class is immutable.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <E> the enum bit flag type for which this mask is valid
 */
public final class EnumBitMask<E extends Enum<E> & BitFlag>
{
  /** Key mask for excluded keys in 32bit cache. */
  private static final int EXCLUDE_MASK32 = 0xFFFFFFFF << Utility.getIntParameter("ebm.mask32.shift",
                                                                                  16);
  /** The underlying simple bit mask. */
  @NotNull
  private final BitMask bitMask;

  /**
   * Constructor.
   * @param bitMask initial flags
   */
  public EnumBitMask(@NotNull BitMask bitMask)
  {
    this.bitMask = bitMask;
  }

  /**
   * Get the underlying bit mask,
   * @return bit mask
   */
  @NotNull
  public BitMask getBitMask()
  {
    return bitMask;
  }

  /**
   * Is the given flag set?
   * @param flag flag to check
   * @return {@code true}: the flag is set<br>
   *         {@code false}: the flag is not set
   */
  public boolean isSet(@NotNull E flag)
  {
    return flag.isSetIn(bitMask);
  }

  /**
   * Is any flag in both this and the other bit mask set?
   * @param other other bit mask
   * @return {@code true}: if any bit is set in both masks<br>
   *         {@code false}: if no bit is set in both masks
   */
  public boolean isSetAny(@NotNull EnumBitMask<E> other)
  {
    return !other.getBitMask().and(getBitMask()).isEmpty();
  }

  /**
   * Are all flags set in the other bitmask set in this bitmask, too?
   * @param other other bit mask
   * @return {@code true}: if all set flags of the other mask are set in this mask<br>
   *         {@code false}: if any flag set in the other mask is not set in this mask
   */
  public boolean isSetAll(@NotNull EnumBitMask<E> other)
  {
    final BitMask combined = other.getBitMask().and(getBitMask());
    return BitMaskUtil.areEqual(combined, bitMask);
  }

  /**
   * Is any flag from the given sequence set?
   * @param flags flags to check
   * @return {@code true}: if any bit flag is set in this masks<br>
   *         {@code false}: if no bit flag is set in this masks
   */
  @SafeVarargs
  public final boolean isSetAny(@NotNull E... flags)
  {
    for (E flag : flags) {
      if (isSet(flag)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Is any flag from the given collection set?
   * @param flags flags to check
   * @return {@code true}: if any bit flag is set in this masks<br>
   *         {@code false}: if no bit flag is set in this masks
   */
  public boolean isSetAny(@NotNull Collection<E> flags)
  {
    for (E flag : flags) {
      if (isSet(flag)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Are all flags from the given sequence set?
   * @param flags flags to check
   * @return {@code true}: if every bit flag is set in this masks<br>
   *         {@code false}: if any bit flag is not set in this masks
   */
  @SafeVarargs
  public final boolean isSetAll(@NotNull E... flags)
  {
    for (E flag : flags) {
      if (!isSet(flag)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Are all flags from the given collection set?
   * @param flags flags to check
   * @return {@code true}: if every bit flag is set in this masks<br>
   *         {@code false}: if any bit flag is not set in this masks
   */
  public boolean isSetAll(@NotNull Collection<E> flags)
  {
    for (E flag : flags) {
      if (!isSet(flag)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Get a bit mask where all flags are set which are set in this or in another one.
   * @param other other bit mask defining the flags which are set
   * @return this bit mask, but with set flags
   * @see #and(EnumBitMask)
   * @see #clear(EnumBitMask)
   */
  @NotNull
  public EnumBitMask<E> or(@NotNull EnumBitMask<E> other)
  {
    final BitMask combined = bitMask.or(other.getBitMask());
    return combined.equals(bitMask)
            ? this
            : new EnumBitMask<E>(combined);
  }

  /**
   * Get a bit mask where all flags are set which are set in this and in another one.
   * @param other other bit mask defining the flags which are set
   * @return this bit mask, but with set flags
   * @see #or(EnumBitMask)
   * @see #clear(EnumBitMask)
   */
  @NotNull
  public EnumBitMask<E> and(@NotNull EnumBitMask<E> other)
  {
    final BitMask combined = bitMask.and(other.getBitMask());
    return combined.equals(bitMask)
            ? this
            : new EnumBitMask<E>(combined);
  }

  /**
   * Set the given flag.
   * @param flag flag to set
   * @return bit mask where the given flag is set
   */
  @NotNull
  public EnumBitMask<E> set(@NotNull E flag)
  {
    if (isSet(flag)) {
      return this;
    }
    return new EnumBitMask<>(flag.setIn(bitMask));
  }

  /**
   * Set or clear the given flag.
   * @param flag  flag to change
   * @param onOff {@code true}: set the flag<br>
   *              {@code false}: clear the flag
   * @return bit mask with flag change appropriately
   */
  public EnumBitMask<E> setTo(@NotNull E flag, boolean onOff)
  {
    return onOff
            ? set(flag)
            : clear(flag);
  }

  /**
   * Set all given flags.
   * @param flags flags to set
   * @return bit mask, where the flags of this mask and the given flags are set
   */
  @SafeVarargs
  @NotNull
  public final EnumBitMask<E> setAll(@NotNull E... flags)
  {
    BitMask combined = bitMask;
    for (E flag : flags) {
      combined = flag.setIn(combined);
    }
    return combined.equals(bitMask)
            ? this
            : new EnumBitMask<E>(combined);
  }

  /**
   * Clear the given flag.
   * @param flag flag to clear
   * @return bit flags where the given flag is cleared
   */
  @NotNull
  public EnumBitMask<E> clear(@NotNull E flag)
  {
    if (!isSet(flag)) {
      return this;
    }
    return new EnumBitMask<>(flag.clearIn(bitMask));
  }

  /**
   * Get a bit mask where all flags are cleared which are set in another one.
   * @param other other bit mask defining the flags which are cleared
   * @return this bit mask, but with cleared flags
   * @see #and(EnumBitMask)
   * @see #or(EnumBitMask)
   */
  public EnumBitMask<E> clear(@NotNull EnumBitMask<E> other)
  {
    final BitMask combined = getBitMask().andNot(other.getBitMask());
    return combined.equals(bitMask)
            ? this
            : new EnumBitMask<E>(combined);
  }

  /**
   * Clear all given flags.
   * @param flags flags to clear
   * @return bit mask, with the flags of this mask where all given flags are cleared
   */
  @SafeVarargs
  @NotNull
  public final EnumBitMask<E> clearAll(@NotNull E... flags)
  {
    BitMask combined = bitMask;
    for (E flag : flags) {
      combined = flag.clearIn(combined);
    }
    return combined.equals(bitMask)
            ? this
            : new EnumBitMask<E>(combined);
  }

  /**
   * Flip the given flag.
   * @param flag flag to flip
   * @return bit flags where the given flag is inverted
   */
  @NotNull
  public EnumBitMask<E> flip(@NotNull E flag)
  {
    return isSet(flag)
            ? new EnumBitMask<E>(flag.clearIn(bitMask))
            : new EnumBitMask<E>(flag.setIn(bitMask));
  }

  /**
   * Flip all given flags.
   * @param flags flags to flip
   * @return bit mask, with the flags of this mask where all given flags are flipped
   */
  @SafeVarargs
  @NotNull
  public final EnumBitMask<E> flipAll(@NotNull E... flags)
  {
    BitMask combined = bitMask;
    for (E flag : flags) {
      if (flag.isSetIn(combined)) {
        combined = flag.clearIn(combined);
      }
      else {
        combined = flag.setIn(combined);
      }
    }
    return new EnumBitMask<E>(combined);
  }

  /**
   * Get the cardinality of this enum bit mask.
   * The cardinality is the number of flags which are set in this mask.
   * @return cardinality
   * @see BitMask#getCardinality()
   */
  public int getCardinality()
  {
    return bitMask.getCardinality();
  }

  /**
   * Get a list of all flags out of a given set which are set in this mask.
   * @param flags flags to check
   * @return a list of all checked flags which are set in this mask
   * @see #flags(Class)
   * @see #getAllSet(Iterable)
   */
  @NotNull
  @SafeVarargs
  public final List<E> getAllSet(@NotNull E... flags)
  {
    final List<E> result = new ArrayList<>(flags.length);
    for (E f : flags) {
      if (isSet(f)) {
        result.add(f);
      }
    }
    return result;
  }

  /**
   * Call a handler for each of the given flags which is set in this bit mask.
   * @param flags   flags to check
   * @param handler handler called with all flags from {@code flags} that are set
   */
  public void forEachSet(@NotNull Iterable<E> flags,
                         @NotNull Consumer<? super E> handler)
  {
    for (E flag : flags) {
      if (isSet(flag)) {
        handler.accept(flag);
      }
    }
  }

  /**
   * Call a handler for each of the given flags which is unset in this bit mask.
   * @param flags   flags to check
   * @param handler handler called with all flags from {@code flags} that are unset
   */
  public void forEachUnset(@NotNull Iterable<E> flags,
                           @NotNull Consumer<? super E> handler)
  {
    for (E flag : flags) {
      if (!isSet(flag)) {
        handler.accept(flag);
      }
    }
  }

  /**
   * Get a list of all flags which are set in this mask.
   * @param flagType type of flags in this mask, must match the type of flags used in this mask.
   *                 Necessary due to the quirks of the Java type system.
   * @return flags set in this mask, in enum order
   * @see #getAllSet(Enum[])
   * @see #getAllSet(Iterable)
   */
  @NotNull
  public List<E> flags(@NotNull Class<E> flagType)
  {
    return getAllSet(flagType.getEnumConstants());
  }

  /**
   * Get a list of all flags which are set in this mask.
   * @param flags flags to check
   * @return ma list of all checked flags which are set in this maks
   * @see #flags(Class)
   * @see #getAllSet(Enum[])
   */
  @NotNull
  public List<E> getAllSet(@NotNull Iterable<E> flags)
  {
    final List<E> result = new LinkedList<>();
    for (E f : flags) {
      if (isSet(f)) {
        result.add(f);
      }
    }
    return result;
  }

  /**
   * Get a list of all flags which are not set in this mask.
   * @param flags flags to check
   * @return a list of all checked flags which are not set in this mask
   */
  @NotNull
  @SafeVarargs
  public final List<E> getAllCleared(@NotNull E... flags)
  {
    final List<E> result = new ArrayList<>(flags.length);
    for (E f : flags) {
      if (!isSet(f)) {
        result.add(f);
      }
    }
    return result;
  }

  /**
   * Get a list of all flags which are not set in this mask.
   * @param flags flags to check
   * @return ma list of all checked flags which are not set in this maks
   */
  @NotNull
  public List<E> getAllCleared(@NotNull Iterable<E> flags)
  {
    final List<E> result = new LinkedList<>();
    for (E f : flags) {
      if (!isSet(f)) {
        result.add(f);
      }
    }
    return result;
  }

  /**
   * Get the highest flag set.
   * This method interprets &quot;highest&quot; in the sense
   * of the associated bit in the underlying bit mask
   * (compare {@link BitMask#getHighestBitSet()})
   * and <b>not</b> in enum order.
   * @return position of the highest flag bit set, {@code -1} if no flag is set
   */
  public int getHighestFlagSet()
  {
    return bitMask.getHighestBitSet();
  }

  /**
   * Get the lowest flag set.
   * This method interprets &quot;lowest&quot; in the sense
   * of the associated bit in the underlying bit mask
   * (compare {@link BitMask#getLowestBitSet()})
   * and <b>not</b> in enum order.
   * @return position of the lowest flag bit set, {@code -1} if no flag is set
   */
  public int getLowestFlagSet()
  {
    return bitMask.getLowestBitSet();
  }

  /**
   * Is no flag set?
   * @return {@code true}: if no flag in this bit mask is set<br>
   *         {@code false}: if any flag in this bit mask is set
   */
  public boolean isEmpty()
  {
    return bitMask.isEmpty();
  }

  /**
   * Get a version of this enum bit mask where all flags are cleared.
   * @return empty enum bit mask
   */
  @NotNull
  public EnumBitMask<E> cleared()
  {
    if (isEmpty()) {
      return this;
    }
    return new EnumBitMask<>(bitMask.cleared());
  }

  /**
   * Cast this enum bit mask to one using a different kind of bit flags.
   * @param enumClass new enum bit flag class
   * @param <F> type of enum bit flag class
   * @return enum bit mask with same bit flags, but different enum flag type
   */
  @SuppressWarnings("unchecked")
  @NotNull
  public <F extends Enum<F> & BitFlag> EnumBitMask<F> cast(@NotNull Class<F> enumClass)
  {
    // no need to create a new object, as this class is immutable,
    // and the casting is only necessary to keep the compiler happy
    return (EnumBitMask<F>)this;
  }

  /**
   * Get the unique object representing the given 16bit flags.
   * This is an optimization which shares all enum bit masks with 16 bits.
   * @return unique bitmask object representing the same flags as this one
   */
  @SuppressWarnings("unchecked")
  @NotNull
  public EnumBitMask<E> unique16()
  {
    // cast is okay because all 16bit flag classes share the same underlying objects
    return getCachedEnumBitMask16(this);
  }

  /**
   * Get the unique object representing the given 32bit flags.
   * This is an optimization which shares many enum bit masks with 32 bits.
   * @return unique bitmask object representing the same flags as this one
   */
  @SuppressWarnings("unchecked")
  @NotNull
  public EnumBitMask<E> unique32()
  {
    // cast is okay because all 16bit flag classes share the same underlying objects
    return getCachedEnumBitMask32(this);
  }

  /**
   * Map the chosen set flags from this mask to another mask.
   * @param flagsToCheck  set of flags to check from this mask
   * @param mapping       predicate called for each checked flag which is set
   *                      deciding which flags are set in the result
   * @return new mask
   */
  @NotNull
  public EnumBitMask<E> mapFromSet(@NotNull Iterable<E> flagsToCheck,
                                   @NotNull Predicate<? super E> mapping)
  {
    EnumBitMask<E> result = cleared();
    for (E flag : flagsToCheck) {
      if (isSet(flag)  &&  mapping.test(flag)) {
        result = result.set(flag);
      }
    }
    return result;
  }

  /**
   * Map the chosen unset flags from this mask to another mask.
   * @param flagsToCheck  set of flags to check from this mask
   * @param mapping       predicate called for each checked flag which is unset
   *                      deciding which flags are set in the result
   * @return new mask
   */
  @NotNull
  public EnumBitMask<E> mapFromUnset(@NotNull Iterable<E> flagsToCheck,
                                     @NotNull Predicate<? super E> mapping)
  {
    EnumBitMask<E> result = cleared();
    for (E flag : flagsToCheck) {
      if (!isSet(flag)  &&  mapping.test(flag)) {
        result = result.set(flag);
      }
    }
    return result;
  }

  /**
   * Map the chosen flags from this mask to another flag
   * @param flagsToCheck  set of flags to check from this mask
   * @param mapping       predicate called for each flag from the set,
   *                      second parameter is {@code true} if the flag is set
   *                      and {@code false} if it is unset
   * @return new mask
   */
  @NotNull
  public EnumBitMask<E> map(@NotNull Iterable<E> flagsToCheck,
                            @NotNull BiPredicate<? super E, Boolean> mapping)
  {
    EnumBitMask<E> result = cleared();
    for (E flag : flagsToCheck) {
      if (mapping.test(flag, isSet(flag))) {
        result = result.set(flag);
      }
    }
    return result;
  }

  /**
   * Convert a flag to a generic bit mask.
   * <p>
   * This internally uses a bit set bit mask, which allows for all possible flags.
   * @param flag flag to convert
   * @param <F>  type of bit flag
   * @return bit mask where bit flag is set
   */
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> toMask(@NotNull F flag)
  {
    return new EnumBitMask<F>(flag.setIn(BitSetBitMask.ZERO));
  }

  /**
   * Convert a flag to a 16 bit bit mask.
   *
   * @param flag flag to convert
   * @param <F>  type of bit flag
   * @return bit mask where bit flag is set
   */
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> toMask16(@NotNull F flag)
  {
    return new EnumBitMask<F>(flag.setIn(BitMask16.ZERO)).unique16();
  }

  /**
   * Convert a flag to a 32 bit bit mask.
   *
   * @param flag flag to convert
   * @param <F>  type of bit flag
   * @return bit mask where bit flag is set
   */
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> toMask32(@NotNull F flag)
  {
    return new EnumBitMask<F>(flag.setIn(BitMask32.ZERO)).unique32();
  }

  /**
   * Convert a flag to a 64 bit bit mask.
   *
   * @param flag flag to convert
   * @param <F>  type of bit flag
   * @return bit mask where bit flag is set
   */
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> toMask64(@NotNull F flag)
  {
    return new EnumBitMask<F>(flag.setIn(BitMask64.ZERO));
  }

  /**
   * Divide a bit mask into a set of flags.
   * Only valid bits (i.e. bits which have a flag counterpart) will be considered.
   * @param mask bit mask
   * @param enumClass class of the enum flags in this mask, necessary to overcome
   *                  the shortcomings of Java generics
   * @param <F> type of bit flag
   * @return set of bit flags which are set in the given mask
   * @see #combine(Enum[])
   * @see #combine(BitMask, Enum[])
   * @see #combine16(Collection)
   * @see #combine32(Collection)
   * @see #combine64(Collection)
   */
  @NotNull
  public static <F extends Enum<F> & BitFlag> Set<F> toFlags(@NotNull EnumBitMask<F> mask,
                                                             @NotNull Class<F> enumClass)
  {
    Set<F> result = EnumSet.noneOf(enumClass);
    for (F flag : enumClass.getEnumConstants()) {
      if (mask.isSet(flag)) {
        result.add(flag);
      }
    }
    return result;
  }

  /**
   * Combine some flags into a bit mask.
   * @param zero     zero bit mask fitting for the given type
   * @param bitFlags bit flags to combine
   * @param <F> enum flag type
   * @return combined mask
   */
  @NotNull
  private static <F extends Enum<F> & BitFlag> EnumBitMask<F> combine(@NotNull BitMask zero,
                                                                      F[] bitFlags)
  {
    EnumBitMask<F> mask = new EnumBitMask<>(zero);
    for (F f  : bitFlags) {
      mask = mask.set(f);
    }
    return mask;
  }

  /**
   * Combine some flags into a bit mask.
   * @param zero     zero bit mask fitting for the given type
   * @param bitFlags bit flags to combine
   * @param <F> enum flag type
   * @return combined mask
   */
  @NotNull
  private static <F extends Enum<F> & BitFlag> EnumBitMask<F> combine(@NotNull BitMask zero,
                                                                      @NotNull Iterable<F> bitFlags)
  {
    EnumBitMask<F> mask = new EnumBitMask<>(zero);
    for (F f  : bitFlags) {
      mask = mask.set(f);
    }
    return mask;
  }

  /**
   * Combine bit flags using at maximum 16 bits into a bit mask.
   * @param bitFlags bit flags
   * @param <F>      bit flag type
   * @return bit mask
   * @see #get16(Enum[])
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> combine16(F ... bitFlags)
  {
    return combine(BitMask16.ZERO, bitFlags);
  }

  /**
   * Combine bit flags using at maximum 16 bits into a bit mask.
   * @param bitFlags bit flags to combine
   * @param <F>      bit flag type
   * @return bit mask
   */
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> combine16(@NotNull Collection<F> bitFlags)
  {
    return combine(BitMask16.ZERO, bitFlags);
  }

  /**
   * Combine bit flags using at maximum 32 bits into a bit mask.
   * @param bitFlags bit flags
   * @param <F>      bit flag type
   * @return bit mask
   * @see #get32(Enum[])
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> combine32(F ... bitFlags)
  {
    return combine(BitMask32.ZERO, bitFlags);
  }

  /**
   * Combine bit flags using at maximum 32 bits into a bit mask.
   * @param bitFlags bit flags to combine
   * @param <F>      bit flag type
   * @return bit mask
   */
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> combine32(@NotNull Collection<F> bitFlags)
  {
    return combine(BitMask32.ZERO, bitFlags);
  }

  /**
   * Combine bit flags using at maximum 64 bits into a bit mask.
   * @param bitFlags bit flags
   * @param <F>      bit flag type
   * @return bit mask
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> combine64(F ... bitFlags)
  {
    return combine(BitMask64.ZERO, bitFlags);
  }

  /**
   * Combine bit flags using at maximum 64 bits into a bit mask.
   * @param bitFlags bit flags
   * @param <F>      bit flag type
   * @return bit mask
   */
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> combine64(@NotNull Collection<F> bitFlags)
  {
    return combine(BitMask64.ZERO, bitFlags);
  }

  /**
   * Combine bit flags using an arbitrary number of bits into a bit mask.
   * @param bitFlags bit flags
   * @param <F>      bit flag type
   * @return bit mask
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> combine(F ... bitFlags)
  {
    return combine(BitSetBitMask.ZERO, bitFlags);
  }

  /**
   * Combine bit flags using an arbitrary number of bits into a bit mask.
   * @param bitFlags bit flags to combine
   * @param <F>      bit flag type
   * @return bit mask
   */
  @NotNull
  public static <F extends Enum<F> & BitFlag> EnumBitMask<F> combine(@NotNull Iterable<F> bitFlags)
  {
    return combine(BitSetBitMask.ZERO, bitFlags);
  }

  @Override
  @SuppressWarnings("unchecked")
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    EnumBitMask<E> that = (EnumBitMask<E>)o;

    return bitMask.equals(that.bitMask);

  }

  @Override
  public int hashCode()
  {
    return bitMask.hashCode();
  }

  /**
   * Get a string representation.
   * @return string representation
   */
  public String toString()
  {
    return bitMask.toString();
  }

  /** Have an enum for fooling the compiler. */
  private enum Potemkin
    implements BitFlag
  {
    ;

    /**
     * Is this flag set in the given mask?
     *
     * @param mask bit mask
     * @return {@code true}: if the flag is set<br>
     * {@code false}: if the flag is not set
     */
    @Override
    public boolean isSetIn(@NotNull BitMask mask)
    {
      return false;
    }

    /**
     * Set this flag in the given bit mask.
     *
     * @param mask bit mask
     * @return new bit mask where this flag is set
     */
    @NotNull
    @Override
    public BitMask setIn(@NotNull BitMask mask)
    {
      return mask;
    }

    /**
     * Clear this flag in the given bit mask.
     *
     * @param mask bit mask
     * @return new bit mask where this flag is cleared
     */
    @NotNull
    @Override
    public BitMask clearIn(@NotNull BitMask mask)
    {
      return mask;
    }
  }

  /** 16 bit mask with no flag set. */
  public static final EnumBitMask<?> NONE16 = new EnumBitMask<Potemkin>(BitMask16.ZERO);
  /** 32 bit mask with no flag set. */
  public static final EnumBitMask<?> NONE32 = new EnumBitMask<Potemkin>(BitMask32.ZERO);
  /** 64 bit mask with no flag set. */
  public static final EnumBitMask<?> NONE64 = new EnumBitMask<Potemkin>(BitMask64.ZERO);

  /** Cached values using 16bit masks. */
  private static final Map<Short, EnumBitMask<?>> CACHE16 = new HashMap<>();
  static {
    CACHE16.put((short)0, NONE16);
    CACHE16.put((short)-1, new EnumBitMask<>(new BitMask16(-1)));
    /* Insert common values for and operation. */
    for (int shift = 1;  shift < 16;  ++shift) {
      final int value = (1 << shift) - 1;
      CACHE16.put((short)value, new EnumBitMask<>(new BitMask16(value)));
    }
  }

  /**
   * Get a cached 16bit mask for the given value.
   * <p>
   * This is an optimization reducing created bit masks.
   *
   * @param <T> enum type
   * @param value value (an int only for convenience, only the low 16bits are used)
   * @param type  enum type of returned
   * @return matching bit mask
   */
  @NotNull
  public static <T extends Enum<T> & BitFlag> EnumBitMask<T> get16(int value, @NotNull Class<T> type)
  {
    final Short key = (short)value;
    synchronized (CACHE16) {
      EnumBitMask<?> mask = CACHE16.get(key);
      if (mask == null) {
        mask = new EnumBitMask<T>(new BitMask16(value));
        CACHE16.put(key, mask);
      }
      return mask.cast(type);
    }
  }

  /**
   * Get a cached 16bit mask for a combination of flags.
   * <p>
   * This is an optimization reducing created bit masks.
   *
   * @param <T> enum type
   * @param flags flags to combine
   * @return matching bit mask
   */
  @SafeVarargs
  @NotNull
  @SuppressWarnings({"unchecked", "varargs"})
  public static <T extends Enum<T> & BitFlag> EnumBitMask<T> get16(T ... flags)
  {
    final EnumBitMask<T> mask = combine16(flags);
    return getCachedEnumBitMask16(mask);
  }

  /**
   * Get the cached value with the same settings as a given 16 bit mask.
   * @param mask bit mask using 16 bits
   * @param <T> bit flag type
   * @return cached mask
   */
  @NotNull
  @SuppressWarnings("unchecked")
  private static <T extends Enum<T> & BitFlag> EnumBitMask<T> getCachedEnumBitMask16(@NotNull EnumBitMask<T> mask)
  {
    final Short key = (short)mask.getBitMask().low32();
    synchronized (CACHE16) {
      final EnumBitMask<?> cachedMask = CACHE16.get(key);
      if (cachedMask == null) {
        CACHE16.put(key, mask);
        return mask;
      }
      // all enum bit masks with the same width and value are internally equal
      return (EnumBitMask<T>)cachedMask;
    }
  }

  /**
   * Get a cached 16bit mask for a combination of flags.
   * <p>
   * This is an optimization reducing created bit masks.
   *
   * @param <T> enum type
   * @param mask  basic mask
   * @param flags flags to combine with mask
   * @return combined bit mask
   */
  @SafeVarargs
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T extends Enum<T> & BitFlag> EnumBitMask<T> get16(@NotNull EnumBitMask<T> mask,
                                                                   @NotNull T ... flags)
  {
    for (T flag : flags) {
      mask = mask.set(flag);
    }
    return getCachedEnumBitMask16(mask);
  }

  /** Cached values using 32bit masks. */
  private static final Map<Integer, EnumBitMask<?>> CACHE32 = new HashMap<>();
  static {
    CACHE32.put(0, NONE32);
    CACHE32.put(-1, new EnumBitMask<>(new BitMask32(-1)));
    /* Insert common values for and operation. */
    for (int shift = 1;  shift < 32;  ++shift) {
      final int value = (1 << shift) - 1;
      CACHE32.put(value, new EnumBitMask<>(new BitMask32(value)));
    }
  }

  /**
   * Get a cached 32bit mask for the given value.
   * <p>
   * This is an optimization reducing the number of created bit masks.
   *
   * @param <T> enum type
   * @param value value
   * @param type  enum type of returned
   * @return matching bit mask
   */
  @NotNull
  public static <T extends Enum<T> & BitFlag> EnumBitMask<T> get32(@NotNull Integer value, @NotNull Class<T> type)
  {
    synchronized (CACHE32) {
      EnumBitMask<?> mask = CACHE32.get(value);
      if (mask == null) {
        mask = new EnumBitMask<T>(new BitMask32(value));
        if ((value & EXCLUDE_MASK32) == 0) {
          CACHE32.put(value, mask);
        }
      }
      return mask.cast(type);
    }
  }

  /**
   * Get a cached 32bit mask for a combination of flags.
   * <p>
   * This is an optimization reducing created bit masks.
   *
   * @param <T> enum type
   * @param flags flags to combine
   * @return matching bit mask
   */
  @SafeVarargs
  @NotNull
  @SuppressWarnings("varargs")
  public static <T extends Enum<T> & BitFlag> EnumBitMask<T> get32(T ... flags)
  {
    final EnumBitMask<T> mask = combine32(flags);
    return getCachedEnumBitMask32(mask);
  }

  /**
   * Get the cached value with the same settings as a given 16 bit mask.
   * @param mask bit mask using 16 bits
   * @param <T> bit flag type
   * @return cached mask
   */
  @NotNull
  @SuppressWarnings("unchecked")
  private static <T extends Enum<T> & BitFlag> EnumBitMask<T> getCachedEnumBitMask32(@NotNull EnumBitMask<T> mask)
  {
    final Integer key = mask.getBitMask().low32();
    synchronized (CACHE32) {
      final EnumBitMask<?> cachedMask = CACHE32.get(key);
      if (cachedMask == null) {
        if ((key & EXCLUDE_MASK32) == 0) {
          CACHE32.put(key, mask);
        }
        return mask;
      }
      return (EnumBitMask<T>)cachedMask;
    }
  }

  /**
   * Get a cached 32bit mask for a combination of flags.
   * <p>
   * This is an optimization reducing created bit masks.
   *
   * @param <T> enum type
   * @param mask  basic mask
   * @param flags flags to combine with mask
   * @return combined bit mask
   */
  @SafeVarargs
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T extends Enum<T> & BitFlag> EnumBitMask<T> get32(@NotNull EnumBitMask<T> mask,
                                                                   @NotNull T ... flags)
  {
    for (T flag : flags) {
      mask = mask.set(flag);
    }
    return getCachedEnumBitMask32(mask);
  }
}
