// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;

import java.io.IOException;
import java.io.OutputStream;

/**
 * A wrapper output streams which counts the bytes written so far.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since 2005
 */
public class CountingOutputStream
        extends OutputStream
{
  /** The wrapped output stream. */
  private final OutputStream wrapped;
  /** The count of bytes written so far. */
  private long count = 0;
  /**
   *  Creat a counting output stream which wraps the given output stream.
   *  @param wrapped wrapped
   */
  public CountingOutputStream(OutputStream wrapped)
  {
    this.wrapped = wrapped;
  }

  /**
   *  Get the bytes written so far.
   *  Notice that it may be necessary to call flush to be sure that the bytes are indeed written
   *  and not buffered in underlying streams.
   *  @return number of bytes written so far
   */
  public long getCount()
  {
    return count;
  }

  /**
   * Writes the specified byte to this output stream. The general
   * contract for {@code write} is that one byte is written
   * to the output stream. The byte to be written is the eight
   * low-order bits of the argument {@code b}. The 24
   * high-order bits of {@code b} are ignored.
   * <p> 
   * Subclasses of {@code OutputStream} must provide an
   * implementation for this method.
   *
   * @param b the {@code byte}.
   * @throws java.io.IOException if an I/O error occurs. In particular,
   *                             an {@code IOException} may be thrown if the
   *                             output stream has been closed.
   */
  @Override
  public void write(int b) throws IOException
  {
    wrapped.write(b);
    ++count;
  }

  /**
   * Closes this output stream and releases any system resources
   * associated with this stream. The general contract of {@code close}
   * is that it closes the output stream. A closed stream cannot perform
   * output operations and cannot be reopened.
   * <p> 
   * The {@code close} method of {@code OutputStream} does nothing.
   *
   * @throws java.io.IOException if an I/O error occurs.
   */
  @Override
  public void close() throws IOException
  {
    wrapped.close();
  }

  /**
   * Flushes this output stream and forces any buffered output bytes
   * to be written out. The general contract of {@code flush} is
   * that calling it is an indication that, if any bytes previously
   * written have been buffered by the implementation of the output
   * stream, such bytes should immediately be written to their
   * intended destination.
   * <p> 
   * The {@code flush} method of {@code OutputStream} does nothing.
   *
   * @throws java.io.IOException if an I/O error occurs.
   */
  @Override
  public void flush() throws IOException
  {
    wrapped.flush();
  }

  /**
   * Writes {@code b.length} bytes from the specified byte array
   * to this output stream. The general contract for {@code write(b)}
   * is that it should have exactly the same effect as the call
   * {@code write(b, 0, b.length)}.
   *
   * @param b the data.
   * @throws java.io.IOException if an I/O error occurs.
   * @see java.io.OutputStream#write(byte[], int, int)
   */
  @Override
  public void write(@NotNull byte[] b) throws IOException
  {
    wrapped.write(b);
    count += b.length;
  }

  /**
   * Writes {@code len} bytes from the specified byte array
   * starting at offset {@code off} to this output stream.
   * The general contract for {@code write(b, off, len)} is that
   * some of the bytes in the array {@code b} are written to the
   * output stream in order; element {@code b[off]} is the first
   * byte written and {@code b[off+len-1]} is the last byte written
   * by this operation.
   * <p> 
   * The {@code write} method of {@code OutputStream} calls
   * the write method of one argument on each of the bytes to be
   * written out. Subclasses are encouraged to override this method and
   * provide a more efficient implementation.
   * <p> 
   * If {@code b} is {@code null}, a
   * {@code NullPointerException} is thrown.
   * <p> 
   * If {@code off} is negative, or {@code len} is negative, or
   * {@code off+len} is greater than the length of the array
   * {@code b}, then an <tt>IndexOutOfBoundsException</tt> is thrown.
   *
   * @param b   the data.
   * @param off the start offset in the data.
   * @param len the number of bytes to write.
   * @throws java.io.IOException if an I/O error occurs. In particular,
   *                             an {@code IOException} is thrown if the output
   *                             stream is closed.
   */
  @Override
  public void write(@NotNull byte[] b, int off, int len) throws IOException
  {
    wrapped.write(b, off, len);
    count += len;
  }

}
