// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.generics.ByteIndexable;
import de.caff.generics.Empty;

import java.util.Arrays;

/**
 * A byte buffer to which you can add data bytes up to a maximum size.
 * <p>
 * If the buffer overflows you can only retrieve the latest
 * {@linkplain #getLimit()} bytes.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August 10, 2022
 */
public class ByteRingBuffer
        implements ByteCollector
{
  /** Initial limit: 1 MiB. */
  static final int INITIAL_LIMIT = 0x100000;

  /** Buffer limit. */
  private final int limit;
  /**
   * Internal ring buffer.
   * May change size until it overflows.
   */
  private byte[] ringBuffer;
  /** Offset in the active buffer where the next byte will be added. */
  private int bufferOffset;
  /** Number of cached bytes. */
  private int bytesCached;
  /** Total number of bytes appended to this ring buffer. */
  private long totalNumberOfBytes;

  /**
   * Create a ring buffer with the given size.
   * @param limit size limit of this buffer
   */
  public ByteRingBuffer(int limit)
  {
    this(limit, INITIAL_LIMIT);

  }
  /**
   * Create a ring buffer with the given size.
   * This allows to set the initial limit for testing.
   * @param limit size limit of this buffer
   * @param initialLimit initial limit
   */
  ByteRingBuffer(int limit, int initialLimit)
  {
    if (limit <= 0) {
      throw new IllegalArgumentException("limit has to be positive, but is " + limit);
    }
    if (limit < initialLimit) {
      ringBuffer = new byte[limit];
    }
    else {
      ringBuffer = new byte[initialLimit];
    }
    this.limit = limit;
  }

  /**
   * Get the limit of this ring buffer.
   * @return buffer limit
   */
  public int getLimit()
  {
    return limit;
  }

  /**
   * Get the number of valid bytes cached in this buffer.
   * @return number of bytes cached
   */
  public synchronized int size()
  {
    return bytesCached;
  }

  /**
   * Append one byte
   * @param value byte value. Accepts integer for convenience, but will cast the given value to byte.
   */
  @Override
  public synchronized void append(int value)
  {
    final byte byteValue = (byte) value;
    if (bufferOffset < ringBuffer.length) {
      ringBuffer[bufferOffset++] = byteValue;
      if (bytesCached < limit) {
        ++bytesCached;
      }
      ++totalNumberOfBytes;
    }
    else {
      if (ringBuffer.length == limit) {
        ringBuffer[0] = byteValue;
        bufferOffset = 1;
        ++totalNumberOfBytes;
      }
      else {
        append(new byte[] { byteValue });
      }
    }
  }

  /**
   * Append the given byte array part to this buffer.
   * @param bytes  basic byte array
   * @param offset start offset of the bytes to append
   * @param length number of bytes to append
   * @throws IndexOutOfBoundsException if the parameters define a part which is at least partially outside the {@code bytes}
   */
  @Override
  public synchronized void append(@NotNull byte[] bytes, int offset, int length)
  {
    if ((offset < 0) || (offset > bytes.length) || (length < 0) ||
        ((offset + length) - bytes.length > 0)) {
      throw new IndexOutOfBoundsException();
    }
    appendInternally(bytes, offset, length);
  }

  /**
   * Internal implementation of appending.
   * @param bytes  basic byte array
   * @param offset start offset of the bytes to append
   * @param length number of bytes to append
   */
  private void appendInternally(@NotNull byte[] bytes, int offset, int length)
  {
    if (length > ringBuffer.length - bufferOffset) {
      if (ringBuffer.length < limit) {
        // expand buffer
        final int newSize;
        if (ringBuffer.length > Integer.MAX_VALUE / 2) {
          newSize = limit;
        }
        else {
          newSize = Math.min(limit, 2 * ringBuffer.length);
        }
        ringBuffer = Arrays.copyOf(ringBuffer, newSize);
        appendInternally(bytes, offset, length);
        return;
      }
      if (length > limit) {
        clear();
        final int delta = length - limit;
        appendInternally(bytes, offset + delta, limit);
        return;
      }
      assert ringBuffer.length == limit;
      final int atEnd = limit - bufferOffset;
      if (atEnd > 0) {
        System.arraycopy(bytes, offset, ringBuffer, bufferOffset, atEnd);
        offset += atEnd;
        length -= atEnd;
        totalNumberOfBytes += atEnd;
      }
      bufferOffset = 0;
      bytesCached = limit;
    }
    System.arraycopy(bytes, offset, ringBuffer, bufferOffset, length);
    bufferOffset += length;
    if (bytesCached < limit) {
      bytesCached += Math.min(length, limit - bytesCached);
    }
    totalNumberOfBytes += length;
  }

  /**
   * Create a byte array with the bytes in this buffer.
   * @return byte array of size {@linkplain #size()} containing with the buffered bytes
   */
  @NotNull
  public synchronized byte[] toByteArray()
  {
    if (bytesCached == 0) {
      return Empty.BYTE_ARRAY;
    }

    final byte[] result = new byte[bytesCached];
    if (bufferOffset < bytesCached) {
      final int atEnd = bytesCached - bufferOffset;
      System.arraycopy(ringBuffer, limit - atEnd, result, 0, atEnd);
      System.arraycopy(ringBuffer, 0, result, atEnd, bufferOffset);
    }
    else {
      System.arraycopy(ringBuffer, bufferOffset - bytesCached,
                       result, 0, bytesCached);
    }
    return result;
  }

  /**
   * Get a byte indexable view of this ring buffer.
   * <p>
   * This is more efficient than copying all bytes in {@linkplain #toByteArray()},
   * especially if only a few bytes of a larger content are of interest. But the
   * price is high: any change to this buffer can break the returned view. So this
   * is best used on a finished buffer.
   *
   * @return byte indexable view of this buffer
   * @see #toByteIndexable()
   * @see #toByteArray()
   */
  @NotNull
  public synchronized ByteIndexable view()
  {
    if (bytesCached == 0) {
      return ByteIndexable.EMPTY;
    }
    if (bufferOffset >= bytesCached) {
      return ByteIndexable.viewArray(ringBuffer, bufferOffset - bytesCached, bytesCached);
    }
    final int size = bytesCached;
    final int start = bufferOffset;
    final int boundary = ringBuffer.length - start;
    return new ByteIndexable.Base()
    {
      @Override
      public byte get(int index)
      {
        if (index < boundary) {
          return ringBuffer[start + index];
        }
        return ringBuffer[index - boundary];
      }

      @Override
      public int size()
      {
        return size;
      }
    };
  }

  /**
   * Get a stable byte indexable from this buffer.
   * <p>
   * Compared to {@link #view()} the returned indexable
   * is decoupled from this buffer, with the price
   * of a copy of the bytes. Compared to the result of
   * <pre>{@code
   * ByteIndexable.viewArray(ringBuffer.toByteArray())
   * }</pre>
   * this method provides an already {@linkplain ByteIndexable#frozen() frozen}
   * byte indexable.
   * @return stable frozen byte indexable
   * @see #toByteArray()
   * @see #view()
   */
  @NotNull
  public synchronized ByteIndexable toByteIndexable()
  {
    return view().frozen();
  }

  /**
   * Clear all data.
   * This will clear everything but {@link #getNumberOfCollectedBytes() the total number of bytes handled by this buffer}.
   */
  @Override
  public synchronized void clear()
  {
    bufferOffset = 0;
    bytesCached = 0;
  }

  /**
   * Get the total number of bytes appended to this buffer.
   * @return total number of appended bytes
   */
  @Override
  public synchronized long getNumberOfCollectedBytes()
  {
    return totalNumberOfBytes;
  }
}
