// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2013-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Types;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * Byte order mark support.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public enum ByteOrderMark
{
  /** UTF-8 byte order mark. */
  Utf8(StandardCharsets.UTF_8,
       0xef, 0xbb, 0xbf),
  /** UTF-16 big endian byte order mark. */
  Utf16BE(StandardCharsets.UTF_16BE,
          0xfe, 0xff),
  /** UTF-16 little endian byte order mark. */
  Utf16LE(StandardCharsets.UTF_16LE,
          0xff, 0xfe),
  /** Utf-32 big endian byte order mark. */
  Utf32BE(Charset.forName("UTF-32BE"), // standard Java 8 is required to support this charset
          0x00, 0x00, 0xfe, 0xff),
  /** Utf-32 little endian byte order mark. */
  Utf32LE(Charset.forName("UTF-32LE"), // standard Java 8 is required to support this charset
          0xff, 0xfe, 0x00, 0x00);

  /** Maximum supported byte order mark length. */
  private static final int MAX_BOM_LENGTH;
  static {
    int max = 0;
    for (ByteOrderMark bom : values()) {
      if (bom.mark.length > max) {
        max = bom.mark.length;
      }
    }
    MAX_BOM_LENGTH = max;
  }

  /** The charset. */
  @NotNull
  private final Charset charset;
  @NotNull
  private final byte[] mark;

  /**
   * Constructor.
   * @param charset charset indicated for this byte order mark whe
   * @param mark mark (only byte values)
   */
  ByteOrderMark(@NotNull Charset charset,
                int ... mark)
  {
    this.charset = charset;
    this.mark = Types.toByteArray(mark);
  }

  /**
   * Get the charset indicated by this byte order mark.
   * @return associated charset
   */
  @NotNull
  public Charset getCharset()
  {
    return charset;
  }

  /**
   * Write the byte order mark to the given stream.
   * @param os           output stream
   * @throws IOException on write errors
   */
  public void write(@NotNull OutputStream os) throws IOException
  {
    os.write(mark);
  }

  /**
   * Check whether the given stream starts with a byte order mark.
   * If a byte order mark is found, it is skipped.
   * @param is     input stream, which has to support marking
   * @return       the found BOM or {@code null} if no BOM was found
   * @throws IOException on read errors
   */
  @Nullable
  public static ByteOrderMark check(@NotNull InputStream is) throws IOException
  {
    if (!is.markSupported()) {
      throw new IOException("Can only check BOM for markable streams!");
    }
    byte[] check = new byte[MAX_BOM_LENGTH];
    for (ByteOrderMark bom : ByteOrderMark.values()) {
      is.mark(bom.mark.length);
      int bytesRead = is.read(check, 0, bom.mark.length);
      if (bytesRead == bom.mark.length) {
        if (equals(check, bom.mark, bom.mark.length)) {
          return bom;
        }
      }
      is.reset();
    }
    return null;
  }

  /**
   * Check whether the first bytes of two arrays are equal.
   * @param array1 first array
   * @param array2 second array
   * @param length length to check
   * @return {@code true} if the first length bytes are equal,<br>
   *         {@code false} otherwise
   */
  private static boolean equals(@NotNull byte[] array1, @NotNull byte[] array2, int length)
  {
    for (int i = length - 1;  i >= 0;  --i) {
      if (array1[i] != array2[i]) {
        return false;
      }
    }
    return true;
  }
}
