// ============================================================================
// File:               ByteCollector
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2022-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            8/12/22 10:53 AM
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;

/**
 * Byte collector interface.
 * <p>
 * To byte collector you can append bytes.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August 12, 2022
 */
public interface ByteCollector
{
  /**
   * Append one byte
   * @param value byte value. Accepts integer for convenience, but will cast the given value to byte.
   */
  void append(int value);

  /**
   * Append the given byte array to this buffer.
   * @param bytes bytes to append
   */
  default void append(@NotNull byte... bytes)
  {
    append(bytes, 0, bytes.length);
  }

  /**
   * Append the given byte array part to this buffer.
   * @param bytes  basic byte array
   * @param offset start offset of the bytes to append
   * @param length number of bytes to append
   * @throws IndexOutOfBoundsException if the parameters define a part which is at least partially outside the {@code bytes}
   */
  void append(@NotNull byte[] bytes, int offset, int length);

  /**
   * Clear all data.
   * This will clear everything but {@link #getNumberOfCollectedBytes() the total number of bytes handled by this buffer}.
   */
  void clear();

  /**
   * Get the total number of bytes appended to this buffer.
   * @return total number of appended bytes
   */
  long getNumberOfCollectedBytes();
}
