// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;

import java.util.BitSet;

/**
 * A bit set based bit mask.
 * This allows for arbitrary sized bit counts.
 * This class is immutable.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class BitSetBitMask
        implements BitMask
{
  /** Bit set bit mask with no bits set. */
  public static final BitSetBitMask ZERO = new BitSetBitMask(new BitSet());

  @NotNull
  private final BitSet bitSet;

  /**
   * Optimized internal constructor.
   * @param doClone clone parameter?
   * @param bitSet   flags to use
   */
  private BitSetBitMask(boolean doClone,
                        @NotNull BitSet bitSet)
  {
    this.bitSet = doClone
            ? (BitSet)bitSet.clone()
            : bitSet;
  }

  /**
   * Constructor.
   * @param bitSet flags on which this mask is based
   */
  public BitSetBitMask(@NotNull BitSet bitSet)
  {
    this(true, bitSet);
  }

  /**
   * Constructor.
   * This creates a bit mask with a capacity for the given number of flags.
   * @param flagCount flag count
   */
  public BitSetBitMask(int flagCount)
  {
    this(false, new BitSet(flagCount));
  }

  /**
   * Is the flag at the given position set?
   *
   * @param pos position (non-negative)
   * @return {@code true}: the flag is set<br>
   * {@code false}: the flag is not set
   */
  @Override
  public boolean isSet(int pos)
  {
    return bitSet.get(pos);
  }

  /**
   * Get the number of possible bits used in this flag.
   *
   * @return bit count
   */
  @Override
  public int getBitCount()
  {
    return bitSet.size();
  }

  /**
   * Get the number of bits set in this flag.
   *
   * @return set bit count
   */
  @Override
  public int getCardinality()
  {
    return bitSet.cardinality();
  }

  /**
   * Is no flag set?
   *
   * @return {@code true}: if no flag in this bit mask is set<br>
   * {@code false}: if any flag in this bit mask is set
   */
  @Override
  public boolean isEmpty()
  {
    return bitSet.isEmpty();
  }

  /**
   * Set the flag at the given position.
   *
   * @param pos position
   * @return bit mask where the flag at the given position is set
   */
  @NotNull
  @Override
  public BitMask set(int pos)
  {
    if (isSet(pos)) {
      return this;
    }
    final BitSet newFlags = (BitSet)bitSet.clone();
    newFlags.set(pos);
    return new BitSetBitMask(false, newFlags);
  }

  /**
   * Clear the flag at the given position.
   *
   * @param pos position
   * @return bit mask where the flag at the given position is cleared
   */
  @NotNull
  @Override
  public BitMask clear(int pos)
  {
    if (!isSet(pos)) {
      return this;
    }
    final BitSet newFlags = (BitSet)bitSet.clone();
    newFlags.clear(pos);
    return new BitSetBitMask(false, newFlags);
  }

  /**
   * Flip the flag at the given position.
   *
   * @param pos position
   * @return bit mask where the flag at the given position is flipped
   */
  @NotNull
  @Override
  public BitMask flip(int pos)
  {
    final BitSet newFlags = (BitSet)bitSet.clone();
    newFlags.flip(pos);
    return new BitSetBitMask(false, newFlags);
  }

  /**
   * Get the inverse of this bit mask.
   *
   * @return inverse bit mask
   */
  @NotNull
  @Override
  public BitMask not()
  {
    BitSet newFlags = (BitSet)bitSet.clone();
    newFlags.flip(0, newFlags.length());
    return new BitSetBitMask(false, newFlags);
  }

  /**
   * Get the result of a logical <b>and</b> of this bit mask and another.
   *
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  @Override
  public BitMask and(@NotNull BitMask other)
  {
    BitSet newFlags = (BitSet)bitSet.clone();
    newFlags.and(other.toBitSet());
    return new BitSetBitMask(false, newFlags);
  }

  /**
   * Get the result of a logical <b>and</b> of this bit mask and the inverse of another.
   *
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  @Override
  public BitMask andNot(@NotNull BitMask other)
  {
    BitSet newFlags = (BitSet)bitSet.clone();
    newFlags.andNot(other.toBitSet());
    return new BitSetBitMask(false, newFlags);
  }

  /**
   * Get the result of a logical <b>or</b> of this bit mask and another.
   *
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  @Override
  public BitMask or(@NotNull BitMask other)
  {
    BitSet newFlags = (BitSet)bitSet.clone();
    newFlags.or(other.toBitSet());
    return new BitSetBitMask(false, newFlags);
  }

  /**
   * Get the result of a logical <b>xor</b> (exclusive or) of this bit mask and another.
   *
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  @Override
  public BitMask xor(@NotNull BitMask other)
  {
    BitSet newFlags = (BitSet)bitSet.clone();
    newFlags.xor(other.toBitSet());
    return new BitSetBitMask(false, newFlags);
  }

  /**
   * Convert this bit mask into a bit set.
   *
   * @return bit set
   */
  @NotNull
  @Override
  public BitSet toBitSet()
  {
    return (BitSet)bitSet.clone();
  }

  /**
   * Get the lower 32 bits of this mask.
   *
   * @return lower 32 bits as an int
   */
  @Override
  public int low32()
  {
    return (int)low64();
  }

  /**
   * Get the lower 64 bits of this mask.
   *
   * @return lower 64 bits
   */
  @Override
  public long low64()
  {
    return getBitCount() > 0
            ? bitSet.toLongArray()[0]
            : 0L;
  }

  @Override
  public int getLowestBitSet()
  {
    return bitSet.nextSetBit(0);
  }

  @Override
  public int getHighestBitSet()
  {
    return bitSet.previousSetBit(bitSet.length() - 1);
  }

  @NotNull
  @Override
  public BitMask cleared()
  {
    return ZERO;
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (!(o instanceof BitMask)) {
      return false;
    }

    return BitMaskUtil.areEqual(this, (BitMask)o);
  }

  @Override
  public int hashCode()
  {
    int hash = 0;
    for (long v : bitSet.toLongArray()) {
      hash ^= BitMaskUtil.getHash64(v);
    }
    return hash;
  }

  /**
   * Returns a string representation of the object. In general, the
   * {@code toString} method returns a string that
   * "textually represents" this object. The result should
   * be a concise but informative representation that is easy for a
   * person to read.
   * It is recommended that all subclasses override this method.
   * <p>
   * The {@code toString} method for class {@code Object}
   * returns a string consisting of the name of the class of which the
   * object is an instance, the at-sign character `{@code @}', and
   * the unsigned hexadecimal representation of the hash code of the
   * object. In other words, this method returns a string equal to the
   * value of:
   * <blockquote>
   * <pre>
   * getClass().getName() + '@' + Integer.toHexString(hashCode())
   * </pre></blockquote>
   *
   * @return a string representation of the object.
   */
  @Override
  public String toString()
  {
    return String.format("<%s>", bitSet);
  }
}
