// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;

import java.util.BitSet;

/**
 * Helper class for bit masks.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class BitMaskUtil
{
  /** Maks for the lowest 32 bits. */
  private static final long MASK32 = 0xFFFFFFFFL;
  /** Don't create. */
  private BitMaskUtil() {}

  /**
   * Get a hash value for a long value.
   * @param value long value
   * @return hash value
   */
  public static int getHash64(long value)
  {
    return (int)((value >>> 32) & MASK32) ^ (int)(value & MASK32);
  }

  /**
   * Are two bit masks equal?
   * <p>
   * This compares whether the same bit flags are set in both masks.
   * For masks with different lengths the missing flags in the shorter mask
   * are considered to be unset.
   *
   * @param mask1 first mask
   * @param mask2 second mask
   * @return {@code true} if both masks are considered equal, otherwise {@code false}
   */
  public static boolean areEqual(@NotNull BitMask mask1,
                                 @NotNull BitMask mask2)
  {
    BitMask shorter;
    BitMask longer;
    if (mask1.getBitCount() >= mask2.getBitCount()) {
      longer = mask1;
      shorter = mask2;
    }
    else {
      longer = mask2;
      shorter = mask1;
    }
    for (int i = longer.getBitCount() - 1;  i >= shorter.getBitCount();  --i) {
      if (longer.isSet(i)) {
        return false;
      }
    }
    for (int i = shorter.getBitCount() - 1;  i >= 0;  --i) {
      if (shorter.isSet(i) != longer.isSet(i)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Are two enum bit masks equal?
   *
   * @param mask1 first mask
   * @param mask2 second mask
   * @param <T> enum bit flag
   * @return {@code true} if both masks are considered equal, otherwise {@code false}
   */
  public static <T extends Enum<T> & BitFlag> boolean areEqual(@NotNull EnumBitMask<T> mask1,
                                                               @NotNull EnumBitMask<T> mask2)
  {
    return areEqual(mask1.getBitMask(), mask2.getBitMask());
  }

  /**
   * Convert a bit flag into a bit mask.
   * @param flag bit flag
   * @return bit mask with the given flag set and all others unset
   */
  @NotNull
  public static BitMask toBitMask(@NotNull BitFlag flag)
  {
    final BitMask zero = new BitMask() {
      @Override
      public boolean isSet(int pos)
      {
        return false;
      }

      @Override
      public int getBitCount()
      {
        return 0;
      }

      @Override
      public int getCardinality()
      {
        return 0;
      }

      @Override
      public boolean isEmpty()
      {
        return true;
      }

      @NotNull
      @Override
      public BitMask set(int pos)
      {
        final SimpleBitFlag flag = SimpleBitFlag.get(pos);
        return flag.toBitMask();
      }

      @NotNull
      @Override
      public BitMask clear(int pos)
      {
        return BitMask16.ZERO;
      }

      @NotNull
      @Override
      public BitMask flip(int pos)
      {
        return set(pos);
      }

      @NotNull
      @Override
      public BitMask not()
      {
        throw new IllegalAccessError("Not implemented!");
      }

      @NotNull
      @Override
      public BitMask and(@NotNull BitMask other)
      {
        return BitMask16.ZERO;
      }

      @NotNull
      @Override
      public BitMask andNot(@NotNull BitMask other)
      {
        return BitMask16.ZERO;
      }

      @NotNull
      @Override
      public BitMask or(@NotNull BitMask other)
      {
        return other;
      }

      @NotNull
      @Override
      public BitMask xor(@NotNull BitMask other)
      {
        return other;
      }

      @NotNull
      @Override
      public BitSet toBitSet()
      {
        return new BitSet();
      }

      @Override
      public int low32()
      {
        return 0;
      }

      @Override
      public long low64()
      {
        return 0L;
      }

      @Override
      public int getLowestBitSet()
      {
        return -1;
      }

      @Override
      public int getHighestBitSet()
      {
        return -1;
      }

      @NotNull
      @Override
      public BitMask cleared()
      {
        return this;
      }

      @Override
      public String toString()
      {
        return "<>";
      }
    };
    return flag.setIn(zero);
  }
}
