// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.generics.Primitives;

import java.util.BitSet;

/**
 * A bit mask for 64 bit flags.
 * <p>
 * Please note that this class will not handle more than
 * 64 bits. Setting any higher bits will have no result.
 * <p>
 * This class is immutable.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class BitMask64
        implements BitMask
{
  /** The number of bits in this class. */
  public static final int BIT_COUNT = 64;

  /** The zero bit mask. This has all 64 flags unset. */
  public static final BitMask64 ZERO = new BitMask64(0L);
  /** The bit mask where all 64 bits are set. */
  public static final BitMask64 ALL_SET = new BitMask64(~0L);

  private final long flags;

  /**
   * Constructor.
   * @param flags bit mask, only the 16 low order bits are used
   */
  public BitMask64(long flags)
  {
    this.flags = flags;
  }

  /**
   * Is the flag at the given position set?
   *
   * @param pos position (non-negative)
   * @return {@code true}: the flag is set<br>
   * {@code false}: the flag is not set
   */
  @Override
  public boolean isSet(int pos)
  {
    if (pos < 0) {
      throw new IllegalArgumentException("pos needs to be non-negative!");
    }
    if (pos >= BIT_COUNT) {
      return false;
    }
    return (flags >>> pos & 0x01L) != 0;
  }

  /**
   * Get the number of possible bits used in this flag.
   *
   * @return bit count
   */
  @Override
  public int getBitCount()
  {
    return BIT_COUNT;
  }

  /**
   * Get the number of bits set in this flag.
   *
   * @return set bit count
   */
  @Override
  public int getCardinality()
  {
    return Long.bitCount(flags);
  }

  /**
   * Is no flag set?
   *
   * @return {@code true}: if no flag in this bit mask is set<br>
   * {@code false}: if any flag in this bit mask is set
   */
  @Override
  public boolean isEmpty()
  {
    return flags == 0L;
  }

  /**
   * Set the flag at the given position.
   *
   * @param pos position (non-negative)
   * @return bit mask where the flag at the given position is set
   */
  @NotNull
  @Override
  public BitMask set(int pos)
  {
    if (pos < 0) {
      throw new IllegalArgumentException("pos needs to be non-negative!");
    }
    if (pos >= BIT_COUNT || isSet(pos)) {
      return this;
    }
    return new BitMask64(flags | (0x01L << pos));
  }

  /**
   * Clear the flag at the given position.
   *
   * @param pos position
   * @return bit mask where the flag at the given position is cleared
   */
  @NotNull
  @Override
  public BitMask clear(int pos)
  {
    if (pos < 0) {
      throw new IllegalArgumentException("pos needs to be non-negative!");
    }
    if (pos >= BIT_COUNT || !isSet(pos)) {
      return this;
    }
    return new BitMask64(flags & ~(0x01L << pos));
  }

  /**
   * Flip the flag at the given position.
   *
   * @param pos position
   * @return bit mask where the flag at the given position is flipped
   */
  @NotNull
  @Override
  public BitMask flip(int pos)
  {
    if (pos < 0) {
      throw new IllegalArgumentException("pos needs to be non-negative!");
    }
    if (pos >= BIT_COUNT) {
      return this;
    }
    return new BitMask64(flags ^ (0x01L << pos));
  }

  /**
   * Get the inverse of this bit mask.
   *
   * @return inverse bit mask
   */
  @NotNull
  @Override
  public BitMask not()
  {
    return new BitMask64(~flags);
  }

  /**
   * Get the result of a logical <b>and</b> of this bit mask and another.
   *
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  @Override
  public BitMask and(@NotNull BitMask other)
  {
    if (other.getBitCount() > BIT_COUNT) {
      return other.and(this);
    }
    long combined = flags & other.low64();
    if (combined == flags) {
      return this;
    }
    return new BitMask64(combined);
  }

  /**
   * Get the result of a logical <b>and</b> of this bit mask and the inverse of another.
   *
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  @Override
  public BitMask andNot(@NotNull BitMask other)
  {
    if (other.getBitCount() > BIT_COUNT) {
      return other.not().and(this);
    }
    long combined = flags | ~other.low64();
    if (combined == flags) {
      return this;
    }
    return new BitMask64(combined);
  }

  /**
   * Get the result of a logical <b>or</b> of this bit mask and another.
   *
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  @Override
  public BitMask or(@NotNull BitMask other)
  {
    if (other.getBitCount() > BIT_COUNT) {
      return other.or(this);
    }
    long combined = flags | other.low64();
    if (combined == flags) {
      return this;
    }
    return new BitMask64(combined);
  }

  /**
   * Get the result of a logical <b>xor</b> (exclusive or) of this bit mask and another.
   *
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  @Override
  public BitMask xor(@NotNull BitMask other)
  {
    if (other.getBitCount() > BIT_COUNT) {
      return other.xor(this);
    }
    long combined = flags ^ other.low64();
    if (combined == flags) {
      return this;
    }
    return new BitMask64(combined);
  }

  /**
   * Convert this bit mask into a bit set.
   *
   * @return bit set
   */
  @NotNull
  @Override
  public BitSet toBitSet()
  {
    return BitSet.valueOf(new long[] { low64() });
  }

  /**
   * Get the lower 32 bits of this mask.
   *
   * @return lower 32 bits as an int
   */
  @Override
  public int low32()
  {
    return (int)(flags & 0xFFFFFFFFL);
  }

  /**
   * Get the lower 64 bits of this mask.
   *
   * @return lower 64 bits
   */
  @Override
  public long low64()
  {
    return flags;
  }

  @Override
  public int getLowestBitSet()
  {
    return Primitives.positionOfLowestOneBit(flags);
  }

  @Override
  public int getHighestBitSet()
  {
    return Primitives.positionOfHighestOneBit(flags);
  }

  @NotNull
  @Override
  public BitMask cleared()
  {
    return ZERO;
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (!(o instanceof BitMask)) {
      return false;
    }

    return BitMaskUtil.areEqual(this, (BitMask)o);
  }

  @Override
  public int hashCode()
  {
    return BitMaskUtil.getHash64(low64());
  }

  /**
   * Returns a string representation of the object. In general, the
   * {@code toString} method returns a string that
   * "textually represents" this object. The result should
   * be a concise but informative representation that is easy for a
   * person to read.
   * It is recommended that all subclasses override this method.
   * <p>
   * The {@code toString} method for class {@code Object}
   * returns a string consisting of the name of the class of which the
   * object is an instance, the at-sign character `{@code @}', and
   * the unsigned hexadecimal representation of the hash code of the
   * object. In other words, this method returns a string equal to the
   * value of:
   * <blockquote>
   * <pre>
   * getClass().getName() + '@' + Integer.toHexString(hashCode())
   * </pre></blockquote>
   *
   * @return a string representation of the object.
   */
  @Override
  public String toString()
  {
    return String.format("<%08x>", flags);
  }
}
