// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2015-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;

import java.util.BitSet;

/**
 * A set of bit flags.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public interface BitMask
{
  /**
   * Is the flag at the given position set?
   * @param pos position (non-negative)
   * @return {@code true}: the flag is set<br>
   *         {@code false}: the flag is not set
   */
  boolean isSet(int pos);

  /**
   * Set the flag at the given position.
   * @param pos position
   * @return bit mask where the flag at the given position is set
   */
  @NotNull
  BitMask set(int pos);

  /**
   * Clear the flag at the given position.
   * @param pos position
   * @return bit mask where the flag at the given position is cleared
   */
  @NotNull
  BitMask clear(int pos);

  /**
   * Get the number of possible bits used in this flag.
   * @return bit count
   */
  int getBitCount();

  /**
   * Get the number of bits set in this flag.
   * @return set bit count
   */
  int getCardinality();

  /**
   * Is no flag set?
   * @return {@code true}: if no flag in this bit mask is set<br>
   *         {@code false}: if any flag in this bit mask is set
   */
  boolean isEmpty();

  /**
   * Flip the flag at the given position.
   * @param pos position
   * @return bit mask where the flag at the given position is flipped
   */
  @NotNull
  BitMask flip(int pos);

  /**
   * Get the inverse of this bit mask.
   * @return inverse bit mask
   */
  @NotNull
  BitMask not();

  /**
   * Get the result of a logical <b>and</b> of this bit mask and another.
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  BitMask and(@NotNull BitMask other);

  /**
   * Get the result of a logical <b>and</b> of this bit mask and the inverse of another.
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  BitMask andNot(@NotNull BitMask other);

  /**
   * Get the result of a logical <b>or</b> of this bit mask and another.
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  BitMask or(@NotNull BitMask other);

  /**
   * Get the result of a logical <b>xor</b> (exclusive or) of this bit mask and another.
   * @param other other bit mask
   * @return resulting bit mask
   */
  @NotNull
  BitMask xor(@NotNull BitMask other);

  /**
   * Convert this bit mask into a bit set.
   * @return bit set
   */
  @NotNull
  BitSet toBitSet();

  /**
   * Get the lower 32 bits of this mask.
   * @return lower 32 bits as an int
   */
  int low32();

  /**
   * Get the lower 64 bits of this mask.
   * @return lower 64 bits
   */
  long low64();

  /**
   * Get the position of the lowest bit set.
   * @return position of the lowest bit set, or {@code -1} if no bit is set
   */
  int getLowestBitSet();

  /**
   * Get the position of the highest bit set.
   * @return position of the highest bit set, or {@code -1} if no bit is set
   */
  int getHighestBitSet();

  /**
   * Get a bit mask from this where all flags are cleared.
   * @return a bit mask of this type where all flags are cleared
   */
  @NotNull
  BitMask cleared();
}
