// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util;

import de.caff.annotation.NotNull;
import de.caff.i18n.I18n;

import java.util.Locale;

/**
 *  Bit flag enum with internationalized description.
 *  Use this as the base status flag enums.
 */
public abstract class AbstractBasicStatusFlag
{
  /** The extension appended to the i18n base tag if this flag is enabled. */
  public static final String ENABLED  = "_ENABLED";
  /** The extension appended to the i18n base tag if this flag is disabled. */
  public static final String DISABLED = "_DISABLED";
  /** I18n tag for info. */
  private final String basetag;

  /**
   * Constructor.
   * @param basetag i18n tag start. Depending on the value of the flag this will
   */
  protected AbstractBasicStatusFlag(@NotNull String basetag)
  {
    this.basetag = basetag;
  }

  /**
   * Get the i18n tag for the flag description.
   * @return i18n tag
   */
  @NotNull
  public String getBaseTag()
  {
    return basetag;
  }

  /**
   *  Get the i18n tag of this flag depending
   *  whether it is set or not.
   *  @param set is this flag set?
   *  @return complete i18n tag
   */
  @NotNull
  public String getI18nTag(boolean set)
  {
    return basetag + (set ? ENABLED : DISABLED);
  }

  /**
   *  Get the internationalized description of this flag depending
   *  whether it is set or not, using the default locale.
   *  @param set is this flag set?
   *  @return description text
   */
  public String getDescription(boolean set)
  {
    return I18n.getString(getI18nTag(set));
  }

  /**
   *  Get the internationalized description of this flag depending
   *  whether it is set or not.
   *  @param set is this flag set?
   *  @param locale locale to use
   *  @return description text
   */
  public String getDescription(boolean set, Locale locale)
  {
    return I18n.getString(getI18nTag(set), locale);
  }
}
