// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.io.xml;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Empty;
import de.caff.generics.Indexable;
import de.caff.generics.Types;
import de.caff.generics.function.FragileFunction1;
import de.caff.util.ModuleVersion;
import org.w3c.dom.*;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;

/**
 * Helper methods for XML handling.
 * <p>
 * This provides methods for both SAX and DOM handling.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class XmlTool
{
  /** The current version of the library. */
  public static final String RELEASE_DATE = ModuleVersion.getReleaseDate();
  /** Format for mismatch exceptions. */
  private static final String NOT_MASK = "Attribute value '%s' of attribute %s of element %s is %s!";
  /**
   * Version attribute which should be used in a lot of cases.
   * Use an integer as version indicator, then it is possible to
   * use {@linkplain #checkTagAndVersion(Element, String, int, int)}
   * and {@linkplain #checkTagAndVersion(Element, String, int)} to
   * run a fast version check. It is recommended to start versions
   * with {@code 1} which allows to usage of the shorter check method.
   */
  public static final String XML_ATTR_VERSION = "version";

  /** No construction. */
  private XmlTool()
  {
  }

  /**
   * Get the value of the attribute of the given element.
   * Throw an exception if it does not exist.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @return value of attribute
   * @throws SAXException if attribute is non-existent
   */
  @NotNull
  public static String getValue(@NotNull Element element,
                                @NotNull String attributeName) throws SAXException
  {
    final Attr attribute = element.getAttributeNode(attributeName);
    if (attribute == null) {
      throw new SAXException(String.format("Required attribute %s is missing in %s!",
                                           attributeName,
                                           element.getTagName()));
    }
    return attribute.getValue();
  }

  /**
   * Get the value of an optional attribute of the given element.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @return value of attribute, or {@code null} if the attribute does not exist
   */
  @Nullable
  public static String getOptionalValue(@NotNull Element element,
                                        @NotNull String attributeName)
  {
    final Attr attribute = element.getAttributeNode(attributeName);
    return attribute == null
            ? null
            : attribute.getValue();
  }

  /**
   * Get a unwrapped value from an attribute.
   * This assumes that the textual value of the attribute is a string
   * in a certain format which can be converted to an object.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @param extractor     converter from the textual value to the expected value type
   * @param <T> result type of this method
   * @return value of attribute
   * @throws SAXException when the attribute does not exist, or when the extractor throws
   */
  public static <T> T getValue(@NotNull Element element,
                               @NotNull String attributeName,
                               @NotNull Function<String, ? extends T> extractor)
          throws SAXException
  {
    try {
      return extractor.apply(getValue(element, attributeName));
    } catch (Throwable e) {
      throw new SAXException(String.format("Extractor failed for attribute %s in element %s!",
                                           attributeName, element));
    }
  }

  /**
   * Get a unwrapped value from an attribute.
   * This assumes that the textual value of the attribute is a string
   * in a certain format which can be converted to an object of the required type.
   * This method allows to use an extractor which can throw an exception-
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @param extractor     converter from the textual value to the expected value type which may throw an exception
   * @param <T> result type of this method
   * @param <E> exception type of extractor
   * @return value of attribute
   * @throws SAXException when the attribute does not exist, or when the extractor throws an exception
   */
  public static <T, E extends Exception> T getValueF(@NotNull Element element,
                                                     @NotNull String attributeName,
                                                     @NotNull FragileFunction1<? extends T, E, String> extractor)
          throws SAXException
  {
    try {
      return extractor.apply(getValue(element, attributeName));
    } catch (SAXException s) {
      throw s;
    } catch (Exception e) {
      throw new SAXException(String.format("Extractor failed for attribute %s in element %s!",
                                           attributeName, element),
                             e);
    } catch (Throwable t) {
      throw new SAXException(String.format("Extractor failed for attribute %s in element %s!",
                                           attributeName, element));
    }
  }

  /**
   * Get a unwrapped value from an attribute.
   * This assumes that the textual value of the attribute is a string
   * in a certain format which can be converted to an object.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @param extractor     converter from the textual value to the expected value type
   * @param <T> result type of this method
   * @return value of attribute, or {@code null} if the value does not exist
   */
  @Nullable
  public static <T> T getOptionalValue(@NotNull Element element,
                                       @NotNull String attributeName,
                                       @NotNull Function<String, ? extends T> extractor)
  {
    final String value = getOptionalValue(element, attributeName);
    return value != null
            ? extractor.apply(value)
            : null;
  }

  /**
   * Get a unwrapped value from an attribute.
   * This assumes that the textual value of the attribute is a string
   * in a certain format which can be converted to an object of the required type.
   * This method allows to use an extractor which can throw an exception-
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @param extractor     converter from the textual value to the expected value type
   * @param <T> result type of this method
   * @param <E> exception type of extractor
   * @return value of attribute, or {@code null} if the value does not exist
   * @throws E when the extractor throws an exception
   */
  @Nullable
  public static <T, E extends Exception> T getOptionalValueF(@NotNull Element element,
                                                             @NotNull String attributeName,
                                                             @NotNull FragileFunction1<? extends T, E, String> extractor)
          throws E
  {
    final String value = getOptionalValue(element, attributeName);
    return value != null
            ? extractor.apply(value)
            : null;
  }

  /**
   * Get the value of the attribute of the given element.
   * Give back a default value if it does not exist.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @param defaultValue  default value
   * @return value of attribute, or default value
   */
  @NotNull
  public static String getValue(@NotNull Element element,
                                @NotNull String attributeName,
                                @NotNull String defaultValue)
  {
    final Attr attribute = element.getAttributeNode(attributeName);
    if (attribute == null) {
      return defaultValue;
    }
    return attribute.getValue();
  }

  /**
   * Get an integer value from an attribute.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @return value of attribute
   * @throws SAXException if attribute is non-existent or not an integer value
   */
  public static int getIntValue(@NotNull Element element,
                                @NotNull String attributeName) throws SAXException
  {
    final String value = getValue(element, attributeName);
    try {
      return Integer.parseInt(value);
    } catch (NumberFormatException e) {
      throw new SAXException(String.format(NOT_MASK,
                                           value, attributeName, element.getTagName(),
                                           "not an integer"),
                             e);
    }
  }

  /**
   * Get a double value from an attribute.
   * This expects double to be in US format.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @return value of attribute
   * @throws SAXException if attribute is non-existent or not a double value
   */
  public static double getDoubleValue(@NotNull Element element,
                                      @NotNull String attributeName) throws SAXException
  {
    final String value = getValue(element, attributeName);
    try {
      return Double.parseDouble(value);
    } catch (NumberFormatException e) {
      throw new SAXException(String.format(NOT_MASK,
                                           value, attributeName, element.getTagName(),
                                           "not a double"),
                             e);
    }
  }

  /**
   * Get a boolean value from an attribute.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @return boolean value of attribute
   * @throws SAXException if attribute is non-existent or not a boolean value
   */
  public static boolean getBooleanValue(@NotNull Element element,
                                        @NotNull String attributeName) throws SAXException
  {
    final String value = getValue(element, attributeName);
    switch (value.toLowerCase()) {
    case "true":
      return true;
    case "false":
      return false;
    default:
      throw new SAXException(String.format("Not a boolean value: <%s ... %s=\"%s\" ...>", element.getTagName(), attributeName, value));
    }
  }

  /**
   * Get an optional boolean value from an attribute.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @param fallback      default value returned if attribute is not set
   * @return boolean value of attribute
   * @throws SAXException if attribute is not a boolean value
   */
  public static boolean getBooleanValue(@NotNull Element element,
                                        @NotNull String attributeName,
                                        boolean fallback) throws SAXException
  {

    final String value = getOptionalValue(element, attributeName);
    if (value == null) {
      return fallback;
    }
    switch (value.toLowerCase()) {
    case "true":
      return true;
    case "false":
      return false;
    default:
      throw new SAXException(String.format("Not a boolean value: <%s ... %s=\"%s\" ...>", element.getTagName(), attributeName, value));
    }
  }

  /**
   * Get a tri-state boolean value from an attribute.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @return tri-state boolean value of attribute, {@code null} for undefined
   * @throws SAXException if attribute is non-existent or not a boolean value
   */
  @Nullable
  public static Boolean getTriStateValue(@NotNull Element element,
                                         @NotNull String attributeName) throws SAXException
  {
    final String value = getValue(element, attributeName);
    switch (value.toLowerCase()) {
    case "true":
      return true;
    case "false":
      return false;
    case "null":
      return null;
    default:
      throw new SAXException(String.format("Not a tri-state boolean value: <%s ... %s=\"%s\" ...>", element.getTagName(), attributeName, value));
    }
  }

  /**
   * Get a tri-state optional boolean value from an attribute.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @param fallback      default value returned if attribute is not set
   * @return tri-state boolean value of attribute, {@code null} for undefined
   * @throws SAXException if attribute is not a boolean value
   */
  @Nullable
  public static Boolean getTriStateValue(@NotNull Element element,
                                         @NotNull String attributeName,
                                         @Nullable Boolean fallback) throws SAXException
  {

    final String value = getOptionalValue(element, attributeName);
    if (value == null) {
      return fallback;
    }
    switch (value.toLowerCase()) {
    case "true":
      return true;
    case "false":
      return false;
    case "null":
      return null;
    default:
      throw new SAXException(String.format("Not a tri-state boolean value: <%s ... %s=\"%s\" ...>", element.getTagName(), attributeName, value));
    }
  }

  /**
   * Get an enum value from an attribute.
   * @param element       element containing attribute
   * @param attributeName name of attribute
   * @param enumClass     enum class
   * @param <E>           enum type
   * @return enum of given type
   * @throws SAXException if attribute is non-existent or not an enum of the given type
   */
  @NotNull
  public static <E extends Enum<E>> E getEnumValue(@NotNull Element element,
                                                   @NotNull String attributeName,
                                                   @NotNull Class<E> enumClass) throws SAXException
  {
    final String value = getValue(element, attributeName);
    try {
      return Enum.valueOf(enumClass, value);
    } catch (IllegalArgumentException e) {
      throw new SAXException(String.format(NOT_MASK,
                                           value, attributeName, element.getTagName(),
                                           "no known enum value of type "+enumClass),
                             e);
    }
  }

  /**
   * Get the value from attributes.
   * Throw an exception if it does not exist.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @return value of attribute
   * @throws SAXException if attribute is non-existent
   */
  @NotNull
  public static String getValue(@NotNull String elementTag,
                                @NotNull Attributes attributes,
                                @NotNull String attributeName) throws SAXException
  {
    final String value = attributes.getValue(attributeName);
    if (value == null) {
      throw new SAXException(String.format("Required attribute %s is missing in %s!",
                                           attributeName,
                                           elementTag));
    }
    return value;
  }

  /**
   * Get the value of the attribute of the given element.
   * Give back a default value if it does not exist.
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @param defaultValue  default value
   * @return value of attribute, or default value
   */
  @NotNull
  public static String getValue(@NotNull Attributes attributes,
                                @NotNull String attributeName,
                                @NotNull String defaultValue)
  {
    return Types.notNull(attributes.getValue(attributeName), defaultValue);
  }

  /**
   * Get an integer value from an attribute.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @return value of attribute
   * @throws SAXException if attribute is non-existent or not an integer value
   */
  public static int getIntValue(@NotNull String elementTag,
                                @NotNull Attributes attributes,
                                @NotNull String attributeName) throws SAXException
  {
    final String value = getValue(elementTag, attributes, attributeName);
    try {
      return Integer.parseInt(value);
    } catch (NumberFormatException e) {
      throw new SAXException(String.format(NOT_MASK,
                                           value, attributeName, elementTag,
                                           "not an integer"),
                             e);
    }
  }

  /**
   * Get an optional integer value from an attribute.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @param fallback      fallback value used if attribute is not defined
   * @return value of attribute
   * @throws SAXException if attribute is not an integer value
   */
  public static int getIntValue(@NotNull String elementTag,
                                @NotNull Attributes attributes,
                                @NotNull String attributeName,
                                int fallback) throws SAXException
  {
    final String value = getOptionalValue(attributes, attributeName);
    if (value == null) {
      return fallback;
    }
    try {
      return Integer.parseInt(value);
    } catch (NumberFormatException e) {
      throw new SAXException(String.format(NOT_MASK,
                                           value, attributeName, elementTag,
                                           "not an integer"),
                             e);
    }
  }

  /**
   * Get a double value from an attribute.
   * This expects double to be in US format.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @return value of attribute
   * @throws SAXException if attribute is non-existent or not a double value
   */
  public static double getDoubleValue(@NotNull String elementTag,
                                      @NotNull Attributes attributes,
                                      @NotNull String attributeName) throws SAXException
  {
    final String value = getValue(elementTag, attributes, attributeName);
    try {
      return Double.parseDouble(value);
    } catch (NumberFormatException e) {
      throw new SAXException(String.format(NOT_MASK,
                                           value, attributeName, elementTag,
                                           "not a double"),
                             e);
    }
  }

  /**
   * Get an optional double value from an attribute.
   * This expects double to be in US format.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @param fallback      fallback value returned if attribute is not defined
   * @return value of attribute
   * @throws SAXException if attribute is not a double value
   */
  public static double getDoubleValue(@NotNull String elementTag,
                                      @NotNull Attributes attributes,
                                      @NotNull String attributeName,
                                      double fallback) throws SAXException
  {
    final String value = getOptionalValue(attributes, attributeName);
    if (value == null) {
      return fallback;
    }
    try {
      return Double.parseDouble(value);
    } catch (NumberFormatException e) {
      throw new SAXException(String.format(NOT_MASK,
                                           value, attributeName, elementTag,
                                           "not a double"),
                             e);
    }
  }

  /**
   * Get a boolean value from an attribute.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @return boolean value of attribute
   * @throws SAXException if attribute is non-existent or not a boolean value
   */
  public static boolean getBooleanValue(@NotNull String elementTag,
                                        @NotNull Attributes attributes,
                                        @NotNull String attributeName) throws SAXException
  {
    final String value = getValue(elementTag, attributes, attributeName);
    switch (value.toLowerCase()) {
    case "true":
      return true;
    case "false":
      return false;
    default:
      throw new SAXException(String.format("Not a boolean value: <%s ... %s=\"%s\" ...>", elementTag, attributeName, value));
    }
  }

  /**
   * Get an optional boolean value from an attribute.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @param fallback      default value returned if attribute is not set
   * @return boolean value of attribute
   * @throws SAXException if attribute is not a boolean value
   */
  public static boolean getBooleanValue(@NotNull String elementTag,
                                        @NotNull Attributes attributes,
                                        @NotNull String attributeName,
                                        boolean fallback) throws SAXException
  {

    final String value = getOptionalValue(attributes, attributeName);
    if (value == null) {
      return fallback;
    }
    switch (value.toLowerCase()) {
    case "true":
      return true;
    case "false":
      return false;
    default:
      throw new SAXException(String.format("Not a boolean value: <%s ... %s=\"%s\" ...>", elementTag, attributeName, value));
    }
  }

  /**
   * Get a tri-state boolean value from an attribute.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @return tri-state boolean value of attribute, {@code null} if undefined
   * @throws SAXException if attribute is non-existent or not a boolean value
   */
  @Nullable
  public static Boolean getTriStateValue(@NotNull String elementTag,
                                         @NotNull Attributes attributes,
                                         @NotNull String attributeName) throws SAXException
  {
    final String value = getValue(elementTag, attributes, attributeName);
    switch (value.toLowerCase()) {
    case "true":
      return true;
    case "false":
      return false;
    case "null":
      return null;
    default:
      throw new SAXException(String.format("Not a tri-state boolean value: <%s ... %s=\"%s\" ...>", elementTag, attributeName, value));
    }
  }

  /**
   * Get an optional tri-state boolean value from an attribute.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @param fallback      default value returned if attribute is not set
   * @return tri-state boolean value of attribute, {@code null} if undefined
   * @throws SAXException if attribute is not a boolean value
   */
  @Nullable
  public static Boolean getTriStateValue(@NotNull String elementTag,
                                         @NotNull Attributes attributes,
                                         @NotNull String attributeName,
                                         @Nullable Boolean fallback) throws SAXException
  {

    final String value = getOptionalValue(attributes, attributeName);
    if (value == null) {
      return fallback;
    }
    switch (value.toLowerCase()) {
    case "true":
      return true;
    case "false":
      return false;
    case "null":
      return null;
    default:
      throw new SAXException(String.format("Not a tri-state boolean value: <%s ... %s=\"%s\" ...>", elementTag, attributeName, value));
    }
  }

  /**
   * Get an enum value from an attribute.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @param enumClass     enum class
   * @param <E>           enum type
   * @return enum of given type
   * @throws SAXException if attribute is non-existent or not an enum of the given type
   */
  @NotNull
  public static <E extends Enum<E>> E getEnumValue(@NotNull String elementTag,
                                                   @NotNull Attributes attributes,
                                                   @NotNull String attributeName,
                                                   @NotNull Class<E> enumClass) throws SAXException
  {
    final String value = getValue(elementTag, attributes, attributeName);
    try {
      return Enum.valueOf(enumClass, value);
    } catch (IllegalArgumentException e) {
      throw new SAXException(String.format(NOT_MASK,
                                           value, attributeName, elementTag,
                                           "no known enum value of type "+enumClass),
                             e);
    }
  }

  /**  /**
   * Get a unwrapped value from an attribute.
   * This assumes that the textual value of the attribute is a string
   * in a certain format which can be converted to an object.
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @param extractor     converter from the textual value to the expected value type
   * @param <T> result type of this method
   * @return value of attribute
   * @throws SAXException when the attribute does not exist, or when the extractor throws
   */
  public static <T> T getValue(@NotNull String elementTag,
                               @NotNull Attributes attributes,
                               @NotNull String attributeName,
                               @NotNull Function<String, ? extends T> extractor)
          throws SAXException
  {
    try {
      return extractor.apply(getValue(elementTag, attributes, attributeName));
    } catch (Throwable e) {
      throw new SAXException(String.format("Extractor failed for attribute %s in element <%s>!",
                                           attributeName, elementTag));
    }
  }

  /**
   * Get a unwrapped value from an attribute.
   * This assumes that the textual value of the attribute is a string
   * in a certain format which can be converted to an object of the required type.
   * This method allows to use an extractor which can throw an exception-
   * @param elementTag    element tag, used for errors
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @param extractor     converter from the textual value to the expected value type which throws an exception
   * @param <T> result type of this method
   * @param <E> exception type of extractor
   * @return value of attribute
   * @throws SAXException when the attribute does not exist, or when the extractor throws an exception
   */
  public static <T, E extends Exception> T getValueF(@NotNull String elementTag,
                                                     @NotNull Attributes attributes,
                                                     @NotNull String attributeName,
                                                     @NotNull FragileFunction1<? extends T, E, String> extractor)
          throws SAXException
  {
    try {
      return extractor.apply(getValue(elementTag, attributes, attributeName));
    } catch (SAXException s) {
      throw s;
    } catch (Exception e) {
      throw new SAXException(String.format("Extractor failed for attribute %s in element %s!",
                                           attributeName, elementTag),
                             e);
    } catch (Throwable t) {
      throw new SAXException(String.format("Extractor failed for attribute %s in element %s!",
                                           attributeName, elementTag));
    }
  }

  /**
   * Get the value of an optional attribute.
   * @param attributes    attributes of XML element
   * @param attributeName name of requested attribute
   * @return attribute value, or {@code null} if attribute is not set
   */
  @Nullable
  public static String getOptionalValue(@NotNull Attributes attributes,
                                        @NotNull String attributeName)
  {
    return attributes.getValue(attributeName);
  }

  /**
   * Get an unwrapped value from an attribute.
   * This assumes that the textual value of the attribute is a string
   * in a certain format which can be converted to an object.
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @param extractor     converter from the textual value to the expected value type
   * @param <T> result type of this method
   * @return value of attribute, or {@code null} if the value does not exist
   */
  @Nullable
  public static <T> T getOptionalValue(@NotNull Attributes attributes,
                                       @NotNull String attributeName,
                                       @NotNull Function<String, ? extends T> extractor)
  {
    final String value = attributes.getValue(attributeName);
    return value != null
            ? extractor.apply(value)
            : null;
  }

  /**
   * Get a unwrapped value from an attribute.
   * This assumes that the textual value of the attribute is a string
   * in a certain format which can be converted to an object of the required type.
   * This method allows to use an extractor which can throw an exception-
   * @param attributes    attributes of XML element
   * @param attributeName name of attribute
   * @param extractor     converter from the textual value to the expected value type
   * @param <T> result type of this method
   * @param <E> exception type of extractor
   * @return value of attribute, or {@code null} if the value does not exist
   * @throws E when the extractor throws an exception
   */
  @Nullable
  public static <T, E extends Exception> T getOptionalValueF(@NotNull Attributes attributes,
                                                             @NotNull String attributeName,
                                                             @NotNull FragileFunction1<? extends T, E, String> extractor)
          throws E
  {
    final String value = attributes.getValue(attributeName);
    return value != null
            ? extractor.apply(value)
            : null;
  }

  /**
   * Get the single child of an element, require that it is an element itself.
   * @param element parent element
   * @return single child element
   * @throws SAXException if there is none or more than one child, or the single child is not an element
   */
  @NotNull
  public static Element getSingleChild(@NotNull Element element) throws SAXException
  {
    final NodeList subElements = element.getChildNodes();
    if (subElements.getLength() != 1) {
      throw new SAXException(String.format("Expected only a single child for <%s>, but got %d!", element.getTagName(), subElements.getLength()));
    }
    final Node item = subElements.item(0);
    if (item instanceof Element) {
      return (Element)item;
    }
    throw new SAXException(String.format("Expected element as only child of <%s>, but got %s!", element.getTagName(), item));
  }

 /**
   * Get a singular sub element from a given element.
   * This will throw an exception if there is no given sub elements
   * or if there are more than one.
   * @param element      element containing sub element
   * @param subElementTag  name of sub element
   * @return sub element
   * @throws SAXException if sub element is missing or appearing more than once
   */
  @NotNull
  public static Element getSingleChild(@NotNull Element element,
                                       @NotNull String subElementTag) throws SAXException
  {
    final NodeList subElements = element.getElementsByTagName(subElementTag);
    switch (subElements.getLength()) {
    case 0:
      throw new SAXException(String.format("Expected subelement %s is missing in %s!",
                                           subElementTag, element.getTagName()));
    case 1:
      return (Element)subElements.item(0);

    default:
      throw new SAXException(String.format("Subelement %s of element %s may only appear once!",
                                           subElementTag, element.getTagName()));
    }
  }

  /**
   * Get a optional singular sub element from a given element.
   * This will throw an exception if there are more than one.
   * @param element      element containing sub element
   * @param subElementTag  name of sub element
   * @return sub element, {@code null} if there is none
   * @throws SAXException if sub element is missing or appearing more than once
   */
  @Nullable
  public static Element getOptionalSingleChild(@NotNull Element element,
                                               @NotNull String subElementTag) throws SAXException
  {
    final NodeList subElements = element.getElementsByTagName(subElementTag);
    switch (subElements.getLength()) {
    case 0:
      return null;
    case 1:
      try {
        return (Element)subElements.item(0);
      } catch (ClassCastException e) {
        throw new SAXException("Expected Element, but got "+subElements.item(0).getClass().getName());
      }

    default:
      throw new SAXException(String.format("Subelement %s of element %s may only appear once!",
                                           subElementTag, element.getTagName()));
    }
  }

  /**
   * Get the direct children of an element with a given tag.
   * @param element       element for which children are requested
   * @param subElementTag sub element tag
   * @return indexable view of the elements
   * @throws SAXException if the required size is non-negative and not matched
   * @see #getChildren(Element, String, int)
   */
  @NotNull
  public static Indexable<Element> getChildren(@NotNull Element element,
                                               @NotNull String subElementTag)
          throws SAXException
  {
    return getChildren(element, subElementTag, -1);
  }

  /**
   * Get the direct children of an element with a given tag.
   * @param element       element for which children are requested
   * @param subElementTag sub element tag
   * @param requiredSize  required and checked size if non-negative, ignored if negative
   * @return indexable view of the elements
   * @throws SAXException if the required size is non-negative and not matched
   */
  @NotNull
  public static Indexable<Element> getChildren(@NotNull Element element,
                                               @NotNull String subElementTag,
                                               int requiredSize)
          throws SAXException
  {
    final NodeList subElements = element.getElementsByTagName(subElementTag);
    final int size = subElements.getLength();
    if (requiredSize >= 0 && requiredSize != size) {
      throw new SAXException(String.format("<%s>: Required %d sub elements with tag <%s>, but got %d!",
                                           element.getTagName(), requiredSize, subElements, size));
    }
    if (size == 0) {
      return Indexable.emptyIndexable();
    }
    return Indexable.viewByIndex(size, idx -> (Element)subElements.item(idx));
  }

  /**
   * Get the text content of the given element.
   * This will throw an exception if the element does not have any content,
   * or if it has sub elements.
   * @param element element containing content
   * @param forceNonEmpty force that element tag is not empty, i.e. &lt;tag/%gt;?
   * @return content of element
   * @throws SAXException if there are other nodes than text or attribute,
   *                      or if {@code forceContent} is {@code true} and
   *                      the element uses an empty element tag
   */
  @NotNull
  public static String getContent(@NotNull Element element, boolean forceNonEmpty)
          throws SAXException
  {
    final NodeList nodes = element.getChildNodes();
    final int length = nodes.getLength();
    String result = null;
    for (int n = 0;  n < length;  ++n) {
      final Node node = nodes.item(n);
      switch (node.getNodeType()) {
      case Node.ATTRIBUTE_NODE:
        // this is okay
        break;

      case Node.TEXT_NODE:
        if (result == null) {
          result = ((Text)node).getWholeText();
        }
        break;

      case Node.CDATA_SECTION_NODE:
        if (result == null) {
          result = node.getTextContent().trim() + "\n";
        }
        else {
          result += node.getTextContent().trim() + "\n";
        }
        break;

      default:
        throw new SAXException(String.format("Unexpected sub node type %d for text element %s!",
                                             node.getNodeType(), element.getTagName()));
      }
    }
    if (result == null) {
      if (forceNonEmpty) {
        throw new SAXException(String.format("Empty element not allowed for text element %s!",
                                             element.getTagName()));
      }
      result = Empty.STRING;
    }
    return result;
  }

  /**
   * Get the child elements with a given name.
   * @param element element for which child elements are requested
   * @param tagName tag name of requested children
   * @return list of child elements using the given tag name
   */
  @NotNull
  public static List<Element> getChildElements(@NotNull Element element, @NotNull String tagName)
  {
    final NodeList nodes = element.getElementsByTagName(tagName);
    final int length = nodes.getLength();
    if (length == 0) {
      return Collections.emptyList();
    }
    final List<Element> result = new ArrayList<>(length);
    for (int i = 0;  i < length;  ++i) {
       result.add((Element)nodes.item(i));
    }
    return Collections.unmodifiableList(result);
  }

  /**
   * Load a document from a generic input source.
   * @param inputSource inputSource, providing XML data in text form
   * @return document contained the source
   * @throws ParserConfigurationException on parser config errors
   * @throws IOException on read errors
   * @throws SAXException on format errors
   */
  @NotNull
  public static Document loadFromSource(@NotNull InputSource inputSource)
          throws ParserConfigurationException, IOException, SAXException
  {
    final DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
    final DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
    final Document doc = dBuilder.parse(inputSource);
    doc.getDocumentElement().normalize();
    return doc;
  }

  /**
   * Load a document from a file or URL.
   * @param path path of XML file
   * @return document contained in file
   * @throws ParserConfigurationException on parser config errors
   * @throws IOException on read errors
   * @throws SAXException on format errors
   */
  @NotNull
  public static Document loadFromFile(@NotNull String path)
          throws ParserConfigurationException, IOException, SAXException
  {
    final DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
    final DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
    final Document doc = dBuilder.parse(path);
    doc.getDocumentElement().normalize();
    return doc;
  }

  /**
   * Load a document from a file.
   * @param path path of XML file
   * @return document contained in file
   * @throws ParserConfigurationException on parser config errors
   * @throws IOException on read errors
   * @throws SAXException on format errors
   */
  @NotNull
  public static Document loadFromFile(@NotNull File path)
          throws ParserConfigurationException, IOException, SAXException
  {
    return loadFromSource(new InputSource(path.toURI().toString()));
  }

  /**
   * Load a document from a text string.
   * @param xmlText textual representation of XML data
   * @return document contained in data
   * @throws ParserConfigurationException on parser config errors
   * @throws IOException on read errors
   * @throws SAXException on format errors
   */
  @NotNull
  public static Document loadFromText(@NotNull String xmlText)
          throws ParserConfigurationException, IOException, SAXException
  {
    return loadFromSource(new InputSource(new StringReader(xmlText)));
  }

  /**
   * Check that the given element has a defined tag name and contains an integer
   * {@link #XML_ATTR_VERSION version} attribute which is not higher than a given
   * number. This method assumes that the minimal version number is {@code 1}.
   * @param element    element to check
   * @param tag        required tag name of the element
   * @param maxVersion required maximal version of the element
   * @return the actual version found
   * @throws SAXException if element has another tag name, no integer version attribute, or
   *                      if the version is too high
   */
  public static int checkTagAndVersion(@NotNull Element element,
                                       @NotNull String tag,
                                       int maxVersion)
          throws SAXException
  {
    return checkTagAndVersion(element, tag, 1, maxVersion);
  }

  /**
   * Check that the given element has a defined tag name and contains an integer
   * {@link #XML_ATTR_VERSION version} attribute which is inside a given range.
   * @param element    element to check
   * @param tag        required tag name of the element
   * @param minVersion required minimal version of the element
   * @param maxVersion required maximal version of the element
   * @return the actual version found
   * @throws SAXException if element has another tag name, no integer version attribute, or
   *                      if the version is too high
   */
  public static int checkTagAndVersion(@NotNull Element element,
                                       @NotNull String tag,
                                       int minVersion,
                                       int maxVersion)
          throws SAXException
  {
    if (!tag.equals(element.getTagName())) {
      throw new SAXException(String.format("Expected element <%s>, but found <%s>!",
                                           tag, element.getTagName()));
    }

    final int version = getIntValue(element, XML_ATTR_VERSION);
    if (version < minVersion) {
      throw new SAXException(String.format("Reading only versions beginning with %d, but found version %d!",
                                           maxVersion, version));
    }
    if (version > maxVersion) {
      throw new SAXException(String.format("Reading only versions up to %d, but found version %d!",
                                           maxVersion, version));
    }

    return version;
  }
}
