// ============================================================================
// File:               DispatchableXmlStorable
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            6/15/23 12:30 PM
//=============================================================================
package de.caff.io.xml;

import de.caff.annotation.NotNull;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Objects;
import java.util.regex.Pattern;

/**
 * Interface for objects of varying classes which can be stored to XML.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since June 15, 2023
 */
public interface XmlDispatchedStorable
        extends XmlStorable
{
  @NotNull
  String XML_ATTR_READ_METHOD = "reader";

  /**
   * Write this dispatched storable to XML using the given XML tag.
   * This writes out an outer element with the given tag and version
   * and reader attributes, which contains whatever {@link #storeInnerXml(SimpleXmlWriter)}
   * is creating.
   * <p>
   * This method is assumed to be called from parent class/interfaces implementations.
   * @param xml     XML writer
   * @param xmlTag  XML tag
   */
  default void storeOuterXml(@NotNull SimpleXmlWriter xml,
                             @NotNull String xmlTag)
  {
    xml.begin(xmlTag,
              XmlTool.XML_ATTR_VERSION, 1,
              XML_ATTR_READ_METHOD, getXmlReaderMethod());
    storeInnerXml(xml);
    xml.end(xmlTag);
  }

  /**
   * Store the individual XML
   * @param xml XML writer
   */
  void storeInnerXml(@NotNull SimpleXmlWriter xml);

  /**
   * Get the method used to reread an object of this kind from XML.
   * The method is expected to have one {@link org.w3c.dom.Element} argument
   * and return an object of the stored kind.
   * <p>
   * The reader is expected to ignore the exact element and restore the object
   * from the children of the given element. This is the reverse of what has
   * happened during {@linkplain #storeInnerXml(SimpleXmlWriter)}.
   * @return XML restore method in the form {@code "full.class.name#methodName"}
   */
  @NotNull
  String getXmlReaderMethod();

  /**
   * Recreate an object of the given class from its stored XML representation.
   * @param element outer XML element as created by {@link #storeOuterXml(SimpleXmlWriter, String)}
   * @param xmlTag  outer XML tag used for storage
   * @param type    type of stored object
   * @return recreated object of the given type
   * @param <T> type of object
   * @throws SAXException on read, access or format errors
   */
  @NotNull
  static <T> T fromXmlElement(@NotNull Element element,
                              @NotNull String xmlTag,
                              @NotNull  Class<T> type)
          throws SAXException
  {
    XmlTool.checkTagAndVersion(element, xmlTag, 1);

    final Element concreteElement = XmlTool.getSingleChild(element);

    final String readMethod = XmlTool.getValue(element, XML_ATTR_READ_METHOD);
    final String[] parts = readMethod.split(Pattern.quote("#"));
    if (parts.length != 2) {
      throw new SAXException(String.format("Expecting argument in the form full.class.name#methodName, but got %s!",
                                           readMethod));
    }
    try {
      final Class<?> readClass = Class.forName(parts[0]);
      final Method method = readClass.getDeclaredMethod(parts[1], Element.class);
      if (!Modifier.isStatic(method.getModifiers())) {
        throw new SAXException(String.format("Expecting a method %s to be static!", readMethod));
      }
      final Object obj = method.invoke(null, concreteElement);
      try {
        return type.cast(Objects.requireNonNull(obj));
      } catch (ClassCastException e) {
        throw new SAXException(String.format("Recreate object is of type %s, but required type is %s!",
                                             obj.getClass().getName(), type.getName()));
      }
    } catch (ClassNotFoundException e) {
      throw new SAXException(String.format("Could not find class %s necessary to recreate parametric curve!",
                                           parts[0]),
                             e);
    } catch (NoSuchMethodException e) {
      throw new SAXException(String.format("Could not find method %s.%s(Element)!", parts[0], parts[1]),
                             e);
    } catch (InvocationTargetException e) {
      final Throwable targetException = e.getTargetException();
      if (targetException instanceof SAXException) {
        throw (SAXException)targetException;
      }
      if (targetException instanceof Exception) {
        throw new SAXException(String.format("Parametric curve recreation by %s failed with unexpected exception!",
                                             readMethod),
                               (Exception)targetException);
      }
      throw new SAXException(String.format("Parametric curve recreation by %s failed with unexpected exception!",
                                           readMethod),
                             e);
    } catch (IllegalAccessException e) {
      throw new SAXException(String.format("Cannot access reader method %s for recreating parametric curve!",
                                           readMethod),
                             e);
    }
  }
}
