// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2022-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.io.json;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.*;
import de.caff.io.OutputStreamStringCollector;
import de.caff.io.StringCollector;

import java.io.Closeable;
import java.io.IOException;
import java.util.Stack;
import java.util.function.*;

/**
 * Simple JSON writer.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since 2022/04/22
 */
public class SimpleJsonWriter implements Closeable
{
  /** Currently open items (objects or arrays). */
  @NotNull
  private final Stack<Json.Env> openItems = new Stack<>();

  /** String output collector. */
  @NotNull
  private final StringCollector stringCollector;

  /** Indentation (only used if prettify is true, in which case it is not null). */
  @Nullable
  private final String basicIndent;

  /** Current indentation (if prettify is true). */
  @NotNull
  private String indentation;

  /** Add new lines (and possibly indentation)? */
  private final boolean prettify;

  /** Current writing state. */
  private State state;

  /**
   * Constructor.
   * This writes without any superfluous white space.
   * @param stringCollector string collector to write to
   */
  public SimpleJsonWriter(@NotNull StringCollector stringCollector)
  {
    this(stringCollector, 0);
  }

  /**
   * Constructor.
   * @param stringCollector string collector to write to
   * @param basicIndent     basic indentation, use {@code -1} for optimal compactness w/o newlines,
   *                        {@code 0} for no indentation but newlines, or greater for newlines and indentation
   */
  public SimpleJsonWriter(@NotNull StringCollector stringCollector, int basicIndent)
  {
    this(stringCollector, basicIndent, false);
  }

  /**
   * Constructor.
   * @param stringCollector string collector to write to
   * @param basicIndent     basic indentation, use {@code -1} for optimal compactness w/o newlines,
   *                        {@code 0} for no indentation but newlines, or greater for newlines and indentation
   * @param relaxed         by default the writer expects to be started with a value, object, or array.
   *                        If this parameter is {@code true} a key is also allowed.
   */
  public SimpleJsonWriter(@NotNull StringCollector stringCollector, int basicIndent, boolean relaxed)
  {
    this.stringCollector = stringCollector;
    this.basicIndent = basicIndent > 0 ? Types.multiple(" ", basicIndent) : null;
    indentation = Empty.STRING;
    this.prettify = basicIndent >= 0;
    state = relaxed ? State.Relaxed : State.FirstValue;
  }

  /**
   * Prepare writing the next item.
   */
  private void nextItem()
  {
    if (state.needsSeparator(openItems)) {
      stringCollector.add(',');
      if (prettify) {
        stringCollector.add('\n');
        if (basicIndent != null) {
          stringCollector.add(indentation);
        }
      }
    }
  }

  /**
   * Output the key of a key-value pair.
   * @param key key to add
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding a key is currently not allowed
   */
  @NotNull
  public SimpleJsonWriter key(@NotNull String key)
  {
    if (!state.allowsKey) {
      throw new IllegalStateException("Key is not allowed in state " + state);
    }
    nextItem();
    stringCollector.add("\"" + Json.maskForOutput(key) + "\":");
    if (prettify) {
      stringCollector.add(' ');
    }
    state = State.FirstValue; // no sep required
    return this;
  }

  /**
   * Output a raw value.
   * @param value raw value
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding a value is currently not allowed
   */
  private SimpleJsonWriter writeRawValue(@NotNull String value)
  {
    if (!state.allowsValue) {
      throw new IllegalStateException("Value is not allowed in state " + state);
    }
    nextItem();
    stringCollector.add(value);
    state = state.advance(openItems);
    return this;
  }

  /**
   * Output a string value.
   * <p>
   * A string value can appear after a key as part of a key-value pair,
   * inside an array, or as single element.
   * @param value string value
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding a value is currently not allowed
   */
  @NotNull
  public SimpleJsonWriter value(@Nullable String value)
  {
    return value != null
            ? writeRawValue("\"" + Json.maskForOutput(value) + "\"")
            : nul();
  }

  /**
   * Output an integer value.
   * <p>
   * A numeric value can appear after a key as part of a key-value pair,
   * inside an array, or as single element.
   * Please note that all numeric values in JSON are forwarded to
   * 64bit double values, so this method is only for convenience.
   *
   * @param value integer value
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding a value is currently not allowed
   */
  @NotNull
  public SimpleJsonWriter value(int value)
  {
    return writeRawValue(Integer.toString(value));
  }

  /**
   * Output a double value.
   * <p>
   * A numeric value can appear after a key as part of a key-value pair,
   * inside an array, or as single element.
   * Please note that all numeric values in JSON are forwarded to
   * 64bit double values, so if you call this with a {@code long}
   * argument you might not be able to reread this value correctly.
   * @param value double value
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding a value is currently not allowed
   */
  @NotNull
  public SimpleJsonWriter value(double value)
  {
    return writeRawValue(Double.toString(value));
  }

  /**
   * Output a boolean value.
   * <p>
   * A boolean value can appear after a key as part of a key-value pair,
   * inside an array, or as single element if this is enabled for this writer.
   * @param value boolean value
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding a value is currently not allowed
   */
  @NotNull
  public SimpleJsonWriter value(boolean value)
  {
    return writeRawValue(value ? "true" : "false");
  }

  /**
   * Output a {@code null} value.
   * <p>
   * A null value can appear after a key as part of a key-value pair,
   * inside an array, or as single element.
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding a value is currently not allowed
   */
  @NotNull
  public SimpleJsonWriter nul()
  {
    return writeRawValue("null");
  }

  /**
   * Begin grouping (i.e. a JSON object or array).
   * @param groupingOpen grouping open character, i.e. {@code '{'} or {@code '['}
   */
  private void beginGrouping(char groupingOpen)
  {
    stringCollector.add(groupingOpen);
    if (prettify) {
      stringCollector.add('\n');
      if (basicIndent != null) {
        indentation += basicIndent;
        stringCollector.add(indentation);
      }
    }
  }

  /**
   * End grouping (i.e. of a JSON object or array).
   * @param groupingClose grouping open character, i.e. {@code '}'} or {@code ']'}
   */
  private void endGrouping(char groupingClose)
  {
    if (prettify) {
      stringCollector.add('\n');
      if (basicIndent != null) {
        indentation = indentation.substring(basicIndent.length());
        stringCollector.add(indentation);
      }
    }
    stringCollector.add(groupingClose);
  }

  /**
   * Start outputting a JSON object.
   * <p>
   * An object can appear as value of a key-value pair,
   * as element of an array, or as root element.
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an object is currently not allowed
   */
  @NotNull
  public SimpleJsonWriter beginObject()
  {
    if (!state.allowsValue) {
      throw new IllegalStateException("Object is not allowed in state " + state);
    }
    nextItem();
    openItems.push(Json.Env.Object);
    state = State.FirstKey;
    beginGrouping('{');
    return this;
  }

  /**
   * End outputting a JSON object.
   * <p>
   * This will close the currently outputted object, and is only
   * allowed at places where a key is also allowed.
   * @return JSON writer for chaining
   * @throws IllegalStateException if no object is open, if this is called inside an array,
   *                               or when the value of a key-value pair is not yet outputted
   */
  @NotNull
  public SimpleJsonWriter endObject()
  {
    if (openItems.isEmpty()) {
      throw new IllegalStateException("Cannot end object, none is open!");
    }
    final Json.Env top = openItems.pop();
    if (top != Json.Env.Object) {
      throw new IllegalStateException("Cannot end object, open is "+top+".");
    }
    if (openItems.isEmpty()) {
      state = State.None;
    }
    else {
      state = openItems.peek().defaultState;
    }
    endGrouping('}');
    return this;
  }

  /**
   * Start outputting a JSON array.
   * <p>
   * An array can appear as value of a key-value pair,
   * as element of an array, or as root element.
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an object is currently not allowed
   */
  @NotNull
  public SimpleJsonWriter beginArray()
  {
    if (!state.allowsValue) {
      throw new IllegalStateException("Cannot start array in state "+state+"!");
    }
    nextItem();
    openItems.push(Json.Env.Array);
    state = State.FirstValue;
    beginGrouping('[');
    return this;
  }

  /**
   * End outputting a JSON array.
   * <p>
   * This will close the currently outputted array, and is only
   * if currently inside an array.
   * @return JSON writer for chaining
   * @throws IllegalStateException if no array is open, or if this is called inside an object
   */
  @NotNull
  public SimpleJsonWriter endArray()
  {
    if (openItems.isEmpty()) {
      throw new IllegalStateException("Cannot end array, none is open!");
    }
    final Json.Env top = openItems.pop();
    if (top != Json.Env.Array) {
      throw new IllegalStateException("Cannot end object, open is "+top+".");
    }
    if (openItems.isEmpty()) {
      state = State.None;
    }
    else {
      state = openItems.peek().defaultState;
    }
    endGrouping(']');
    return this;
  }

  /**
   * Convenience method for outputting a whole array of strings.
   * @param elements string elements
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public SimpleJsonWriter array(@NotNull String ... elements)
  {
    return array(Indexable.viewArray(elements));
  }

  /**
   * Convenience method for outputting a whole array of strings.
   * @param elements string elements
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public SimpleJsonWriter array(@NotNull Iterable<String> elements)
  {
    beginArray();
    elements.forEach(this::value);
    endArray();
    return this;
  }

  /**
   * Convenience method for outputting a whole array of strings
   * created from general elements.
   * @param elements general elements
   * @param toStringFunction function which creates strings from the elements
   * @return JSON writer for chaining
   * @param <T> element type
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public <T> SimpleJsonWriter stringArray(@NotNull Iterable<T> elements,
                                          @NotNull Function<? super T, String> toStringFunction)
  {
    return array(Types.view(elements, toStringFunction));
  }

  /**
   * Convenience method for outputting a whole array of integers.
   * @param numbers int elements
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public SimpleJsonWriter array(@NotNull int ... numbers)
  {
    return array(IntIndexable.viewArray(numbers));
  }

  /**
   * Convenience method for outputting a whole array of integers.
   * @param numbers int elements
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public SimpleJsonWriter array(@NotNull PrimitiveIntIterable numbers)
  {
    beginArray();
    numbers.forEachInt(this::value);
    endArray();
    return this;
  }

  /**
   * Convenience method for outputting a whole array of numbers as integers.
   * @param numbers number elements
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public SimpleJsonWriter intArray(@NotNull Iterable<? extends Number> numbers)
  {
    return intArray(numbers, Number::intValue);
  }

  /**
   * Convenience method for outputting a whole array of integers
   * created from general elements.
   * @param elements general elements
   * @param intMapping function which creates integers from the elements
   * @return JSON writer for chaining
   * @param <T> element type
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public <T> SimpleJsonWriter intArray(@NotNull Iterable<T> elements,
                                       @NotNull ToIntFunction<? super T> intMapping)
  {
    return array(Types.intView(elements, intMapping));
  }

  /**
   * Convenience method for outputting a whole array of double values.
   * @param numbers double elements
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public SimpleJsonWriter array(@NotNull double ... numbers)
  {
    return array(DoubleIndexable.viewArray(numbers));
  }

  /**
   * Convenience method for outputting a whole array of double values.
   * @param numbers double elements
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public SimpleJsonWriter array(@NotNull PrimitiveDoubleIterable numbers)
  {
    beginArray();
    numbers.forEachDouble(this::value);
    endArray();
    return this;
  }

  /**
   * Convenience method for outputting a whole array of numbers as double values.
   * @param numbers number elements
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public SimpleJsonWriter doubleArray(@NotNull Iterable<? extends Number> numbers)
  {
    return doubleArray(numbers, Number::doubleValue);
  }

  /**
   * Convenience method for outputting a whole array of double values
   * created from general elements.
   * @param elements general elements
   * @param doubleMapping function which creates doubles from the elements
   * @return JSON writer for chaining
   * @param <T> element type
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public <T> SimpleJsonWriter doubleArray(@NotNull Iterable<T> elements,
                                          @NotNull ToDoubleFunction<? super T> doubleMapping)
  {
    return array(Types.doubleView(elements, doubleMapping));
  }

  /**
   * Convenience method for outputting a whole array of boolean flags.
   * @param flags boolean elements
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public SimpleJsonWriter array(@NotNull boolean ... flags)
  {
    return array(BooleanIndexable.viewArray(flags));
  }

  /**
   * Convenience method for outputting a whole array of boolean flags.
   * @param flags boolean elements
   * @return JSON writer for chaining
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public SimpleJsonWriter array(@NotNull PrimitiveBooleanIterable flags)
  {
    beginArray();
    for (PrimitiveBooleanIterator it = flags.booleanIterator();
         it.hasNext(); )
    {
      value(it.nextBool());
    }
    endArray();
    return this;
  }

  /**
   * Convenience method for outputting a whole array of boolean flags
   * created from general elements.
   * @param elements general elements
   * @param boolMapping function which creates booleans from the elements
   * @return JSON writer for chaining
   * @param <T> element type
   * @throws IllegalStateException if adding an array is not allowed
   */
  @NotNull
  public <T> SimpleJsonWriter booleanArray(@NotNull Iterable<T> elements,
                                           @NotNull Predicate<? super T> boolMapping)
  {
    return array(Types.booleanView(elements, boolMapping));
  }

  /**
   * Close this writer.
   */
  @Override
  public void close()
  {
    if (prettify) {
      stringCollector.add('\n');
    }
    while (!openItems.isEmpty()) {
      openItems.pop().close(this);
    }
  }

  /**
   * Test code
   * @param args ignored
   * @throws IOException when writing fails
   */
  public static void main(String[] args)
          throws IOException
  {
    final OutputStreamStringCollector collector = new OutputStreamStringCollector(System.out);
    try (SimpleJsonWriter writer = new SimpleJsonWriter(collector, 2)) {
      writer.beginObject();
      {
        writer.key("name").value("Henrik Nielsen")
                .key("size").value(1.84)
                .key("age").value(42);

        writer.key("address").beginObject()
                .key("street").value("Kastanievej 15, 2, Agerskov")
                .key("city").value("SKANDERBØRG")
                .key("zipcode").value(8660)
                .key("country").value("DENMARK")
                .endObject();
        writer.key("foods").beginArray()
                .value("Pizza")
                .value("Bami Goreng")
                .value("Ginger Icecream")
                .endArray();
        writer.key("payments").beginArray()    // nested arrays
                .array("22.74€", "199.21€", "0.95€")
                .array("42.00€", "1234.56€")
                .endArray();
      }
      writer.endObject();
    }
    System.out.println("\n======================================\n");
    try (SimpleJsonWriter writer = new SimpleJsonWriter(collector, 2, true)) {
      writer.key("test").beginObject();
      {
        writer.key("name").value("Henrik Nielsen")
                .key("size").value(1.84)
                .key("age").value(42);

        writer.key("address").beginObject()
                .key("street").value("Kastanievej 15, 2, Agerskov")
                .key("city").value("SKANDERBØRG")
                .key("zipcode").value(8660)
                .key("country").value("DENMARK")
                .endObject();
        writer.key("foods").beginArray()
                .value("Pizza")
                .value("Bami Goreng")
                .value("Ginger Icecream")
                .endArray();
      }
      writer.endObject();
    }
  }

  /**
   * Internal state.
   */
  enum State
  {
    /**
     * Special start state which allows JSON output which starts with
     * a key or a value.
     */
    Relaxed(true, true) {
      @NotNull
      @Override
      State advance(@NotNull Stack<Json.Env> stack)
      {
        return None;
      }

      @Override
      boolean needsSeparator(@NotNull Stack<Json.Env> stack)
      {
        return false;
      }
    },
    /**
     * State after all open environments are closed.
     */
    None(false, false) {
      @Override
      @NotNull
      State advance(@NotNull Stack<Json.Env> stack)
      {
        return None;
      }

      @Override
      boolean needsSeparator(@NotNull Stack<Json.Env> stack)
      {
        return false;
      }
    },
    /**
     * Expects a value after a key or the first value in an array.
     * No previous separator required.
     * Standard start state for JSON files.
     */
    FirstValue(false, true) {
      @Override
      @NotNull
      State advance(@NotNull Stack<Json.Env> stack)
      {
        return stack.isEmpty()
                ? None
                : stack.peek().defaultState;
      }

      @Override
      boolean needsSeparator(@NotNull Stack<Json.Env> stack)
      {
        return false;
      }
    },
    /**
     * Expects followup value in an array.
     * Separator required.
     */
    Value(false, true) {
      @Override
      @NotNull
      State advance(@NotNull Stack<Json.Env> stack)
      {
        return stack.peek() == Json.Env.Object
                ? Key
                : Value;
      }

      @Override
      boolean needsSeparator(@NotNull Stack<Json.Env> stack)
      {
        return !stack.isEmpty()  &&
               stack.peek() == Json.Env.Array;
      }
    },
    /**
     * Expects first key in an object.
     * No separator required.
     */
    FirstKey(true, false) {
      @NotNull
      @Override
      State advance(@NotNull Stack<Json.Env> stack)
      {
        return Key;
      }

      @Override
      boolean needsSeparator(@NotNull Stack<Json.Env> stack)
      {
        return false;
      }
    },
    /**
     * Expects further key in an object.
     * Separator required.
     */
    Key(true, false) {
      @NotNull
      @Override
      State advance(@NotNull Stack<Json.Env> stack)
      {
        return Key;
      }

      @Override
      boolean needsSeparator(@NotNull Stack<Json.Env> stack)
      {
        return true;
      }
    };

    final boolean allowsKey;
    final boolean allowsValue;

    /**
     * Constructor.
     * @param allowsKey   does this state allow to write a key?
     * @param allowsValue does this state allow to write a value?
     */
    State(boolean allowsKey, boolean allowsValue)
    {
      this.allowsKey = allowsKey;
      this.allowsValue = allowsValue;
    }

    /**
     * Advance to the next state.
     * @param stack current environment stack
     * @return next state
     */
    @NotNull
    abstract State advance(@NotNull Stack<Json.Env> stack);

    /**
     * Is a separator required?
     * @param stack current stack
     * @return {@code true} if a separator is required,
     *         {@code false} if not
     */
    abstract boolean needsSeparator(@NotNull Stack<Json.Env> stack);
  }
}
