// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2022-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.io.json;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Empty;
import de.caff.generics.OrderedPair;
import de.caff.generics.function.BooleanConsumer;
import de.caff.io.TextStringCollector;
import de.caff.util.Utility;

import java.io.*;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.DoubleConsumer;
import java.util.function.IntConsumer;

/**
 * Simple JSON reader.
 * <p>
 * This provides a very basic way to read JSON following RFC 8259.
 * <p>
 * JSON data is decomposed into a stream of simple events
 * which are handled by a user-defined handler which reacts
 * on values or starting of objects and arrays.
 * <p>
 * Because of the recursive nature of JSON handlers may
 * apply sub-handlers by means of {@linkplain #pushSubHandler(Handler)}
 * and {@linkplain #popSubHandler()}. There is no automatic
 * popping, so user code has to take care to pop handlers whenever necessary.
 * The current environment type can be requested with the method
 * {@linkplain #getEnvironment()}. When requested inside
 * {@link Handler#beginObject(SimpleJsonReader)},
 * {@link Handler#endObject(SimpleJsonReader)},
 * {@link Handler#beginArray(SimpleJsonReader)},
 * and {@link Handler#endArray(SimpleJsonReader)} the reader
 * argument will return the surrounding environment.
 * <p>
 * A set of useful default implementations which make use of the above
 * stacking is also provided:
 * <ul>
 *   <li>
 *     {@link SimpleJsonReader.ObjectHandler}: handle the content of an object.
 *     Methods to implement are:
 *     <ul>
 *       <li>
 *         {@link SimpleJsonReader.ObjectHandler#nullValue(String)}: receive null value for the given key
 *       </li>
 *       <li>
 *         {@link SimpleJsonReader.ObjectHandler#value(String, String)}: receive string value for the given key
 *       </li>
 *       <li>
 *         {@link SimpleJsonReader.ObjectHandler#value(String, Number)}: receive numeric value for the given key
 *       </li>
 *       <li>
 *         {@link SimpleJsonReader.ObjectHandler#value(String, boolean)}: receive boolean value for the given key
 *       </li>
 *       <li>
 *         {@link SimpleJsonReader.ObjectHandler#objectValue(String)}: provide a handler for an object value for the given key
 *       </li>
 *       <li>
 *         {@link SimpleJsonReader.ObjectHandler#arrayValue(String)}: provide a handler for an array value for the given
 *       </li>
 *     </ul>
 *   </li>
 *   <li>
 *     {@link SimpleJsonReader.ArrayHandler}: handle the content of an array.
 *     Methods to implement are:
 *     <ul>
 *       <li>
 *         {@link SimpleJsonReader.ArrayHandler#nullElement(int)}: receive null value for the given index
 *       </li>
 *       <li>
 *         {@link SimpleJsonReader.ArrayHandler#element(int, String)}: receive string value for the given index
 *       </li>
 *       <li>
 *         {@link SimpleJsonReader.ArrayHandler#element(int, Number)}: receive numeric value for the given index
 *       </li>
 *       <li>
 *         {@link SimpleJsonReader.ArrayHandler#element(int, boolean)}: receive boolean value for the given index
 *       </li>
 *       <li>
 *         {@link SimpleJsonReader.ObjectHandler#objectValue(String)}: receive provide a handler for an object value for the given key
 *       </li>
 *       <li>
 *         {@link SimpleJsonReader.ObjectHandler#arrayValue(String)}: receive provide a handler for an array value for the given
 *       </li>
 *     </ul>
 *   </li>
 *   <li>
 *     {@link SimpleJsonReader.ValueHandler}: this is useful as the starting handler for a complete JSON file.
 *     By default it is expected that the content is a JSON object, therefore only one method has to be implemented:
 *     <ul>
 *       <li>
 *         {@link SimpleJsonReader.ValueHandler#object()}: provide a handler (usually an {@link ObjectHandler})
 *         for handling the object's content.
 *       </li>
 *     </ul>
 *     But the handler is prepared to read any valid JSON file, therefore the following methods
 *     throw an exception by default, but you can override them to fit your needs:
 *     <ul>
 *       <li>
 *         {@link ValueHandler#array()}: provide a handler (usually an {@link ArrayHandler} for handling
 *         the array's content.
 *       </li>
 *       <li>
 *         {@link ValueHandler#nullValue()}: receive a null value.
 *       </li>
 *       <li>
 *         {@link ValueHandler#value(String)}: receive a string value.
 *       </li>
 *       <li>
 *         {@link ValueHandler#value(Number)}: receive a numeric value.
 *       </li>
 *       <li>
 *         {@link ValueHandler#value(boolean)}: receive a boolean value.
 *       </li>
 *     </ul>
 *   </li>
 * </ul>
 *
 * Also provided is a method to read a JSON file into a {@link Map} with string keys which might come
 * handy for simple files: {@link #readToMap(Reader)}. The created map is an exact mirror of the JSON
 * content. Obviously, it requires JSON files which contain a JSON object.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since 2022/05/01
 */
public class SimpleJsonReader
{
  /** Debug the tokenizer. */
  private static final boolean DEBUG_TOKENS = Utility.getBooleanParameter("json.tokens.dump", false);
  @NotNull
  private final Tokenizer tokenizer;
  @NotNull
  private final Stack<Handler> handlerStack = new Stack<>();

  @NotNull
  private final Stack<OrderedPair<Json.Env, Integer>> environmentStack = new Stack<>();

  /**
   * Private constructor.
   * @param reader  reader to read from
   * @param handler initial handler
   */
  private SimpleJsonReader(@NotNull Reader reader,
                           @NotNull Handler handler)
  {
    tokenizer = new Tokenizer(reader);
    handlerStack.push(handler);
  }

  /**
   * Push a sub handler.
   * The given sub handler will be used for handling until
   * {@link #popSubHandler() popped} or another call to this
   * method happens. Using this method requires handling
   * popping accordingly, this class will not pop anything automatically.
   * @param handler handler to be pushed
   */
  public void pushSubHandler(@NotNull Handler handler)
  {
    handlerStack.push(handler);
  }

  /**
   * Pop a previously pushed sub handler.
   * This will make the handler which was active before pushing the
   * active handler again.
   * @return the popped handler, allowing user code to do basic checks
   */
  @NotNull
  public Handler popSubHandler()
  {
    if (handlerStack.size() <= 1) {
      throw new IllegalStateException("Popping too often!");
    }
    return handlerStack.pop();
  }

  @NotNull
  public Handler getSubHandler()
  {
    if (handlerStack.isEmpty()) {
      throw new IllegalStateException("No subhandler available!");
    }
    return  handlerStack.peek();
  }

  /**
   * Get the current environment.
   * @return environment, or {@code null} for initial and final state
   */
  @Nullable
  public Json.Env getEnvironment()
  {
    return environmentStack.isEmpty()
            ? null
            : environmentStack.peek().first;
  }

  private void read()
    throws IOException
  {
    Token token;
    boolean expectKey = false;
    boolean expectValue = true;
    boolean expectSep = false;
    boolean expectClose = false;
    while ((token = tokenizer.nextToken()) != null) {
      if (DEBUG_TOKENS) {
        System.out.println(token);
      }
      switch (token.type) {
      case Colon:  // colon is consumed in StringValue case
        throw new IOException("Unexpected colon in line " + tokenizer.getLineNumber());
      case Comma:
        if (expectSep) {
          expectValue = environmentStack.peek().first == Json.Env.Array;
          expectKey = !expectValue;
          expectSep = expectClose = false;
        }
        else {
          throw new IOException("Unexpected comma in line " + tokenizer.getLineNumber());
        }
        continue;
      case OpenBrace:
        if (!expectValue) {
          throw new IOException("Unexpected start of object ('{') in line " + tokenizer.getLineNumber());
        }
        expectKey = expectClose = true;
        expectValue = expectSep = false;
        handlerStack.peek().beginObject(this);
        environmentStack.push(OrderedPair.create(Json.Env.Object, tokenizer.getLineNumber()));
        continue;
      case CloseBrace:
        if (!expectClose  ||  environmentStack.isEmpty()  ||  environmentStack.pop().first != Json.Env.Object) {
          throw new IOException("Unexpected end of object ('}') in line " + tokenizer.getLineNumber());
        }
        expectKey = expectValue = false;
        expectSep = expectClose = true;
        handlerStack.peek().endObject(this);
        continue;
      case OpenBracket:
        if (!expectValue) {
          throw new IOException("Unexpected start of array ('[') in line " + tokenizer.getLineNumber());
        }
        expectKey = expectSep = false;
        expectClose = expectValue = true;
        handlerStack.peek().beginArray(this);
        environmentStack.push(OrderedPair.create(Json.Env.Array, tokenizer.getLineNumber()));
        continue;
      case CloseBracket:
        if (!expectClose  ||  environmentStack.isEmpty()  || environmentStack.pop().first != Json.Env.Array) {
          throw new IOException("Unexpected end of array ('}') in line " + tokenizer.getLineNumber());
        }
        expectKey = expectValue = false;
        expectClose = expectSep = true;
        handlerStack.peek().endArray(this);
        continue;
      case NullValue:
        if (!expectValue) {
          throw new IOException(String.format("Unexpected token %s in line %d!",
                                              token, tokenizer.getLineNumber()));
        }
        handlerStack.peek().nullValue(this);
        break;
      case TrueValue:
        if (!expectValue) {
          throw new IOException(String.format("Unexpected token %s in line %d!",
                                              token, tokenizer.getLineNumber()));
        }
        handlerStack.peek().value(this, true);
        break;
      case FalseValue:
        if (!expectValue) {
          throw new IOException(String.format("Unexpected token %s in line %d!",
                                              token, tokenizer.getLineNumber()));
        }
        handlerStack.peek().value(this, false);
        break;
      case IntegerValue:
        if (!expectValue) {
          throw new IOException(String.format("Unexpected token %s in line %d!",
                                              token, tokenizer.getLineNumber()));
        }
        handlerStack.peek().value(this, (Integer)token.value);
        break;
      case DoubleValue:
        if (!expectValue) {
          throw new IOException(String.format("Unexpected token %s in line %d!",
                                              token, tokenizer.getLineNumber()));
        }
        handlerStack.peek().value(this, (Double)token.value);
        break;
      case StringValue:
        if (expectKey) {
          final Token peek = tokenizer.nextToken();
          if (peek == null  ||  peek.type != Token.Type.Colon) {
            throw new IOException(String.format("Expected a key with a colon in line %d, but got %s!",
                                                tokenizer.getLineNumber(), peek));
          }
          expectValue = true;
          expectKey = expectSep = expectClose = false;
          handlerStack.peek().key(this, (String)token.value);
          continue;
        }
        if (!expectValue) {
          throw new IOException(String.format("Unexpected token %s in line %d!",
                                              token, tokenizer.getLineNumber()));
        }
        handlerStack.peek().value(this, (String)token.value);
        break;
      }
      // reaching here only after value is read
      expectValue = expectKey = false;
      expectClose = expectSep = true;
    }
    if (!environmentStack.isEmpty()) {
      final OrderedPair<Json.Env, Integer> env = environmentStack.peek();
      throw new IOException(String.format("Unclosed %s environment started in line %d at EOF!",
                                          env.first, env.second));
    }
  }

  /**
   * Read JSON data from a reader.
   * @param reader  reader to read from
   * @param handler handler which is called with all JSON items in sequence
   * @throws IOException on read or format errors
   * @see #read(InputStream, Charset, Handler)
   * @see #read(InputStream, Handler)
   */
  public static void read(@NotNull Reader reader, @NotNull Handler handler)
    throws IOException
  {
    new SimpleJsonReader(reader, handler).read();
  }

  /**
   * Read JSON from an input stream.
   * @param is       input stream to read from
   * @param charset  charset of the given input stream. Note that RFC 8259 assumes that UTF-8 is uses
   * @param handler  handler which is called with all JSON items in sequence
   * @throws IOException on read or format errors
   * @see #read(InputStream, Handler)
   * @see #read(Reader, Handler)
   */
  public static void read(@NotNull InputStream is, @NotNull Charset charset, @NotNull Handler handler)
          throws IOException
  {
    read(new InputStreamReader(is, charset), handler);
  }

  /**
   * Read JSON from an input stream.
   * This uses {@link StandardCharsets#UTF_8 the UTF-8 charset recommended by RFC 8259}.
   * @param is       input stream to read from
   * @param handler  handler which is called with all JSON items in sequence
   * @throws IOException on read or format errors
   * @see #read(InputStream, Charset, Handler)
   * @see #read(Reader, Handler)
   */
  public static void read(@NotNull InputStream is, @NotNull Handler handler)
          throws IOException
  {
    read(is, StandardCharsets.UTF_8, handler);
  }

  /**
   * Read a JSON file into a Java Map.
   * This assumes that the JSON file contains an object,
   * and will throw {@link IllegalStateException} otherwise.
   * @param reader reader to read from
   * @return map with String keys with values which are either strings, number, booleans,
   *         {@code null} values, arrays of the above value types (which again
   *         can contain any type of element, including arrays and maps), and recursive
   *         maps
   * @throws IOException on read or format errors
   */
  @NotNull
  public static Map<String, Object> readToMap(@NotNull Reader reader)
    throws IOException
  {
    final MapObjectHandler mapHandler = new MapObjectHandler();
    read(reader,
         new ValueHandler() {
           @NotNull
           @Override
           protected TemporaryHandler object()
           {
             return mapHandler;
           }
         });
    return mapHandler.getMap();
  }

  /**
   * Read a JSON file into a Java Map.
   * This assumes that the JSON file contains an object,
   * and will throw {@link IllegalStateException} otherwise.
   * @param is       input stream to read from
   * @param charset  charset of the given input stream
   * @return map with String keys with values which are either strings, number, booleans,
   *         {@code null} values, arrays of the above value types (which again
   *         can contain any type of element, including arrays and maps), and recursive
   *         maps
   * @throws IOException on read or format errors
   */
  @NotNull
  public static Map<String, Object> readToMap(@NotNull InputStream is,
                                              @NotNull Charset charset)
          throws IOException
  {
    return readToMap(new InputStreamReader(is, charset));
  }

  /**
   * Read a JSON file into a Java Map, using the UTF-8 charset recommended by RFC 8259.
   * This assumes that the JSON file contains an object,
   * and will throw {@link IllegalStateException} otherwise.
   * @param is       input stream to read from
   * @return map with String keys with values which are either strings, number, booleans,
   *         {@code null} values, arrays of the above value types (which again
   *         can contain any type of element, including arrays and maps), and recursice
   *         maps
   * @throws IOException on read or format errors
   */
  @NotNull
  public static Map<String, Object> readToMap(@NotNull InputStream is)
          throws IOException
  {
    return readToMap(is, StandardCharsets.UTF_8);
  }

  /**
   * Map object handler which collects the JSON data into a Java map.
   * Map keys are strings. Allowed map values are
   * <ul>
   *   <li>{@code null}</li>
   *   <li>{@link String}</li>
   *   <li>{@link Boolean}</li>
   *   <li>{@link Number}, either {@link Double} or {@link Integer}</li>
   *   <li>{@link Object[]}, where each element is one of the allowed values</li>
   *   <li>{@code Map<String, Object>}, where again each value is one of the allowed values</li>
   * </ul>
   * @see #readToMap(Reader)
   * @see #readToMap(InputStream, Charset)
   * @see #readToMap(InputStream)
   */
  public static class MapObjectHandler
          extends ObjectHandler
  {
    @NotNull
    private final Map<String, Object> map = new LinkedHashMap<>();

    @Override
    public void initialize(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void finish(@NotNull SimpleJsonReader reader)
    {
    }

    @NotNull
    @Override
    protected TemporaryHandler objectValue(@NotNull String key)
    {
      final MapObjectHandler mapHandler = new MapObjectHandler();
      this.map.put(key, mapHandler.map);
      return mapHandler;
    }

    @NotNull
    @Override
    protected TemporaryHandler arrayValue(@NotNull String key)
    {
      return new MapArrayHandler(arr -> map.put(key, arr));
    }

    @Override
    protected void nullValue(@NotNull String key)
    {
      map.put(key, null);
    }

    @Override
    protected void value(@NotNull String key, @NotNull String v)
    {
      map.put(key, v);
    }

    @Override
    protected void value(@NotNull String key, boolean v)
    {
      map.put(key, v);
    }

    @Override
    protected void value(@NotNull String key, @NotNull Number v)
    {
      map.put(key, v);
    }

    @NotNull
    public Map<String, Object> getMap()
    {
      return map;
    }
  }

  private static class MapArrayHandler
          extends ArrayHandler
  {
    @NotNull
    private final Consumer<Object[]> arrayCollector;
    @NotNull
    private final List<Object> elements = new LinkedList<>();

    /**
     * Constructor.
     * @param arrayCollector array collector
     */
    public MapArrayHandler(@NotNull Consumer<Object[]> arrayCollector)
    {
      this.arrayCollector = arrayCollector;
    }

    @Override
    public void initialize(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void finish(@NotNull SimpleJsonReader reader)
    {
      arrayCollector.accept(elements.toArray());
    }

    @NotNull
    @Override
    protected TemporaryHandler objectElement(int index)
    {
      final MapObjectHandler mapObjectHandler = new MapObjectHandler();
      elements.add(mapObjectHandler.map);
      return mapObjectHandler;
    }

    @NotNull
    @Override
    protected TemporaryHandler arrayElement(int index)
    {
      return new MapArrayHandler(elements::add);
    }

    @Override
    protected void nullElement(int index)
    {
      elements.add(null);
    }

    @Override
    protected void element(int index, @NotNull String v)
    {
      elements.add(v);
    }

    @Override
    protected void element(int index, boolean v)
    {
      elements.add(v);
    }

    @Override
    protected void element(int index, @NotNull Number v)
    {
      elements.add(v);
    }
  }

  /**
   * A scanner token.
   */
  private static class Token
  {
    /**
     * Token type.
     * This is already quite near to JSON, the only difference is
     * that JSON does not really care for differencing numbers,
     * and a JSON key is the combination of a {@link #StringValue}
     * followed by a {@link #Colon}.
     */
    private enum Type
    {
      Colon,
      Comma,
      OpenBrace,
      CloseBrace,
      OpenBracket,
      CloseBracket,
      NullValue,
      TrueValue,
      FalseValue,
      IntegerValue,
      DoubleValue,
      StringValue
    }
    /** Token type. */
    @NotNull
    private final Type type;
    /**
     * Token value.
     * Depends on token type.
     */
    @NotNull
    private final Object value;

    Token(@NotNull Type type, @NotNull Object value)
    {
      this.type = type;
      this.value = value;
    }

    public static final Token NULL = new Token(Type.NullValue, "null");
    public static final Token TRUE = new Token(Type.TrueValue, Boolean.TRUE);
    public static final Token FALSE = new Token(Type.FalseValue, Boolean.FALSE);
    public static final Token COLON = new Token(Type.Colon, ':');
    public static final Token COMMA = new Token(Type.Comma, ',');
    public static final Token OPEN_BRACE = new Token(Type.OpenBrace, '{');
    public static final Token CLOSE_BRACE = new Token(Type.CloseBrace, '}');
    public static final Token OPEN_BRACKET = new Token(Type.OpenBracket, '[');
    public static final Token CLOSE_BRACKET = new Token(Type.CloseBracket, ']');

    @Override
    public String toString()
    {
      return value.toString();
    }
  }

  /** Low level tokenizer. */
  private static class Tokenizer
  {
    @NotNull
    private final LineNumberReader reader;

    int nextChar = Integer.MIN_VALUE;

    Tokenizer(@NotNull Reader reader)
    {
      this.reader = reader instanceof LineNumberReader
              ? (LineNumberReader)reader
              : new LineNumberReader(reader);
    }

    /**
     * Get the next character.
     * Takes core of a possible {@link #unget(int)}.
     * @return next character, {@code -1} after EOF
     * @throws IOException on read errors
     */
    private int nextChar() throws IOException
    {
      if (nextChar != Integer.MIN_VALUE) {
        final int ret = nextChar;
        nextChar = Integer.MIN_VALUE;
        return ret;
      }
      return reader.read();
    }

    /**
     * Unget a character.
     * Can be used in cases where reading is going too far.
     * @param ch character to unget
     */
    private void unget(int ch)
    {
      assert nextChar == Integer.MIN_VALUE  ||  nextChar == ch;
      nextChar = ch;
    }

    /**
     * Get the current line number.
     * @return line number
     */
    private int getLineNumber()
    {
      if (nextChar == '\n') {
        return reader.getLineNumber() - 1;
      }
      return reader.getLineNumber();
    }

    /**
     * Get the next token.
     * @return next token, {@code null} at EOF
     * @throws IOException on read or format errors
     */
    @Nullable
    public Token nextToken() throws IOException
    {
      final int next = nextNonWhite();
      switch (next) {
      case -1:
        // end of stream
        return null;
      case ':':
        return Token.COLON;
      case ',':
        return Token.COMMA;
      case '"':
      // case '\'': // RFC 8259 does not allow single quotes
        return readString((char)next);
      case '{':
        return Token.OPEN_BRACE;
      case '}':
        return Token.CLOSE_BRACE;
      case '[':
        return Token.OPEN_BRACKET;
      case ']':
        return Token.CLOSE_BRACKET;
      case 't':
        unget(next);
        return expectWord("true", Token.TRUE );
      case 'f':
        unget(next);
        return expectWord("false", Token.FALSE);
      case 'n':
        unget(next);
        return expectWord("null", Token.NULL);
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
      case '-':
      case '.':
      case '+':
        return readNumber((char)next);
      default:
        throw new IOException("Unexpected character " + (char)next + " in line " + getLineNumber());
      }
    }

    /**
     * Read a word.
     * @param end   end of the word
     * @param token token to be returned if word is indeed read
     * @return token associated with the word
     * @throws IOException on read or format errors, especially if word is not found
     */
    @NotNull
    private Token expectWord(@NotNull String end, @NotNull Token token) throws IOException
    {
      final String read = readLetters();
      if (end.equals(read)) {
        return token;
      }
      throw new IOException("Unexpected keyword "+read+" in line " + getLineNumber());
    }

    /**
     * Read all letters until something else appears.
     * @return letters read
     * @throws IOException on read or format errors
     */
    @NotNull
    private String readLetters() throws IOException
    {
      final StringBuilder sb = new StringBuilder();
      int next = nextChar();
      while (next >= 0  &&  Character.isLetter(next)) {
        sb.append((char)next);
        next = nextChar();
      }
      unget(next);
      return sb.toString();
    }

    /**
     * Read the next character which is not a white space.
     * @return next non-whitespace character, or {@code -1} at EOF
     * @throws IOException on read errors
     */
    private int nextNonWhite() throws IOException
    {
      int next;
      do {
        next = nextChar();
      } while (next >= 0  &&  Character.isWhitespace((char)next));
      return next;
    }

    /**
     * Read a number.
     * @param first first character of number
     * @return number token, either of type {@link Token.Type#IntegerValue}
     *         or {@link Token.Type#DoubleValue}
     * @throws IOException on read or format errors
     */
    @NotNull
    private Token readNumber(char first) throws IOException
    {
      final StringBuilder sb = new StringBuilder();
      if (first != '+') {
        sb.append(first);
      }
      int next = nextChar();
      while (isForFloatingPoint(next)) {
        sb.append((char)next);
        next = nextChar();
      }
      unget(next);
      final String strValue = sb.toString();
      try {
        return new Token(Token.Type.IntegerValue,
                         Integer.parseInt(strValue));
      } catch (NumberFormatException ignored) {
        try {
          return new Token(Token.Type.DoubleValue,
                           Double.parseDouble(strValue));
        } catch (NumberFormatException x) {
          throw new IOException(String.format("Not a number in line %d: %s", getLineNumber(), strValue),
                                x);
        }
      }
    }

    /**
     * Is the given character a JSON digit?
     * @param ch character to check
     * @return {@code true} if it is a digit, {@code false} otherwise
     */
    private static boolean isDigit(int ch)
    {
      return ch <= '9'  &&  ch >= '0';
    }

    /**
     * Is the given character valid in a JSON floating point number representation?
     * @param ch character to check
     * @return {@code true} if the character might appear in a number, {@code false} otherwise
     */
    private static boolean isForFloatingPoint(int ch)
    {
      return isDigit(ch) || ch == '.'
             ||  ch == 'e'  ||  ch == 'E'
             ||  ch == '-'  ||  ch == '+';
    }

    /**
     * Get the value of a hexadecimal digit.
     * @param ch hexadecimal digit
     * @return integer value
     * @throws IOException if {@code ch} is no hexadecimal digit
     */
    private static int hexDigit(int ch) throws IOException
    {
      if (ch >= '0') {
        if (ch <= '9') {
          return ch - '0';
        }
        if (ch >= 'A') {
          if (ch <= 'F') {
            return ch - 'A' + 10;
          }
          if (ch >= 'a'  &&  ch <= 'f') {
            return ch - 'a' + 10;
          }
        }
      }
      throw new IOException(String.format("Invalid hexadecimal digit: '%s'", (char)ch));
    }

    /**
     * Read a string.
     * This method assumes that the initial quote is already read.
     * Special escape sequences are handled according to RFC 8259.
     * @return the string (w/o quotes)
     * @throws IOException on read or format errors
     */
    @NotNull
    private Token readString(char quote) throws IOException
    {
      boolean lastWasBackslash = false;
      final StringBuilder sb = new StringBuilder();
      final int startLine = reader.getLineNumber();
      int ch = reader.read();
      int hexCode = 0;
      int hexCount = 0;
      while (ch != -1) {
        if (hexCount > 0) {
          hexCode = 16 * hexCode + hexDigit(ch);
          if (--hexCount == 0) {
            sb.append((char)hexCode);
          }
        }
        else if (lastWasBackslash) {
          switch (ch) {
          case 'n':
            sb.append('\n');
            break;
          case 'r':
            sb.append('\r');
            break;
          case 't':
            sb.append('\t');
            break;
          case 'f':
            sb.append('\f');
            break;
          case 'b':
            sb.append('\b');
            break;
          case 'u':
            hexCode = 0;
            hexCount = 4;
            break;
          default:
            sb.append((char)ch);
          }
          lastWasBackslash = false;
        }
        else if (ch == quote) {
          return new Token(Token.Type.StringValue,
                           sb.toString());
        }
        else if (ch == '\\') {
          lastWasBackslash = true;
        }
        else {
          sb.append((char)ch);
        }
        ch = reader.read();
      }
      throw new IOException("No ending quote for string starting in line "+startLine);
    }
  }

  /**
   * Low-level Handler for streamed JSON stream.
   * This handler will be called in sequence for each JSON item.
   */
  public interface Handler
  {
    /**
     * Called when a key ways read.
     * Next is either a value, a
     * {@linkplain #beginObject(SimpleJsonReader)},
     * or {@linkplain #beginArray(SimpleJsonReader)} .
     * @param reader reader which has read the key
     * @param key    the value of the key
     */
    void key(@NotNull SimpleJsonReader reader,
             @NotNull String key);

    /**
     * Called when an object is started.
     * Next is either a
     * {@linkplain #key(SimpleJsonReader, String)}
     * or an {@linkplain #endObject(SimpleJsonReader)}.
     *
     * @param reader reader which is reading the object
     */
    void beginObject(@NotNull SimpleJsonReader reader);

    /**
     * Called when an object has ended.
     * @param reader reader which has read the object
     */
    void endObject(@NotNull SimpleJsonReader reader);

    /**
     * Called when an array is started.
     * @param reader reader which is reading the array
     */
    void beginArray(@NotNull SimpleJsonReader reader);

    /**
     * Called when an array has ended.
     * @param reader reader which has read the array
     */
    void endArray(@NotNull SimpleJsonReader reader);

    /**
     * Called when a null value was read.
     * @param reader reader which has read the value
     */
    void nullValue(@NotNull SimpleJsonReader reader);

    /**
     * Called when a text value was read.
     * @param reader reader which has read the value
     * @param v text value
     */
    void value(@NotNull SimpleJsonReader reader, @NotNull String v);

    /**
     * Called when a boolean value was read.
     * @param reader reader which has read the value
     * @param v boolean value
     */
    void value(@NotNull SimpleJsonReader reader,  boolean v);

    /**
     * Called when a number value was read.
     * @param reader reader which has read the value
     * @param v number value, implementation will provide either
     *          an {@link Integer} (if fit), or a {@link Double}
     */
    void value(@NotNull SimpleJsonReader reader, @NotNull Number v);
  }

  public interface TemporaryHandler
          extends Handler
  {
    /**
     * Called once before the handler is used.
     * @param reader JSON reader access
     */
    void initialize(@NotNull SimpleJsonReader reader);

    /**
     * Called once after the handler is used.
     * @param reader JSON reader access
     */
    void finish(@NotNull SimpleJsonReader reader);
  }

  public static class HandlerWrapper
          implements Handler
  {
    @NotNull
    protected final Handler wrapped;

    public HandlerWrapper(@NotNull Handler wrapped)
    {
      this.wrapped = wrapped;
    }

    @Override
    public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
    {
      wrapped.key(reader, key);
    }

    @Override
    public void beginObject(@NotNull SimpleJsonReader reader)
    {
      wrapped.beginObject(reader);
    }

    @Override
    public void endObject(@NotNull SimpleJsonReader reader)
    {
      wrapped.endObject(reader);
    }

    @Override
    public void beginArray(@NotNull SimpleJsonReader reader)
    {
      wrapped.beginArray(reader);
    }

    @Override
    public void endArray(@NotNull SimpleJsonReader reader)
    {
      wrapped.endArray(reader);
    }

    @Override
    public void nullValue(@NotNull SimpleJsonReader reader)
    {
      wrapped.nullValue(reader);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
    {
      wrapped.value(reader, v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, boolean v)
    {
      wrapped.value(reader, v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
    {
      wrapped.value(reader, v);
    }
  }

  public abstract static class ObjectHandler
          implements TemporaryHandler
  {
    private String lastKey;

    @Override
    public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
    {
      assert Objects.isNull(lastKey);
      lastKey = key;
    }

    @Override
    public void beginObject(@NotNull SimpleJsonReader reader)
    {
      Objects.requireNonNull(lastKey);
      final TemporaryHandler handler = objectValue(lastKey);
      handler.initialize(reader);
      reader.pushSubHandler(new HandlerWrapper(handler)
      {
        int subObjDepth = 1;

        @Override
        public void beginObject(@NotNull SimpleJsonReader reader)
        {
          ++subObjDepth;
          super.beginObject(reader);
        }

        @Override
        public void endObject(@NotNull SimpleJsonReader reader)
        {
          if (--subObjDepth <= 0) {
            final Handler subHandler = reader.popSubHandler();
            assert subHandler == this;
            handler.finish(reader);
            lastKey = null;
            reader.getSubHandler().endObject(reader);
          }
          else {
            super.endObject(reader);
          }
        }
      });
    }

    @Override
    public void endObject(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void beginArray(@NotNull SimpleJsonReader reader)
    {
      Objects.requireNonNull(lastKey);
      final TemporaryHandler handler = arrayValue(lastKey);
      handler.initialize(reader);
      reader.pushSubHandler(new HandlerWrapper(handler) {
        int subArrayDepth = 1;

        @Override
        public void beginArray(@NotNull SimpleJsonReader reader)
        {
          ++subArrayDepth;
          super.beginArray(reader);
        }

        @Override
        public void endArray(@NotNull SimpleJsonReader reader)
        {
          if (--subArrayDepth <= 0) {
            final Handler subHandler = reader.popSubHandler();
            assert subHandler == this;
            handler.finish(reader);
            lastKey = null;
            reader.getSubHandler().endArray(reader);
          }
          else {
            super.endArray(reader);
          }
        }
      });
    }

    @Override
    public void endArray(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void nullValue(@NotNull SimpleJsonReader reader)
    {
      Objects.requireNonNull(lastKey);
      nullValue(lastKey);
      lastKey = null;
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
    {
      Objects.requireNonNull(lastKey);
      value(lastKey, v);
      lastKey = null;
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, boolean v)
    {
      Objects.requireNonNull(lastKey);
      value(lastKey, v);
      lastKey = null;
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
    {
      Objects.requireNonNull(lastKey);
      value(lastKey, v);
      lastKey = null;
    }

    /**
     * Called when a key with the beginning of an object value
     * was read. Expected to return a dedicated reader which is
     * used to read the content of the object. The handler will
     * be initialized with its {@link TemporaryHandler#initialize(SimpleJsonReader)}
     * method, then be used to handle the key-value pairs inside
     * the object. When the object is finished, the handler's
     * {@link TemporaryHandler#finish(SimpleJsonReader)} method
     * is called.
     * <p>
     * If the key is unsupported either throw an exception
     * or return {@link #SKIP_HANDLER} which will ignore
     * the object.
     *
     * @param key key for which the object is read
     * @return dedicated reader for the object, usually another
     *         {@link ObjectHandler}
     */
    @NotNull
    protected abstract TemporaryHandler objectValue(@NotNull String key);

    /**
     * Called when a key with the beginning of an array value
     * was read. Expected to return a dedicated reader which is
     * used to read the content of the array. The handler will
     * be initialized with its {@link TemporaryHandler#initialize(SimpleJsonReader)}
     * method, then be used to handle the key-value pairs inside
     * the object. When the object is finished, the handler's
     * {@link TemporaryHandler#finish(SimpleJsonReader)} method
     * is called.
     * <p>
     * If the key is unsupported either throw an exception
     * or return {@link #SKIP_HANDLER} which will ignore
     * the object.
     *
     * @param key key for which the object is read
     * @return dedicated reader for the object, usually
     *         an {@link ArrayHandler}
     */
    @NotNull
    protected abstract TemporaryHandler arrayValue(@NotNull String key);

    /**
     * Called when a null value was read.
     * @param key key for which the null was read
     */
    protected abstract void nullValue(@NotNull String key);

    /**
     * Called when a text value was read.
     * @param key key for which the text value was read
     * @param v text value
     */
    protected abstract void value(@NotNull String key, @NotNull String v);

    /**
     * Called when a boolean value was read.
     * @param key key for which the boolean value was read
     * @param v boolean value
     */
    protected abstract void value(@NotNull String key,  boolean v);

    /**
     * Called when a number value was read.
     * @param key key for which the number value was read
     * @param v number value, implementation will provide either
     *          an {@link Integer} (if fit), or a {@link Double}
     */
    protected abstract void value(@NotNull String key, @NotNull Number v);
  }

  /**
   * Dedicated array handler.
   */
  public static abstract class ArrayHandler
          implements TemporaryHandler
  {
    private int index = 0;

    @Override
    public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
    {
      throw new IllegalStateException("Not expecting a key inside array elements!");
    }

    @Override
    public void beginObject(@NotNull SimpleJsonReader reader)
    {
      final TemporaryHandler handler = objectElement(index++);
      handler.initialize(reader);
      reader.pushSubHandler(new HandlerWrapper(handler)
      {
        int subObjDepth = 1;

        @Override
        public void beginObject(@NotNull SimpleJsonReader reader)
        {
          ++subObjDepth;
          super.beginObject(reader);
        }

        @Override
        public void endObject(@NotNull SimpleJsonReader reader)
        {
          if (--subObjDepth <= 0) {
            final Handler subHandler = reader.popSubHandler();
            assert subHandler == this;
            handler.finish(reader);
            reader.getSubHandler().endObject(reader);
          }
          else {
            super.endObject(reader);
          }
        }
      });
    }

    @Override
    public void endObject(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void beginArray(@NotNull SimpleJsonReader reader)
    {
      final TemporaryHandler handler = arrayElement(index++);
      handler.initialize(reader);
      reader.pushSubHandler(new HandlerWrapper(handler) {
        int subArrayDepth = 0;

        @Override
        public void beginArray(@NotNull SimpleJsonReader reader)
        {
          ++subArrayDepth;
          super.beginArray(reader);
        }

        @Override
        public void endArray(@NotNull SimpleJsonReader reader)
        {
          if (--subArrayDepth <= 0) {
            final Handler subHandler = reader.popSubHandler();
            assert subHandler == this;
            handler.finish(reader);
            reader.getSubHandler().endArray(reader);
          }
          else {
            super.endArray(reader);
          }
        }
      });
    }

    @Override
    public void endArray(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void nullValue(@NotNull SimpleJsonReader reader)
    {
      nullElement(index++);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
    {
      element(index++, v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, boolean v)
    {
      element(index++, v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
    {
      element(index++, v);
    }

    /**
     * Called when the beginning of an object element
     * was read. Expected to return a dedicated reader which is
     * used to read the content of the object. The handler will
     * be initialized with its {@link TemporaryHandler#initialize(SimpleJsonReader)}
     * method, then be used to handle the key-value pairs inside
     * the object. When the object is finished, the handler's
     * {@link TemporaryHandler#finish(SimpleJsonReader)} method
     * is called.
     * <p>
     * If the key is unsupported either throw an exception
     * or return {@link #SKIP_HANDLER} which will ignore
     * the object.
     *
     * @param index index of the array element which is read
     * @return dedicated reader for the object, usually an
     *         {@link ObjectHandler}
     */
    @NotNull
    protected abstract TemporaryHandler objectElement(int index);

    /**
     * Called when the beginning of an array sub-element
     * was read. Expected to return a dedicated reader which is
     * used to read the content of the array. The handler will
     * be initialized with its {@link TemporaryHandler#initialize(SimpleJsonReader)}
     * method, then be used to handle the key-value pairs inside
     * the object. When the object is finished, the handler's
     * {@link TemporaryHandler#finish(SimpleJsonReader)} method
     * is called.
     * <p>
     * If the key is unsupported either throw an exception
     * or return {@link #SKIP_HANDLER} which will ignore
     * the object.
     *
     * @param index index of the array element which is read
     * @return dedicated reader for the object, usually
     *         another {@link ArrayHandler}
     */
    @NotNull
    protected abstract TemporaryHandler arrayElement(int index);

    /**
     * Called when a null element was read.
     * @param index index of the array element which is read
     */
    protected abstract void nullElement(int index);

    /**
     * Called when a text element was read.
     * @param index index of the array element which is read
     * @param v text value
     */
    protected abstract void element(int index, @NotNull String v);

    /**
     * Called when a boolean element was read.
     * @param index index of the array element which is read
     * @param v boolean value
     */
    protected abstract void element(int index,  boolean v);

    /**
     * Called when a number value was read.
     * @param index index of the array element which is read
     * @param v number value, implementation will provide either
     *          an {@link Integer} (if fit), or a {@link Double}
     */
    protected abstract void element(int index, @NotNull Number v);
  }

  /**
   * Handler which just reads a single value.
   * <p>
   * This is not expected to be used as a sub handler, but
   * as the initial handler to read a complete JSON file.
   * Because most JSON files just contain an object the
   * only method which has to be implemented is
   * {@link ValueHandler#object()} which should return
   * a dedicated {@link ObjectHandler}. If other types
   * of JSON data can appear override the associated methods
   * which by default will throw exceptions when called.
   */
  public static abstract class ValueHandler
          implements Handler
  {
    private boolean called;

    @Override
    public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
    {
      throw new IllegalStateException("Expecting a value but got key "+key+"!");
    }

    private void checkCalled()
    {
      if (called) {
        throw new IllegalStateException("Value handler used for more than one value!");
      }
      called = true;
    }

    @Override
    public void beginObject(@NotNull SimpleJsonReader reader)
    {
      checkCalled();
      final TemporaryHandler handler = object();
      handler.initialize(reader);
      reader.pushSubHandler(new HandlerWrapper(handler)
      {
        int subObjDepth = 1;

        @Override
        public void beginObject(@NotNull SimpleJsonReader reader)
        {
          ++subObjDepth;
          super.beginObject(reader);
        }

        @Override
        public void endObject(@NotNull SimpleJsonReader reader)
        {
          if (--subObjDepth <= 0) {
            final Handler subHandler = reader.popSubHandler();
            assert subHandler == this;
            handler.finish(reader);
            reader.getSubHandler().endObject(reader);
          }
          else {
            super.endObject(reader);
          }
        }
      });
    }

    @Override
    public void endObject(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void beginArray(@NotNull SimpleJsonReader reader)
    {
      checkCalled();
      final TemporaryHandler handler = array();
      handler.initialize(reader);
      reader.pushSubHandler(new HandlerWrapper(handler) {
        int subArrayDepth = 0;

        @Override
        public void beginArray(@NotNull SimpleJsonReader reader)
        {
          ++subArrayDepth;
          super.beginArray(reader);
        }

        @Override
        public void endArray(@NotNull SimpleJsonReader reader)
        {
          if (--subArrayDepth <= 0) {
            final Handler subHandler = reader.popSubHandler();
            assert subHandler == this;
            handler.finish(reader);
          }
          else {
            super.endArray(reader);
          }
        }
      });
    }

    @Override
    public void endArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Not expecting to reach here!");
    }

    @Override
    public void nullValue(@NotNull SimpleJsonReader reader)
    {
      checkCalled();
      nullValue();
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
    {
      checkCalled();
      value(v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, boolean v)
    {
      checkCalled();
      value(v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
    {
      checkCalled();
      value(v);
    }

    /**
     * Called when the beginning of an object value
     * was read. Expected to return a dedicated reader which is
     * used to read the content of the object. The handler will
     * be initialized with its {@link TemporaryHandler#initialize(SimpleJsonReader)}
     * method, and then be used to handle the key-value pairs inside
     * the object. When the object is finished, the handler's
     * {@link TemporaryHandler#finish(SimpleJsonReader)} method
     * is called.
     *
     * @return dedicated reader for the object, usually an
     *         {@link ObjectHandler}
     */
    @NotNull
    protected abstract TemporaryHandler object();

    /**
     * Called when the beginning of an array value
     * was read. Expected to return a dedicated reader which is
     * used to read the content of the array. The handler will
     * be initialized with its {@link TemporaryHandler#initialize(SimpleJsonReader)}
     * method, then be used to handle the key-value pairs inside
     * the object. When the object is finished, the handler's
     * {@link TemporaryHandler#finish(SimpleJsonReader)} method
     * is called.
     * <p>
     * Has to be overridden if an array is a valid value.
     * This default implementation will throw an {@link IllegalStateException}.
     *
     * @return dedicated reader for the object, usually
     *         an {@link ArrayHandler}
     */
    @NotNull
    protected TemporaryHandler array()
    {
      throw new IllegalStateException("Not expecting an array value!");
    }

    /**
     * Read a null value.
     * Has to be overridden if null is a valid value.
     * This default implementation will throw an {@link IllegalStateException}.
     */
    protected void nullValue()
    {
      throw new IllegalStateException("Not expecting a null value!");
    }

    /**
     * Read a string value.
     * Has to be overridden if a string is a valid value.
     * This default implementation will throw an {@link IllegalStateException}.
     * @param v string value
     */
    protected void value(@NotNull String v)
    {
      throw new IllegalStateException(String.format("Not expecting a string value: %s!", v));
    }

    /**
     * Read a number value.
     * Has to be overridden if a number is a valid value.
     * This default implementation will throw an {@link IllegalStateException}.
     * @param v number value, implementation will either use {@code Integer} or {@code Double}
     */
    protected void value(@NotNull Number v)
    {
      throw new IllegalStateException(String.format("Not expecting a number value: %s!", v));
    }

    /**
     * Read a boolean value.
     * Has to be overridden if a boolean is a valid value.
     * This default implementation will throw an {@link IllegalStateException}.
     * @param v boolean value
     */
    protected void value(boolean v)
    {
      throw new IllegalStateException(String.format("Not expecting a string value: %s!", v));
    }
  }

  /**
   * Handler which only accepts a string value.
   * This handler is useful as a subhandler, or for testing.
   * It will reject objects, arrays, and all other values except string values
   * (and possibly {@code null} values).
   */
  public static class StringValueHandler
          implements Handler
  {
    @NotNull
    private final Consumer<? super String> setter;
    private final boolean allowNull;

    /**
     * Constructor.
     * This will not allow {@code null} values.
     * @param setter setter called for string value
     */
    public StringValueHandler(@NotNull Consumer<? super String> setter)
    {
      this(setter, false);
    }

    /**
     * Constructor.
     * @param setter setter called for string value
     * @param allowNull allow null values?
     */
    public StringValueHandler(@NotNull Consumer<? super String> setter,
                              boolean allowNull)
    {
      this.setter = setter;
      this.allowNull = allowNull;
    }

    @Override
    public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
    {
      throw new IllegalStateException("Unexpected key value: " + key);
    }

    @Override
    public void beginObject(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Unexpected object value!");
    }

    @Override
    public void endObject(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Never should reach here!");
    }

    @Override
    public void beginArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Unexpected array value!");
    }

    @Override
    public void endArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Never should reach here!");
    }

    @Override
    public void nullValue(@NotNull SimpleJsonReader reader)
    {
      if (!allowNull) {
        throw new IllegalStateException("null value not allowed!");
      }
      setter.accept(null);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
    {
      setter.accept(v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, boolean v)
    {
      throw new IllegalStateException("Unexpected boolean value: "+v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
    {
      throw new IllegalStateException("Unexpected numeric value: "+v);
    }
  }

  /**
   * Handler which only accepts a numeric value.
   * This handler is useful as a subhandler, or for testing.
   * It will reject objects, arrays, and all other values except numeric values.
   */
  public static class NumericValueHandler
          implements Handler
  {
    @NotNull
    private final Consumer<? super Number> setter;

    /**
     * Constructor.
     * @param setter setter called for numeric value
     */
    public NumericValueHandler(@NotNull Consumer<? super Number> setter)
    {
      this.setter = setter;
    }

    @Override
    public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
    {
      throw new IllegalStateException("Unexpected key value: " + key);
    }

    @Override
    public void beginObject(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Unexpected object value!");
    }

    @Override
    public void endObject(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Never should reach here!");
    }

    @Override
    public void beginArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Unexpected array value!");
    }

    @Override
    public void endArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Never should reach here!");
    }

    @Override
    public void nullValue(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("null value not allowed!");
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
    {
      throw new IllegalStateException("Unexpected string value: "+v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, boolean v)
    {
      throw new IllegalStateException("Unexpected boolean value: "+v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
    {
      setter.accept(v);
    }
  }

  /**
   * Handler which only accepts a double value.
   * This handler is useful as a subhandler, or for testing.
   * It will reject objects, arrays, and all other values except numeric values.
   */
  public static class DoubleValueHandler
          implements Handler
  {
    @NotNull
    private final DoubleConsumer setter;

    /**
     * Constructor.
     * @param setter setter called for double value
     */
    public DoubleValueHandler(@NotNull DoubleConsumer setter)
    {
      this.setter = setter;
    }

    @Override
    public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
    {
      throw new IllegalStateException("Unexpected key value: " + key);
    }

    @Override
    public void beginObject(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Unexpected object value!");
    }

    @Override
    public void endObject(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Never should reach here!");
    }

    @Override
    public void beginArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Unexpected array value!");
    }

    @Override
    public void endArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Never should reach here!");
    }

    @Override
    public void nullValue(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("null value not allowed!");
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
    {
      throw new IllegalStateException("Unexpected string value: "+v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, boolean v)
    {
      throw new IllegalStateException("Unexpected boolean value: "+v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
    {
      setter.accept(v.doubleValue());
    }
  }

  /**
   * Handler which only accepts an integer value.
   * This handler is useful as a subhandler, or for testing.
   * It will reject objects, arrays, and all other values except numeric values.
   */
  public static class IntValueHandler
          implements Handler
  {
    @NotNull
    private final IntConsumer setter;

    /**
     * Constructor.
     * @param setter setter called for integer value
     */
    public IntValueHandler(@NotNull IntConsumer setter)
    {
      this.setter = setter;
    }

    @Override
    public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
    {
      throw new IllegalStateException("Unexpected key value: " + key);
    }

    @Override
    public void beginObject(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Unexpected object value!");
    }

    @Override
    public void endObject(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Never should reach here!");
    }

    @Override
    public void beginArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Unexpected array value!");
    }

    @Override
    public void endArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Never should reach here!");
    }

    @Override
    public void nullValue(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("null value not allowed!");
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
    {
      throw new IllegalStateException("Unexpected string value: "+v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, boolean v)
    {
      throw new IllegalStateException("Unexpected boolean value: "+v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
    {
      setter.accept(v.intValue());
    }
  }

  /**
   * Handler which only accepts a boolean value.
   * This handler is useful as a subhandler, or for testing.
   * It will reject objects, arrays, and all other values except boolean values.
   */
  public static class BooleanValueHandler
          implements Handler
  {
    @NotNull
    private final BooleanConsumer setter;

    /**
     * Constructor.
     * @param setter setter called for boolean value
     */
    public BooleanValueHandler(@NotNull BooleanConsumer setter)
    {
      this.setter = setter;
    }

    @Override
    public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
    {
      throw new IllegalStateException("Unexpected key value: " + key);
    }

    @Override
    public void beginObject(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Unexpected object value!");
    }

    @Override
    public void endObject(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Never should reach here!");
    }

    @Override
    public void beginArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Unexpected array value!");
    }

    @Override
    public void endArray(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("Never should reach here!");
    }

    @Override
    public void nullValue(@NotNull SimpleJsonReader reader)
    {
      throw new IllegalStateException("null value not allowed!");
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
    {
      throw new IllegalStateException("Unexpected string value: "+v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, boolean v)
    {
      setter.accept(v);
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
    {
      throw new IllegalStateException("Unexpected numeric value: "+v);
    }
  }

  /**
   * Temporary handler which ignores all input.
   */
  public static final TemporaryHandler SKIP_HANDLER = new TemporaryHandler()
  {
    @Override
    public void initialize(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void finish(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
    {
    }

    @Override
    public void beginObject(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void endObject(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void beginArray(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void endArray(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void nullValue(@NotNull SimpleJsonReader reader)
    {
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
    {
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, boolean v)
    {
    }

    @Override
    public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
    {
    }
  };

  /**
   * Test code.
   * @param args paths of JSON files
   * @throws IOException on read or format errors
   */
  public static void main(@NotNull String[] args) throws IOException
  {
    if (args.length > 0) {
      for (String arg : args) {
        System.out.println(readToMap(new FileReader(arg)));
      }
      return;
    }
    // A simple test.

    // The following will create:
    // {
    //   "name": "Henrik Nielsen",
    //   "size": 1.84,
    //   "age": 42,
    //   "clever": true,
    //   "address": {
    //     "street": "Kastanievej 15, 2, Agerskov",
    //     "city": "SKANDERBØRG",
    //     "zipcode": 8660,
    //     "country": "DENMARK"
    //   },
    //   "foods": [
    //     "Pizza",
    //     "Bami Goreng",
    //     "Ginger Icecream"
    //   ]
    // }
    final TextStringCollector collector = new TextStringCollector();
    try (SimpleJsonWriter writer = new SimpleJsonWriter(collector, 2)) {
      writer.beginObject();
      {
        writer.key("name").value("Henrik Nielsen")
                .key("size").value(1.84)
                .key("age").value(42)
                .key("clever").value(true);

        writer.key("address").beginObject()
                .key("street").value("Kastanievej 15, 2, Agerskov")
                .key("city").value("SKANDERBØRG")
                .key("zipcode").value(8660)
                .key("country").value("DENMARK")
                .endObject();
        writer.key("foods").beginArray()
                .value("Pizza")
                .value("Bami Goreng")
                .value("Ginger Icecream")
                .endArray();
      }
      writer.endObject();
    }
    final String original = collector.getText();

    final TextStringCollector c2 = new TextStringCollector();
    try (SimpleJsonWriter writer = new SimpleJsonWriter(c2, 2)) {
      SimpleJsonReader.read(new StringReader(original),
                            new Handler()
                            {
                              @Override
                              public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
                              {
                                writer.key(key);
                              }

                              @Override
                              public void beginObject(@NotNull SimpleJsonReader reader)
                              {
                                writer.beginObject();
                              }

                              @Override
                              public void endObject(@NotNull SimpleJsonReader reader)
                              {
                                writer.endObject();
                              }

                              @Override
                              public void beginArray(@NotNull SimpleJsonReader reader)
                              {
                                writer.beginArray();
                              }

                              @Override
                              public void endArray(@NotNull SimpleJsonReader reader)
                              {
                                writer.endArray();
                              }

                              @Override
                              public void nullValue(@NotNull SimpleJsonReader reader)
                              {
                                writer.nul();
                              }

                              @Override
                              public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
                              {
                                writer.value(v);
                              }

                              @Override
                              public void value(@NotNull SimpleJsonReader reader, boolean v)
                              {
                                writer.value(v);
                              }

                              @Override
                              public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
                              {
                                if (v instanceof Integer) {
                                  writer.value(v.intValue());
                                }
                                else {
                                  writer.value(v.doubleValue());
                                }
                              }
                            });
    }
    final String copy = c2.getText();
    if (!original.equals(copy)) {
      System.err.println("Different!");
      String[] lines1 = original.split("\n");
      String[] lines2 = copy.split("\n");
      final int nLines = Math.max(lines1.length, lines2.length);
      for (int l = 0;  l < nLines;  ++l) {
        final String l1 = l >= lines1.length
                ? Empty.STRING
                : lines1[l];
        final String l2 = l >= lines2.length
                ? Empty.STRING
                : lines2[l];
        if (!l1.equals(l2)) {
          System.out.println("DIFFERENCE:");
        }
        System.out.printf("%d: %s\n", l + 1, l1);
        System.out.printf("%d: %s\n", l + 1, l2);
      }
      System.exit(1);
    }
    else {
      System.out.println("CORRECT!");
    }

    final Map<String, Object> person = new LinkedHashMap<>();
    SimpleJsonReader.read(new StringReader(original),
                          new ValueHandler()
                          {
                            @NotNull
                            @Override
                            protected TemporaryHandler object()
                            {
                              return new ObjectHandler()
                              {
                                @NotNull
                                @Override
                                protected TemporaryHandler objectValue(@NotNull String key)
                                {
                                  if ("address".equals(key)) {
                                    final Map<String, Object> address = new LinkedHashMap<>();
                                    person.put("address", address);
                                    return new ObjectHandler()
                                    {
                                      @NotNull
                                      @Override
                                      protected TemporaryHandler objectValue(@NotNull String key)
                                      {
                                        System.err.println("Unexpected object value for key "+key);
                                        return SKIP_HANDLER;
                                      }

                                      @NotNull
                                      @Override
                                      protected TemporaryHandler arrayValue(@NotNull String key)
                                      {
                                        System.err.println("Unexpected array value for key "+key);
                                        return SKIP_HANDLER;
                                      }

                                      @Override
                                      protected void nullValue(@NotNull String key)
                                      {
                                        System.err.println("Unexpected null value for key "+key);
                                      }

                                      @Override
                                      protected void value(@NotNull String key, @NotNull String v)
                                      {
                                        switch (key) {
                                        case "street":
                                        case "city":
                                        case "country":
                                          address.put(key, v);
                                          break;
                                        default:
                                          System.err.println("Unexpected string value for key "+key);
                                        }
                                      }

                                      @Override
                                      protected void value(@NotNull String key, boolean v)
                                      {
                                        System.err.println("Unexpected boolean value for key "+key);
                                      }

                                      @Override
                                      protected void value(@NotNull String key, @NotNull Number v)
                                      {
                                        if ("zipcode".equals(key)) {
                                          address.put(key, v);
                                        }
                                        else {
                                          System.err.println("Unexpected number value for key "+key);
                                        }
                                      }

                                      @Override
                                      public void initialize(@NotNull SimpleJsonReader reader)
                                      {
                                      }

                                      @Override
                                      public void finish(@NotNull SimpleJsonReader reader)
                                      {
                                      }
                                    };
                                  }
                                  else {
                                    System.err.println("Unexpected object value for key "+key);
                                    return SKIP_HANDLER;
                                  }
                                }

                                @NotNull
                                @Override
                                protected TemporaryHandler arrayValue(@NotNull String key)
                                {
                                  if ("foods".equals(key)) {
                                    final List<String> foods = new ArrayList<>();
                                    person.put("foods", foods);
                                    return new ArrayHandler()
                                    {
                                      @NotNull
                                      @Override
                                      protected TemporaryHandler objectElement(int index)
                                      {
                                        System.err.println("Unexpected object element for index "+index);
                                        return SKIP_HANDLER;
                                      }

                                      @NotNull
                                      @Override
                                      protected TemporaryHandler arrayElement(int index)
                                      {
                                        System.err.println("Unexpected array element for index "+index);
                                        return SKIP_HANDLER;
                                      }

                                      @Override
                                      protected void nullElement(int index)
                                      {
                                        System.err.println("Unexpected null element for index "+index);
                                      }

                                      @Override
                                      protected void element(int index, @NotNull String v)
                                      {
                                        assert index == foods.size();
                                        foods.add(v);
                                      }

                                      @Override
                                      protected void element(int index, boolean v)
                                      {
                                        System.err.println("Unexpected boolean element for index "+index);
                                      }

                                      @Override
                                      protected void element(int index, @NotNull Number v)
                                      {
                                        System.err.println("Unexpected number element for index "+index);
                                      }

                                      @Override
                                      public void initialize(@NotNull SimpleJsonReader reader)
                                      {
                                      }

                                      @Override
                                      public void finish(@NotNull SimpleJsonReader reader)
                                      {
                                      }
                                    };
                                  }
                                  else {
                                    System.err.println("Unexpected array value for key "+key);
                                    return SKIP_HANDLER;
                                  }
                                }

                                @Override
                                protected void nullValue(@NotNull String key)
                                {
                                  System.err.println("Unexpected null value for key "+key);
                                }

                                @Override
                                protected void value(@NotNull String key, @NotNull String v)
                                {
                                  if ("name".equals(key)) {
                                    person.put(key, v);
                                  }
                                  else {
                                    System.err.printf("Unexpected key-value pair: %s -> %s\n", key, v);
                                  }
                                }

                                @Override
                                protected void value(@NotNull String key, boolean v)
                                {
                                  if ("clever".equals(key)) {
                                    person.put(key, v);
                                  }
                                  else {
                                    System.err.printf("Unexpected key-value pair: %s -> %s\n", key, v);
                                  }
                                }

                                @Override
                                protected void value(@NotNull String key, @NotNull Number v)
                                {
                                  switch (key) {
                                  case "age":
                                    assert v instanceof Integer;
                                    person.put(key, v);
                                    break;
                                  case "size":
                                    assert v instanceof Double;
                                    person.put(key, v);
                                    break;
                                  default:
                                    System.err.printf("Unexpected key-value pair: %s -> %s\n", key, v);
                                  }
                                }


                                @Override
                                public void initialize(@NotNull SimpleJsonReader reader)
                                {
                                }

                                @Override
                                public void finish(@NotNull SimpleJsonReader reader)
                                {
                                }
                              };
                            }
                          });
    System.out.println(person);
    System.out.println(readToMap(new StringReader(original)));
  }
}
