// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2022-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.io.json;

import de.caff.annotation.NotNull;
import de.caff.generics.CharIndexable;

import java.io.IOException;

/**
 * JSON tools.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since 2022/05/01
 */
public class Json
{
  /** Don't create. */
  private Json() {}

  /**
   * Mask special characters in the given string for JSON output.
   * This method takes care of special characters like {@code "},
   * {@code \}, {@code \n} and similar.
   * @param str str
   * @return masked string
   */
  @NotNull
  public static String maskForOutput(@NotNull String str)
  {
    if (str.isEmpty()) {
      return str;
    }
    StringBuilder sb = null;
    final int len = str.length();
    for (int idx = 0;  idx < len; ++idx) {
      final char ch = str.charAt(idx);
      final String subst;
      switch (ch) {
      case '\\':
        subst = "\\\\";
        break;
      case '"':
        subst = "\\\"";
        break;
      case '\n':
        subst = "\\n";
        break;
      case '\r':
        subst = "\\r";
        break;
      case '\t':
        subst = "\\t";
        break;
      case '\b':
        subst = "\\b";
        break;
      case '\f':
        subst = "\\f";
        break;
      default:
        if (ch < ' ') {
          subst = String.format("\\u%04X", (int)ch);
        }
        else {
          if (sb != null) {
            sb.append(ch);
          }
          continue;
        }
      }
      if (sb == null) {
        sb = new StringBuilder(str.substring(0, idx));
      }
      sb.append(subst);
    }
    return sb == null
            ? str
            : sb.toString();
  }

  /**
   * Unmask special characters in the incoming string.
   * This method undoes the masking from {@linkplain #maskForOutput(String)}
   * and recreates {@code "}, {@code \}, {@code \n} and similar.
   * @param str incoming JSON string
   * @return unmasked Java string
   * @throws IOException if {@code str} contains invalid escape sequences
   */
  @NotNull
  public static String unmaskFromInput(@NotNull String str)
          throws IOException
  {
    StringBuilder sb = null;
    boolean lastWasBackslash = false;
    final int len = str.length();
    for (int idx = 0;  idx < len; ++idx) {
      final char ch = str.charAt(idx);
      if (lastWasBackslash) {
        final int storeIdx = idx;
        lastWasBackslash = false;
        char subst;
        switch (ch) {
        case 'n':
          subst = '\n';
          break;
        case 'r':
          subst = '\r';
          break;
        case 't':
          subst = '\t';
          break;
        case 'b':
          subst = '\b';
          break;
        case 'f':
          subst = '\f';
          break;
        case 'u':
          final int start = idx + 1;
          final int end = start + 4;
          if (end > len) {
            throw new IOException("Not enough digits for unicode escape \\u!");
          }
          final String hexCode = str.substring(start, end);
          try {
            subst = (char)Integer.parseInt(hexCode, 16);
          } catch (NumberFormatException e) {
            throw new IOException("Not a valid unicode escape: \\u" + hexCode,
                                  e);
          }
          idx = end - 1;
          break;
        default:
          subst = ch;
          break;
        }
        if (sb == null) {
          sb = new StringBuilder(str.substring(0, storeIdx - 1)); // -1 because lastWasBackslash
        }
        sb.append(subst);
      }
      else if (ch == '\\') {
        lastWasBackslash = true;
      }
      else {
        if (sb != null) {
          sb.append(ch);
        }
      }
    }

    return sb == null
            ? str
            : sb.toString();
  }

  public static void main(String[] args) throws IOException
  {
    final String str = CharIndexable.viewByIndex(256, i -> (char)i).makeString();
    final String encoded = maskForOutput(str);
    System.out.println(encoded);
    final String decoded = unmaskFromInput(encoded);

    if (!str.equals(decoded)) {
      System.err.printf("Different:\n%s\n%s\n", str, decoded);
      System.exit(1);
    }
  }

  /** Environment. */
  public enum Env
  {
    /**
     * Inside an object.
     */
    Object(SimpleJsonWriter.State.Key) {
      @Override
      void close(@NotNull SimpleJsonWriter writer)
      {
        writer.endObject();
      }
    },
    /**
     * Inside an array.
     */
    Array(SimpleJsonWriter.State.Value) {
      @Override
      void close(@NotNull SimpleJsonWriter writer)
      {
        writer.endArray();
      }
    };

    /** Default state for this kind of environment. */
    @NotNull
    final SimpleJsonWriter.State defaultState;

    /**
     * Constructor.
     * @param defaultState default state for this environment
     */
    Env(@NotNull SimpleJsonWriter.State defaultState)
    {
      this.defaultState = defaultState;
    }

    /**
     * Close this environment.
     * @param writer writer to write to
     */
    abstract void close(@NotNull SimpleJsonWriter writer);
  }
}
