// ============================================================================
// File:               AbstractSimpleJsonFilter
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2022-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            6/2/22 1:30 PM
//=============================================================================
package de.caff.io.json;

import de.caff.annotation.NotNull;

/**
 * Abstract basic low-level JSON filter.
 * This acts as a handler for a {@link SimpleJsonReader}
 * and writes its output to the wrapped {@link SimpleJsonWriter}.
 * <p>
 * This class is thought for simple transformations like
 * exchanging a key or value.
 * <p>
 * This default implementation just forwards everything
 * to the writer. Extending classes may intercept where
 * they seem fit.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since June 02, 2022
 */
public abstract class AbstractSimpleJsonFilter
        implements SimpleJsonReader.Handler
{
  @NotNull
  private final SimpleJsonWriter writer;

  /**
   * Constructor.
   * @param writer writer to which this filter writes.
   */
  protected AbstractSimpleJsonFilter(@NotNull SimpleJsonWriter writer)
  {
    this.writer = writer;
  }

  @Override
  public void key(@NotNull SimpleJsonReader reader, @NotNull String key)
  {
    writer.key(key);
  }

  @Override
  public void beginObject(@NotNull SimpleJsonReader reader)
  {
    writer.beginObject();
  }

  @Override
  public void endObject(@NotNull SimpleJsonReader reader)
  {
    writer.endObject();
  }

  @Override
  public void beginArray(@NotNull SimpleJsonReader reader)
  {
    writer.beginArray();
  }

  @Override
  public void endArray(@NotNull SimpleJsonReader reader)
  {
    writer.endArray();
  }

  @Override
  public void nullValue(@NotNull SimpleJsonReader reader)
  {
    writer.nul();
  }

  @Override
  public void value(@NotNull SimpleJsonReader reader, @NotNull String v)
  {
    writer.value(v);
  }

  @Override
  public void value(@NotNull SimpleJsonReader reader, boolean v)
  {
    writer.value(v);
  }

  @Override
  public void value(@NotNull SimpleJsonReader reader, @NotNull Number v)
  {
    if (v instanceof Integer) {
      writer.value(v.intValue());
    }
    else {
      writer.value(v.doubleValue());
    }
  }
}
