// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.io;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * String collector which writes all collected strings to
 * an output stream.
 * <p>
 * Usually should be closed after usage. Recommend is to use it in try-with-resources-
 * 
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class OutputStreamStringCollector
        implements AutoCloseable,
                   StringCollector
{
  /** The default charset (UTF_8). */
  public static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;
  @NotNull
  private final OutputStream stream;
  @NotNull
  private final Charset charset;
  /** Possible error. */
  private IOException error;
  private char lastLetter = '\0';

  /**
   * Constructor.
   * This will use the {@link #DEFAULT_CHARSET} for output encoding.
   * @param stream output stream
   */
  public OutputStreamStringCollector(@NotNull OutputStream stream)
  {
    this(stream, DEFAULT_CHARSET);
  }

  /**
   * Constructor.
   * @param stream       output stream
   * @param charsetName  name of charset to use for output encoding
   */
  public OutputStreamStringCollector(@NotNull OutputStream stream,
                                     @NotNull String charsetName)
  {
    this(stream, Charset.forName(charsetName));
  }

  /**
   * Constructor.
   * @param stream output stream
   * @param charset charset used for string output encoding
   */
  public OutputStreamStringCollector(@NotNull OutputStream stream,
                                     @NotNull Charset charset)
  {
    this.stream = stream;
    this.charset = charset;
  }

  /**
   * Add a string t this collector.
   *
   * @param str string collector
   */
  @Override
  public void add(@NotNull String str)
  {
    if (str.isEmpty()) {
      return;
    }
    lastLetter = str.charAt(str.length() - 1);
    if (error == null) {
      try {
        stream.write(str.getBytes(charset));
      } catch (IOException e) {
        error = e;
      }
    }
  }

  /**
   * Add a character to this collector.
   *
   * @param ch character to add
   */
  @Override
  public void add(char ch)
  {
    add(Character.toString(ch));
  }

  /**
   * Get the last letter of the last string which was added.
   *
   * @return last letter, zero at the very beginning
   */
  @Override
  public char getLastLetter()
  {
    return lastLetter;
  }

  /**
   * Closes this stream and releases any system resources associated
   * with it. If the stream is already closed then invoking this
   * method has no effect.
   *
   * @throws IOException if an I/O error occurs
   */
  @Override
  public void close() throws IOException
  {
    rethrow();
    stream.close();
  }

  /**
   * Rethrow an error which was caught during writing of added strings.
   * Does nothing if no error happened.
   * @throws IOException exception caught during writing strings
   */
  public void rethrow() throws IOException
  {
    if (error != null) {
      throw error;
    }
  }

  /**
   * Get a possible error.
   * @return error caught during string writing
   */
  @Nullable
  public IOException getError()
  {
    return error;
  }
}
