// ============================================================================
// File:               LimitedByteArrayOutputStream
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2022-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            8/10/22 12:09 PM
//=============================================================================
package de.caff.io;

import de.caff.annotation.NotNull;
import de.caff.util.ByteRingBuffer;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;

/**
 * A byte array output stream with a limited size.
 * It will discard earlier bytes of more than
 * the given number of bytes is written to it.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August 10, 2022
 */
public class LimitedByteArrayOutputStream
        extends OutputStream
{
  @NotNull
  private final ByteRingBuffer buffer;

  /**
   * Create a buffering output stream with the given size limit.
   * @param bufferSize maximum number of buffered bytes
   */
  public LimitedByteArrayOutputStream(int bufferSize)
  {
    buffer = new ByteRingBuffer(bufferSize);
  }

  @Override
  public void write(int b) throws IOException
  {
    buffer.append(b);
  }

  @Override
  public void write(@NotNull byte[] b) throws IOException
  {
    buffer.append(b);
  }

  @Override
  public void write(@NotNull byte[] b, int off, int len) throws IOException
  {
    buffer.append(b, off, len);
  }

  @Override
  public void flush() throws IOException
  {
  }

  @Override
  public void close() throws IOException
  {
  }

  /**
   * Get the cached bytes.
   * @return bytes cached in this limited byte array output stream
   */
  @NotNull
  public byte[] toByteArray()
  {
    return buffer.toByteArray();
  }

  /**
   * Clear all cached bytes.
   * This will not clear the total {@link #getTotalNumberOfBytes() total number of bytes}.
   */
  public void reset()
  {
    buffer.clear();
  }

  /**
   * Get the total number of bytes written to this stream.
   * @return total number of bytes written so far
   */
  public long getTotalNumberOfBytes()
  {
    return buffer.getNumberOfCollectedBytes();
  }

  /**
   * Get the content of this buffer as a string.
   * Please note that for multi-byte character encodings
   * the internal buffer might start at
   *
   * @param charset charset used to decode the cached bytes
   * @return string representation of the buffer content
   */
  @NotNull
  public String toString(@NotNull Charset charset)
  {
    return new String(buffer.toByteArray(), charset);
  }

  /**
   * Get a string representation.
   * This follows the example of {@link java.io.ByteArrayOutputStream}
   * which uses {@link #toString(Charset)} which is not
   * always a good idea.
   * @return string representation created from the bytes using the default charset
   */
  @Override
  @NotNull
  public String toString()
  {
    return toString(Charset.defaultCharset());
  }
}
