// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.io;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.util.LongCounter;
import de.caff.util.debug.Debug;

import java.io.IOException;
import java.io.InputStream;

/**
 * A sink for data from an input stream which discards the incoming data.
 * <p>
 * It is especially useful for handling the outputting streams of {@link Process}.
 * Create via {@link #startOn(InputStream)} which will automatically discard all
 * bytes from the stream in a background thread.
 * <p>
 * See also {@link InputStreamBuffer} which collects the output in a ring buffer,
 * and {@link InputStreamPipe} which forwards all bytes from one stream to another.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August 10, 2022
 */
public class InputStreamSink
        implements Runnable
{
  /** The input stream. */
  @NotNull
  private final InputStream is;
  /** Byte counter. */
  @NotNull
  private final LongCounter totalNumberOfBytes = LongCounter.THREAD_SAFE.create();
  /** Possible error. */
  @Nullable
  private IOException error;

  /**
   * Constructor.
   * Note that in most use cases the factory method {@link #startOn(InputStream)} is
   * a better way to create an input stream sink.
   * @param is input stream
   */
  public InputStreamSink(@NotNull InputStream is)
  {
    this.is = is;
  }

  /**
   * Get the total number of bytes read by this sink.
   * @return total number of bytes read so far
   */
  public long getTotalByteCount()
  {
    return totalNumberOfBytes.getValue();
  }

  /**
   * Return a possible error which happened during reading the stream
   * when using the {@link #run()} method.
   *
   * @return error caught during {@linkplain #run()} or {@code null},
   *         if no error was caught
   */
  @Nullable
  public IOException getError()
  {
    return error;
  }

  /**
   * When an object implementing interface {@code Runnable} is used
   * to create a thread, starting the thread causes the object's
   * {@code run} method to be called in that separately executing
   * thread.
   * <p>
   * The general contract of the method {@code run} is that it may
   * take any action whatsoever.
   *
   * @see Thread#run()
   */
  @Override
  public void run()
  {
    final byte[] buffer = new byte[256];
    try {
      int bytesRead;
      while ((bytesRead = is.read(buffer)) > 0) {
        totalNumberOfBytes.add(bytesRead);
      }
    } catch (IOException e) {
      error = e;
      Debug.error(e);
    }
  }

  /**
   * Start a sink on the given stream.
   * @param inStream  input stream
   * @return input stream sink based on the given stream
   */
  @NotNull
  public static InputStreamSink startOn(@NotNull InputStream inStream) {
    final InputStreamSink target = new InputStreamSink(inStream);
    new Thread(target).start();
    return target;
  }
}
