// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.io;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Indexable;
import de.caff.util.TextLineRingBuffer;
import de.caff.util.debug.Debug;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.List;

/**
 * A sink for data from an input stream which collects the incoming data as text lines.
 * <p>
 * It is especially useful for handling the outputting streams of {@link Process}.
 * Create via {@link #startOn(InputStream)} or {@link #startOn(InputStream, int)}
 * which will automatically collect all bytes from the stream in a background thread.
 * <p>
 * See {@link InputStreamBuffer} if you are interested in the raw bytes,
 * {@link InputStreamSink} if you are not interested at all in the content of the streams,
 * and {@link InputStreamPipe} which forwards all bytes from one stream to another.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August 10, 2022
 */
public class InputStreamLineBuffer
        implements Runnable
{
  /** This is suggestion from OpenJDK's ArraySupport class which makes sure to work with all Java versions. */
  public static final int MAXIMUM_ARRAY_SIZE = Integer.MAX_VALUE - 8;
  /** Default size for line buffers. */
  public static final int DEFAULT_BUFFER_SIZE = 0x10000;
  /** Line buffer collecting the data read from the stream. */
  @NotNull
  private final TextLineRingBuffer lineBuffer;
  /** The input stream. */
  @NotNull
  private final InputStream is;
  /** Possible error. */
  @Nullable
  private IOException error;

  /**
   * Constructor using the system's default charset.
   * <p>
   * Note that in most use cases the factory methods {@link #startOn(InputStream)}
   * and {@link #startOn(InputStream, int)} are better ways to
   * create an input stream sink.
   * @param is input stream
   * @param maxSize maximum number of buffered lines, implementation might
   *                store one line more
   */
  public InputStreamLineBuffer(@NotNull InputStream is,
                               int maxSize)
  {
    this(is, maxSize, Charset.defaultCharset());
  }

  /**
   * Constructor.
   * <p>
   * Note that in most use cases the factory methods {@link #startOn(InputStream)}
   * and {@link #startOn(InputStream, int)} are better ways to
   * create an input stream sink.
   * @param is input stream
   * @param maxSize maximum number of buffered lines, implementation might
   *                store one line more
   * @param charset charset used to decode the input stream
   */
  public InputStreamLineBuffer(@NotNull InputStream is,
                               int maxSize,
                               @NotNull Charset charset)
  {
    if (maxSize <= 0) {
      throw new IllegalArgumentException("maxSize has to be positive!");
    }
    this.is = is;
    lineBuffer = new TextLineRingBuffer(maxSize, charset);
  }

  /**
   * Return a possible error which happened during reading the stream
   * when using the {@link #run()} method.
   *
   * @return error caught during {@linkplain #run()} or {@code null},
   *         if no error was caught
   */
  @Nullable
  public IOException getError()
  {
    return error;
  }

  /**
   * Get a view to the buffered lines.
   * Note that this does not copy the lines, but has the disadvantage
   * that the view will become invalid as soon as more data is written
   * to this buffer. If the input is not close prefer {@link #toLineIndexable()},
   * {@link #toLineList()}, or {@link #getLines()}.
   * @return lines
   * @see #toLineIndexable()
   * @see #toLineList()
   * @see #getLines()
   */
  @NotNull
  public Indexable<String> viewLines()
  {
    return lineBuffer.viewLines();
  }

  /**
   * Get the buffered lines as an independent indexable.
   * @return buffered lines
   * @see #viewLines()
   * @see #toLineList()
   * @see #getLines()
   */
  @NotNull
  public Indexable<String> toLineIndexable()
  {
    return lineBuffer.toLineIndexable();
  }

  /**
   * Get the buffered lines as an independent indexable.
   * @return buffered lines
   * @see #viewLines()
   * @see #toLineIndexable()
   * @see #getLines()
   */
  @NotNull
  public List<String> toLineList()
  {
    return lineBuffer.toLineList();
  }

  /**
   * Get the collected output lines.
   * @return byte array with the collected bytes
   * @see #viewLines()
   * @see #toLineIndexable()
   * @see #toLineList()
   */
  @NotNull
  public String[] getLines()
  {
    return lineBuffer.toLineArray();
  }

  /**
   * Get the output as text, using the default charset.
   * Note that due to limited buffer size for multi-byte encodings
   * such as UTF-8 the first characters may be broken.
   * @return stream output as text
   */
  @NotNull
  public String getOutput()
  {
    return lineBuffer.toString();
  }

  /**
   * Clear the output.
   * This will remove all cached bytes from the buffer.
   */
  public void clearOutput()
  {
    lineBuffer.clear();
  }

  /**
   * Get the total number of bytes read by this sink.
   * @return total number of bytes read so far
   */
  public long getTotalByteCount()
  {
    return lineBuffer.getNumberOfCollectedBytes();
  }

  /**
   * When an object implementing interface {@code Runnable} is used
   * to create a thread, starting the thread causes the object's
   * {@code run} method to be called in that separately executing
   * thread.
   * <p>
   * The general contract of the method {@code run} is that it may
   * take any action whatsoever.
   *
   * @see Thread#run()
   */
  @Override
  public void run()
  {
    final byte[] buffer = new byte[256];
    try {
      int bytesRead;
      while ((bytesRead = is.read(buffer)) > 0) {
        lineBuffer.append(buffer, 0, bytesRead);
      }
    } catch (IOException e) {
      error = e;
      Debug.error(e);
    }
  }

  /**
   * Start a sink on the given stream.
   * This method uses {@link #DEFAULT_BUFFER_SIZE} as buffer size.
   * @param inStream  input stream
   * @return input stream line buffer based on the given stream
   * @see #startOn(InputStream, int)
   */
  @NotNull
  public static InputStreamLineBuffer startOn(@NotNull InputStream inStream) {
    return startOn(inStream, DEFAULT_BUFFER_SIZE);
  }

  /**
   * Start a sink on the given stream.
   * @param inStream  input stream
   * @param bufferSize maximum number of cached lines
   * @return input stream line buffer based on the given stream
   */
  @NotNull
  public static InputStreamLineBuffer startOn(@NotNull InputStream inStream, int bufferSize) {
    final InputStreamLineBuffer target = new InputStreamLineBuffer(inStream, bufferSize);
    new Thread(target).start();
    return target;
  }

  @Override
  public String toString()
  {
    return getOutput();
  }
}
