// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.io;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.ByteRingBuffer;
import de.caff.util.debug.Debug;

import java.io.*;
import java.nio.charset.Charset;

/**
 * A sink for data from an input stream which collects the incoming data.
 * <p>
 * It is especially useful for handling the outputting streams of {@link Process}.
 * Create via {@link #startOn(InputStream)} or {@link #startOn(InputStream, int)}
 * which will automatically collect all bytes from the stream in a background thread.
 * <p>
 * See {@link InputStreamSink} if you are not interested in the content of the streams,
 * and {@link InputStreamPipe} which forwards all bytes from one stream to another.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since August 10, 2022
 */
public class InputStreamBuffer
        implements Runnable
{
  /** This is suggestion from OpenJDK's ArraySupport class which makes sure to work with all Java versions. */
  public static final int MAXIMUM_ARRAY_SIZE = Integer.MAX_VALUE - 8;
  /** Byte buffer collecting the data read from the stream. */
  @NotNull
  private final ByteRingBuffer byteBuffer;
  /** The input stream. */
  @NotNull
  private final InputStream is;
  /** Possible error. */
  @Nullable
  private IOException error;

  /**
   * Constructor.
   * <p>
   * Note that in most use cases the factory methods {@link #startOn(InputStream)}
   * and {@link #startOn(InputStream, int)} are better ways to
   * create an input stream sink.
   * @param is input stream
   * @param maxSize maximum buffer size
   */
  public InputStreamBuffer(@NotNull InputStream is,
                           int maxSize)
  {
    if (maxSize <= 0) {
      throw new IllegalArgumentException("maxSize has to be positive!");
    }
    this.is = is;
    byteBuffer = new ByteRingBuffer(maxSize);
  }

  /**
   * Return a possible error which happened during reading the stream
   * when using the {@link #run()} method.
   *
   * @return error caught during {@linkplain #run()} or {@code null},
   *         if no error was caught
   */
  @Nullable
  public IOException getError()
  {
    return error;
  }

  /**
   * Get the collected output as binary data.
   * @return byte array with the collected bytes
   */
  @NotNull
  public byte[] getBinaryOutput()
  {
    return byteBuffer.toByteArray();
  }

  /**
   * Get the output as text, using the default charset.
   * Note that due to limited buffer size for multi-byte encodings
   * such as UTF-8 the first characters may be broken.
   * @return stream output as text
   */
  @NotNull
  public String getOutput()
  {
    return getOutput(Charset.defaultCharset());
  }

  /**
   * Get the output as text, using the given charset
   * Note that due to limited buffer size for multi-byte encodings
   * such as UTF-8 the first characters may be broken.
   * @param charset charset to use for decoding the binary output
   * @return stream output as text
   */
  @NotNull
  public String getOutput(@NotNull Charset charset)
  {
    return new String(getBinaryOutput(), charset);
  }

  /**
   * Clear the output.
   * This will remove all cached bytes from the buffer.
   */
  public void clearOutput()
  {
    byteBuffer.clear();
  }

  /**
   * Get the total number of bytes read by this sink.
   * @return total number of bytes read so far
   */
  public long getTotalByteCount()
  {
    return byteBuffer.getNumberOfCollectedBytes();
  }

  /**
   * When an object implementing interface {@code Runnable} is used
   * to create a thread, starting the thread causes the object's
   * {@code run} method to be called in that separately executing
   * thread.
   * <p>
   * The general contract of the method {@code run} is that it may
   * take any action whatsoever.
   *
   * @see Thread#run()
   */
  @Override
  public void run()
  {
    final byte[] buffer = new byte[256];
    try {
      int bytesRead;
      while ((bytesRead = is.read(buffer)) > 0) {
        byteBuffer.append(buffer, 0, bytesRead);
      }
    } catch (IOException e) {
      error = e;
      Debug.error(e);
    }
  }

  /**
   * Start a buffer on the given stream.
   * This method uses {@link #MAXIMUM_ARRAY_SIZE} as size for the ring buffer.
   * @param inStream  input stream
   * @return input stream buffer
   */
  @NotNull
  public static InputStreamBuffer startOn(@NotNull InputStream inStream) {
    return startOn(inStream, MAXIMUM_ARRAY_SIZE);
  }

  /**
   * Start a buffer on the given stream.
   * @param inStream  input stream
   * @param bufferSize maximum number of cached bytes
   * @return input stream buffer
   */
  @NotNull
  public static InputStreamBuffer startOn(@NotNull InputStream inStream, int bufferSize) {
    final InputStreamBuffer target = new InputStreamBuffer(inStream, bufferSize);
    new Thread(target).start();
    return target;
  }

  @Override
  public String toString()
  {
    return getOutput();
  }
}
