// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2009-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.i18n;

import de.caff.annotation.NotNull;
import de.caff.util.Base64;
import de.caff.util.debug.Debug;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import java.io.*;
import java.util.*;
import java.util.regex.Pattern;

/**
 *  A resource bundle based on an XML file defining the resources.
 * <p>
 *  This adds the simplicity of using properties files for resources (no compilation necessary)
 *  with better language support (properties files are always in ISO-8859-1 encoding,
 *  while an XML file can specify its encoding) and some more features like comments.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class XmlResourceBundle
  extends ResourceBundle
{
  /** Extension used for XML resource bundles. */
  public static final String EXTENSION = ".resource.xml";

  public static final String TAG_I18N = "i18n";
  public static final String TAG_GROUP = "grp";
  public static final String TAG_RESOURCE = "res";
  public static final String TAG_ACTION = "action";
  public static final String ATTR_KEY = "key";
  public static final String ATTR_VALUE = "value";
  public static final String ATTR_TYPE  = "type";
  public static final String ATTR_SUBTYPE = "subtype";
  public static final String ATTR_COMMENT = "comment";
  public static final String ATTR_VERSION = "version";
  public static final String ATTR_INHERET = "inheret";
  public static final String ATTR_NAME = "name";
  public static final int MY_VERSION = 1;

  private static final Pattern PATTERN_LINE_END = Pattern.compile("\n");

  public interface ValueType
  {
    String getTypeName();
    String valueToString(Object value);
    Object stringToValue(String str);
    boolean equals(Object value1, Object value2);
  }

  private abstract static class AbstractBasicValueType
          implements ValueType
  {
    private final String typeName;

    private AbstractBasicValueType(String typeName)
    {
      this.typeName = typeName;
    }

    @Override
    public String getTypeName()
    {
      return typeName;
    }
  }

  private static String escapeForXml(String str)
  {
    if (str == null) {
      return null;
    }
    int len = str.length();
    StringBuilder result = new StringBuilder(2*len);
    for (int c = 0;  c < len;  ++c) {
      char ch = str.charAt(c);
      switch (ch) {
      case '&':
        result.append("&amp;");
        break;
      case '<':
        result.append("&lt;");
        break;
      case '>':
        result.append("&gt;");
        break;
      case '"':
        result.append("&quot;");
        break;
      case '\\':
        result.append("\\\\");
        break;
      case '\t':
        result.append("\\t");
        break;
      case '\n':
        result.append("\\n");
        break;
      case '\r':
        result.append("\\r");
        break;
      default:
        result.append(ch);
        break;
      }
    }
    return result.toString();
  }

  private static String unescapeFromXml(String str)
  {
    if (str == null) {
      return null;
    }
    int len = str.length();
    StringBuilder result = new StringBuilder(len);
    for (int c = 0;  c < len;  ++c) {
      char ch = str.charAt(c);
      if (ch == '\\')  {
        if (c < len - 1) {
          ch = str.charAt(++c);
          switch (ch) {
          case '\\':
            result.append(ch);
            break;
          case 'n':
            result.append('\n');
            break;
          case 'r':
            result.append('\r');
            break;
          case 't':
            result.append('\t');
            break;
          default:
            Debug.error("Unknown escape sequence: \\"+ch);
            result.append(ch);
            break;
          }
        }
        else {
          Debug.error("Pending \\");
        }
      }
      else {
        result.append(ch);
      }
    }
    return result.toString();
  }

  private static final Map<String, ValueType> converterMapping = new HashMap<>();
  private static final ValueType STRING_VALUE_TYPE = new AbstractBasicValueType("String")
  {
    @Override
    public String valueToString(Object value)
    {
      return escapeForXml(value.toString());
    }

    @Override
    public Object stringToValue(String str)
    {
      return unescapeFromXml(str);
    }

    @Override
    public boolean equals(Object value1, Object value2)
    {
      return value1.equals(value2);
    }
  };
  private static final ValueType STRING_ARRAY_VALUE_TYPE = new AbstractBasicValueType("String[]")
  {

    @Override
    public String valueToString(Object value)
    {
      String[] array = (String[])value;
      StringBuilder result = new StringBuilder();
      for (int a = 0;  a < array.length;  ++a) {
        result.append(escapeForXml(array[a]));
        if (a < array.length - 1) {
         result.append('\n');
        }
      }
      return result.toString();
    }

    @Override
    public Object stringToValue(String str)
    {
      String[] array = PATTERN_LINE_END.split(str);
      for (int a = array.length - 1;  a >= 0;  --a) {
        array[a] = unescapeFromXml(array[a]);
      }
      return array;
    }

    @Override
    public boolean equals(Object value1, Object value2)
    {
      return Arrays.equals((String[])value1, (String[])value2);
    }
  };

  public static final ValueType OBJECT_VALUE_TYPE = new AbstractBasicValueType("Object")
  {
    @Override
    public String valueToString(Object value)
    {
      try {
        return Base64.encodeObject((Serializable)value);
      } catch (Throwable e) {
        throw new RuntimeException("Not able to serialize object! Reason: "+e);
      }
    }

    @Override
    public Object stringToValue(String str)
    {
      try {
        return Base64.decodeToObject(str);
      } catch (Throwable e) {
        throw new RuntimeException("Not able to serialize object! Reason: "+e);
      }
    }

    @Override
    public boolean equals(@NotNull Object value1, @NotNull Object value2)
    {
      return value1.equals(value2);
    }
  };

  static {
    ValueType[] types = {
            STRING_VALUE_TYPE,
            STRING_ARRAY_VALUE_TYPE,
            OBJECT_VALUE_TYPE
    };
    for (int c = types.length - 1;  c >= 0;  --c) {
      converterMapping.put(types[c].getTypeName(),
                           types[c]);
    }
  }

  /**
   * Basic definition of a resource node in the XML file.
   */
  public interface ResourceNode
  {
    /** Empty resource node array. */
    ResourceNode[] EMPTY_RESOURCE_NODE_ARRAY = {};

    /**
     * Get the comment.
     * @return comment
     */
    String getComment();

    /**
     * Set the comment.
     * @param comment  comment
     */
    void setComment(String comment);

    /**
     * Convert the node to XML.
     * @return XML representation of this node
     */
    String toXml();

    /**
     * Get the sub nodes of this node.
     * @return {@code null} if this is a node which cannot have sub nodes,<br>
     *         array of sub nodes (possibly empty) otherwise
     */
    ResourceNode[] getSubNodes();

    /**
     * Register the resources.
     * @param resourceMapping map where to register
     */
    void registerResources(Map<String, ResourceNode> resourceMapping);

    /**
     * Get the XML tag of this node
     * @return XML tag
     */
    String getXmlTag();

    /**
     * Get an unique id for this resource.
     * This should return either a unique id or {@code null} if this is no resource to look for
     * @return unique id
     */
    String getId();
  }

  /**
   *  Abstract base implementation of a resource node.
   */
  public abstract static class AbstractBasicResourceNode
          implements ResourceNode
  {
    /** Get the comment. */
    private String comment;

    protected AbstractBasicResourceNode(Attributes attributes)
    {
      comment = attributes.getValue(ATTR_COMMENT);
    }

    protected AbstractBasicResourceNode(String comment)
    {
      this.comment = comment;
    }

    /**
     * Get the comment.
     *
     * @return comment
     */
    @Override
    public String getComment()
    {
      return comment;
    }

    /**
     * Set the comment.
     * @param comment  new comment
     */
    @Override
    public void setComment(String comment)
    {
      this.comment = comment;
    }

    protected void addXmlAttributes(StringBuilder xmlString)
    {
      if (comment != null) {
        addAttribute(xmlString, ATTR_COMMENT, comment);
      }
    }

    protected void addXmlCompleteTag(StringBuilder sb)
    {
      sb.append("<")
              .append(getXmlTag());
      addXmlAttributes(sb);
      sb.append("/>\n");
    }

    protected void addXmlStartTag(StringBuilder sb)
    {
      sb.append("<")
              .append(getXmlTag());
      addXmlAttributes(sb);
      sb.append(">");
    }

    protected void addXmlEndTag(StringBuilder sb)
    {
      sb.append("</")
              .append(getXmlTag())
              .append(">\n");
    }

    protected void addXmlContent(StringBuilder sb)
    {
    }

    protected boolean hasContent()
    {
      return true;
    }

    /**
     * Convert the node to XML.
     *
     * @return XML representation of this node
     */
    @Override
    public String toXml()
    {
      StringBuilder sb = new StringBuilder();
      if (hasContent()) {
        addXmlStartTag(sb);
        addXmlContent(sb);
        addXmlEndTag(sb);
      }
      else {
        addXmlCompleteTag(sb);
      }
      return sb.toString();
    }
  }

  /**
   * A group of resource nodes.
   */
  private static class ResourceGroup
          extends AbstractBasicResourceNode
  {
    /** Key to resources list. */
    private final Map<String, ResourceNode> idsToResources = new HashMap<>();
    /** The node list. */
    private final java.util.List<ResourceNode> nodes = new LinkedList<>();

    public ResourceGroup(Attributes attributes)
    {
      super(attributes);
    }

    protected ResourceGroup()
    {
      super((String)null);
    }

    /**
     * Add a sub node to this node.
     * @param node sub node to add
     */
    public void addNode(ResourceNode node)
    {
      nodes.add(node);
      String id = node.getId();
      idsToResources.put(id, node);
    }

    /**
     * Get the sub nodes of this node.
     *
     * @return {@code null} if this is a node which cannot have sub nodes,<br>
     *         array of sub nodes (possibly empty) otherwise
     */
    @Override
    public ResourceNode[] getSubNodes()
    {
      return nodes.toArray(EMPTY_RESOURCE_NODE_ARRAY);
    }

    /**
     * Get a sub node.
     * @param id id of sub node
     * @return sub node with given id or {@code null}
     */
    public ResourceNode getSubNode(String id)
    {
      return idsToResources.get(id);
    }

    /**
     * Register the resources.
     *
     * @param resourceMapping map where to register
     */
    @Override
    public void registerResources(Map<String, ResourceNode> resourceMapping)
    {
      for (ResourceNode resourceNode : nodes) {
        resourceNode.registerResources(resourceMapping);
      }
    }

    @Override
    protected void addXmlContent(StringBuilder sb)
    {
      sb.append("\n");
      for (ResourceNode resourceNode : nodes) {
        sb.append(resourceNode.toXml());
      }
    }

    /**
     * Get the XML tag of this node
     *
     * @return XML tag
     */
    @Override
    public String getXmlTag()
    {
      return TAG_GROUP;
    }

    /**
     * Get an unique id for this resource.
     * This should return either a unique id or {@code null} if this is no resource to look for
     *
     * @return unique id
     */
    @Override
    public String getId()
    {
      return null;
    }
  }

  /**
   * Special resource group for actions.
   */
  public static class ResourceActionGroup
          extends ResourceGroup
  {
    public static final String ACTION_GROUP_SUFFIX = "[ACTGRP]";
    /** Action name. */
    private final String actionName;

    private ResourceActionGroup(Attributes attributes)
    {
      super(attributes);
      actionName = attributes.getValue(ATTR_NAME);
    }

    /**
     * Create action group.
     * @param name action name
     */
    public ResourceActionGroup(String name)
    {
      actionName = name;
    }

    /**
     * Get the XML tag of this node
     *
     * @return XML tag
     */
    @Override
    public String getXmlTag()
    {
      return TAG_ACTION;
    }

    @Override
    protected void addXmlAttributes(StringBuilder xmlString)
    {
      super.addXmlAttributes(xmlString);
      addAttribute(xmlString, ATTR_NAME, actionName);
    }

    /**
     * Get an unique id for this resource.
     * This should return either a unique id or {@code null} if this is no resource to look for
     *
     * @return unique id
     */
    @Override
    public String getId()
    {
      return actionName;
    }

    static String getIdForSubGroup(String subGroupKey)
    {
      if (subGroupKey != null && subGroupKey.endsWith(I18n.ACTION_SUFFIX)) {
        int minus = subGroupKey.lastIndexOf('-', subGroupKey.length() - I18n.ACTION_SUFFIX.length());
        if (minus > 0) {
          String id = subGroupKey.substring(0, minus);
          return id + ACTION_GROUP_SUFFIX;
        }
      }
      return null;
    }
  }

  /**
   * Root group resource node in the XML file.
   */
  private static class I18nGroup
          extends ResourceGroup
  {
    private final String version;

    private I18nGroup(Attributes attributes)
    {
      super(attributes);
      version = attributes.getValue(ATTR_VERSION);
    }

    private I18nGroup()
    {
      version = Integer.toString(MY_VERSION);
    }

    @Override
    protected void addXmlAttributes(StringBuilder xmlString)
    {
      super.addXmlAttributes(xmlString);
      addAttribute(xmlString, ATTR_VERSION, escapeForXml(version));
    }

    public String getVersion()
    {
      return version;
    }

    /**
     * Get the XML tag of this node
     *
     * @return XML tag
     */
    @Override
    public String getXmlTag()
    {
      return TAG_I18N;
    }
  }

  /**
   *  Resource node defining an actual resource.
   */
  private static class Resource
          extends AbstractBasicResourceNode
  {
    /** The resource key. */
    private final String key;
    /** The resource value. */
    private Object value;
    /** The type of the value. */
    private ValueType type;
    /** Is this an inhereted resource? */
    private boolean inhereted;

    private Resource(Attributes attributes) throws SAXException
    {
      super(attributes);
      String keyValue = attributes.getValue(ATTR_KEY);
      if (keyValue == null) {
        throw new SAXException("No attribute "+ATTR_KEY+" in <"+TAG_RESOURCE+"> tag!");
      }
      key = unescapeFromXml(keyValue);
      String inh = attributes.getValue(ATTR_INHERET);
      if (inh != null) {
        inhereted = Boolean.valueOf(inh).booleanValue();
      }
      if (!inhereted) {
        String typeString = attributes.getValue(ATTR_TYPE);
        if (typeString == null) {
          throw new SAXException("No attribute "+ATTR_TYPE+" in <"+TAG_RESOURCE+"> tag!");
        }
        this.type = converterMapping.get(typeString);
        if (this.type == null) {
          throw new SAXException("Unknown value for type attribute in <"+TAG_RESOURCE+"> tag: '"+ typeString +"'!");
        }
      }
    }

    private Resource(String key, Object value, ValueType type, String comment)
    {
      super(comment);
      this.key = key;
      this.value = value;
      this.type = type;
    }

    private Resource(String key)
    {
      super((String)null);
      this.key = key;
      this.inhereted = true;
    }

    void setValue(Object value)
    {
      this.value = value;
    }

    void setValueFromString(String valueAsString)
    {
      if (!inhereted) {
        setValue(type.stringToValue(valueAsString));
      }
    }

    public String getKey()
    {
      return key;
    }

    public Object getValue()
    {
      return value;
    }

    public ValueType getConverter()
    {
      return type;
    }

    public boolean isInhereted()
    {
      return inhereted;
    }

    @Override
    protected void addXmlAttributes(StringBuilder xmlString)
    {
      addAttribute(xmlString, ATTR_KEY, escapeForXml(key));
      if (inhereted) {
        addAttribute(xmlString, ATTR_INHERET, "true");
      }
      else {
        addAttribute(xmlString, ATTR_TYPE, escapeForXml(type.getTypeName()));
      }
      super.addXmlAttributes(xmlString);
    }

    @Override
    protected void addXmlContent(StringBuilder sb)
    {
      if (!inhereted) {
        sb.append(type.valueToString(value));
      }
    }

    @Override
    public boolean equals(Object o)
    {
      if (this == o) {
        return true;
      }
      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      final Resource resource = (Resource)o;

      return inhereted == resource.inhereted &&
             (inhereted ||
              Objects.equals(getComment(), resource.getComment()) &&
              key.equals(resource.key) &&
              type.equals(resource.type) &&
              type.equals(value, resource.value));
    }

    @Override
    public int hashCode()
    {
      return Objects.hash(value, type, getConverter(), inhereted);
    }

    /**
     * Get the sub nodes of this node.
     *
     * @return {@code null} if this is a node which cannot have sub nodes,<br>
     *         array of sub nodes (possibly empty) otherwise
     */
    @Override
    public ResourceNode[] getSubNodes()
    {
      return null;
    }

    /**
     * Register the resources.
     *
     * @param resourceMapping map where to register
     */
    @Override
    public void registerResources(Map<String, ResourceNode> resourceMapping)
    {
      if (!inhereted) {
        resourceMapping.put(key, this);
      }
    }

    /**
     * Get the XML tag of this node
     *
     * @return XML tag
     */
    @Override
    public String getXmlTag()
    {
      return TAG_RESOURCE;
    }

    /**
     * Get an unique id for this resource.
     * This should return either a unique id or {@code null} if this is no resource to look for
     *
     * @return unique id
     */
    @Override
    public String getId()
    {
      return key;
    }

    @Override
    protected boolean hasContent()
    {
      return !inhereted;
    }
  }

  /** XML handler. */
  private class MyHandler
          extends DefaultHandler
  {
    private final Stack<ResourceGroup> groupStack = new Stack<>();
    private Resource pendingResource;
    private final StringBuilder pendingText = new StringBuilder();
    /**
     * Receive notification of the start of an element.
     *
     * <p>By default, do nothing.  Application writers may override this
     * method in a subclass to take specific actions at the start of
     * each element (such as allocating a new tree node or writing
     * output to a file).</p>
     *
     * @param uri         URI
     * @param localName   local name
     * @param qName       The element type name.
     * @param attributes The specified or defaulted attributes.
     * @throws SAXException Any SAX exception, possibly
     *                                  wrapping another exception.
     */
    @Override
    public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
    {
      if (TAG_RESOURCE.equals(qName)) {
        pendingResource = new Resource(attributes);
        pendingText.setLength(0);
        groupStack.peek().addNode(pendingResource);
      }
      else if (TAG_ACTION.equals(qName)) {
        ResourceActionGroup group = new ResourceActionGroup(attributes);
        groupStack.peek().addNode(group);
        groupStack.push(group);
      }
      else if (TAG_GROUP.equals(qName)) {
        ResourceGroup group = new ResourceGroup(attributes);
        groupStack.peek().addNode(group);
        groupStack.push(group);
      }
      else if (TAG_I18N.equals(qName)) {
        root = new I18nGroup(attributes);
        String version = root.getVersion();
        if (version == null) {
          throw new SAXException("No attribute "+ATTR_VERSION+" in <"+qName+"> tag!");
        }
        try {
          if (Integer.parseInt(version) != MY_VERSION) {
            throw new SAXException("File format version mismatch: need "+MY_VERSION+" but have "+version);
          }
        } catch (NumberFormatException e) {
          throw new SAXException("File format version mismatch: need "+MY_VERSION+" but have "+version);
        }
        groupStack.push(root);
      }
    }

    /**
     * Receive notification of the end of an element.
     *
     * <p>By default, do nothing.  Application writers may override this
     * method in a subclass to take specific actions at the end of
     * each element (such as finalising a tree node or writing
     * output to a file).</p>
     *
     * @param uri         URI
     * @param localName   local name
     * @param qName       The element type name.
     * @throws SAXException Any SAX exception, possibly
     *                                  wrapping another exception.
     */
    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException
    {
      if (TAG_RESOURCE.equals(qName)) {
        pendingResource.setValueFromString(pendingText.toString());
        resourceMap.put(pendingResource.getKey(), pendingResource);
        pendingResource = null;
      }
      else if (TAG_ACTION.equals(qName)) {
        groupStack.pop();
      }
      else if (TAG_GROUP.equals(qName)) {
        groupStack.pop();
      }
      else if (TAG_I18N.equals(qName)) {
        resourceMap.clear();
        root.registerResources(resourceMap);
        groupStack.pop();
      }
    }

    /**
     * Receive notification of character data inside an element.
     *
     * <p>By default, do nothing.  Application writers may override this
     * method to take specific actions for each chunk of character data
     * (such as adding the data to a node or buffer, or printing it to
     * a file).</p>
     *
     * @param ch     The characters.
     * @param start  The start position in the character array.
     * @param length The number of characters to use from the
     *               character array.
     * @throws SAXException Any SAX exception, possibly
     *                                  wrapping another exception.
     */
    @Override
    public void characters(char[] ch, int start, int length) throws SAXException
    {
      pendingText.append(new String(ch, start, length));
    }

    /**
     * Receive notification of ignorable whitespace in element content.
     *
     * <p>By default, do nothing.  Application writers may override this
     * method to take specific actions for each chunk of ignorable
     * whitespace (such as adding data to a node or buffer, or printing
     * it to a file).</p>
     *
     * @param ch     The whitespace characters.
     * @param start  The start position in the character array.
     * @param length The number of characters to use from the
     *               character array.
     * @throws SAXException Any SAX exception, possibly
     *                                  wrapping another exception.
     */
    @Override
    public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException
    {
      pendingText.append(new String(ch, start, length));
    }
  }

  /** Root of the resource tree. */
  private I18nGroup root = new I18nGroup();
  /** The mapping of keys to Resources. */
  private final Map<String, ResourceNode> resourceMap = new HashMap<>();

  /**
   *  Read an XML resource from the given input stream.
   *  @param is input stream
   *  @throws IOException on i/o errors
   *  @throws SAXException on XML parse errors
   *  @throws ParserConfigurationException on SAX parser configuration errors
   */
  public XmlResourceBundle(InputStream is)
          throws IOException, SAXException, ParserConfigurationException
  {
    if (is != null) {
      SAXParserFactory factory = SAXParserFactory.newInstance();
      SAXParser parser = factory.newSAXParser();
      parser.parse(is, new MyHandler());
    }
  }

  /**
   * Default constructor.
   */
  public XmlResourceBundle()
  {
  }

  /**
   * Returns an enumeration of the keys.
   */
  @NotNull
  @Override
  public Enumeration<String> getKeys()
  {
    return Collections.enumeration(resourceMap.keySet());
  }

  /**
   * Gets an object for the given key from this resource bundle.
   * Returns null if this resource bundle does not contain an
   * object for the given key.
   *
   * @param key the key for the desired object
   * @return the object for the given key, or null
   * @throws NullPointerException if {@code key} is {@code null}
   */
  @Override
  protected Object handleGetObject(@NotNull String key)
  {
    Resource resource = (Resource)resourceMap.get(key);
    return resource != null  ?  resource.getValue()  :  null;
  }

  public String getComment()
  {
    return root.getComment();
  }

  public void putStringResource(String key, String value)
  {
    putStringResource(key, value, null);
  }

  public void putStringResource(String key, String value, String comment)
  {
    putResource(key, value, STRING_VALUE_TYPE, comment);
  }

  public void putStringArrayResource(String key, String[] value)
  {
    putStringArrayResource(key, value, null);
  }

  public void putStringArrayResource(String key, String[] value, String comment)
  {
    putResource(key, value, STRING_ARRAY_VALUE_TYPE, comment);
  }

  public void putObjectResource(String key, Serializable value)
  {
    putObjectResource(key, value, null);
  }

  public void putObjectResource(String key, Serializable value, String comment)
  {
    putResource(key, value, OBJECT_VALUE_TYPE, comment);
  }

  public void putResource(String key, Object value, ValueType type)
  {
    putResource(key, value, type, null);
  }

  public void putInheretedResource(String key)
  {
    if (key == null) {
      throw new IllegalArgumentException("null key not allowed!");
    }
    putResource(new Resource(key));
  }

  private void putResource(ResourceNode resource)
  {
    String key = resource.getId();
    ResourceGroup myParent = root;
    String actionId = ResourceActionGroup.getIdForSubGroup(key);
    if (actionId != null) {
      // action resource
      // deliberately accepting class cast exception in the next line!
      ResourceActionGroup action = (ResourceActionGroup)root.getSubNode(actionId);
      if (action == null) {
        action = new ResourceActionGroup(actionId);
        root.addNode(action);
      }
      myParent = action;
    }
    myParent.addNode(resource);
    resource.registerResources(resourceMap);
  }

  public void putResource(String key, Object value, ValueType type, String comment)
  {
    if (key == null) {
      throw new IllegalArgumentException("null key not allowed!");
    }
    if (value == null) {
      throw new IllegalArgumentException("null value not allowed!");
    }
    if (type == null) {
      throw new IllegalArgumentException("null type not allowed!");
    }
    Resource resource = new Resource(key, value, type, comment);
    putResource(resource);
  }

  private static void write(OutputStream os, String str) throws IOException
  {
    os.write(str.getBytes("utf-8"));
    os.write('\n');
  }

  public void saveXml(OutputStream os) throws IOException
  {
    write(os, "<?xml version=\"1.0\" encoding=\"utf-8\"?>");
    write(os, root.toXml());
  }

  /**
   * Get a resource bundle with a given basename, using the default locale.
   * Compared to {@code ResourceBundle.getBundle()} this will also look for XML resource bundles.
   * <p>
   * The semantics of this is also differing, because it has no access to the class loader of the calling class.
   * It has to use {@code ClassLoader.getSystemClassLoader()} instead. Access to this method is restricted,
   * so an exception will be thrown in restricted environments, eg unsigned applets. Use
   * {@link #getResourceBundle(String, java.util.Locale, ClassLoader)} instead with an appropriate class loader.
   * @param baseName base name
   * @return best matching resource bundle
   */
  public static ResourceBundle getResourceBundle(String baseName)
  {
    return getResourceBundle(baseName, Locale.getDefault(), ClassLoader.getSystemClassLoader());
  }

  /**
   * Get a resource bundle with a given basename, using the given locale.
   * Compared to {@code ResourceBundle.getBundle()} this will also look for XML resource bundles.
   * <p>
   * The semantics of this is also differing, because it has no access to the class loader of the calling class.
   * It has to use {@code ClassLoader.getSystemClassLoader()} instead. Access to this method is restricted,
   * so an exception will be thrown in restricted environments, eg unsigned applets. Use
   * {@link #getResourceBundle(String, java.util.Locale, ClassLoader)} instead with an appropriate class loader.
   * @param baseName base name
   * @param locale locale to use for resource bundle resolve
   * @return best matching resource bundle
   */
  public static ResourceBundle getResourceBundle(String baseName, Locale locale)
  {
    return getResourceBundle(baseName, locale, ClassLoader.getSystemClassLoader());
  }

  /**
   * Get a resource bundle with a given basename, using the given locale and class loader.
   * Compared to {@code ResourceBundle.getBundle()} this will also look for XML resource bundles.
   *
   * @param baseName base name
   * @param locale locale to use for resource bundle resolve
   * @param classLoader class loader to use for loading resource bundles
   * @return best matching resource bundle
   */
  public static ResourceBundle getResourceBundle(String baseName, Locale locale, ClassLoader classLoader)
  {
    //noinspection ConstantIfStatement
    if (false) {
      // todo: this is making things very slow in applet world as long as the XML resource bundles are not integrated
      String loc = locale != null ? "_"+locale : "";
      String name = baseName.replace('.', '/') + loc + EXTENSION;
      InputStream is = classLoader.getResourceAsStream(name);
      if (is != null) {
        try {
          return new XmlResourceBundle(is);
        } catch (IOException e) {
          Debug.error(e);
        } catch (SAXException e) {
          Debug.error(e);
        } catch (ParserConfigurationException e) {
          Debug.error(e);
        } finally {
          try {
            is.close();
          } catch (IOException e) {
            Debug.error(e);
          }
        }
      }
    }
    return ResourceBundle.getBundle(baseName, locale == null ? Locale.getDefault() : locale, classLoader);
  }

  private static void addAttribute(StringBuilder sb, String attr, String value)
  {
    sb.append(' ')
            .append(escapeForXml(attr))
            .append("=\"")
            .append(escapeForXml(value))
            .append('"');
  }

  public static void main(String[] args) throws IOException, SAXException, ParserConfigurationException
  {
    XmlResourceBundle bundle1 = new XmlResourceBundle();
    bundle1.putStringResource("foo", "bar");
    bundle1.putStringResource("str", "blah", "Just a string \u263a");
    bundle1.putStringResource("ugly: \u00c4\u00d6\u00dc", "&amp;<=\"\"\u00e4\u00f6\u00fc\u00df>");
    bundle1.putStringArrayResource("arr", new String[] { "a", "bb", "c\nd\ne"});
    bundle1.putObjectResource("object", "A string handled as an object", "what about objects?");
    bundle1.putInheretedResource("inhereted");

    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    bundle1.saveXml(bos);
    bos.close();
    byte[] result = bos.toByteArray();
    System.out.println(new String(result, "utf-8"));

    ByteArrayInputStream bis = new ByteArrayInputStream(result);
    XmlResourceBundle bundle2 = new XmlResourceBundle(bis);

    if (bundle2.resourceMap.size() != bundle1.resourceMap.size()) {
      throw new InternalError("Size mismatch!");
    }

    bos = new ByteArrayOutputStream();
    bundle1.saveXml(bos);
    bos.close();
    System.out.println("=====================");
    System.out.println(new String(bos.toByteArray(), "utf-8"));

    for (Enumeration<String> e = bundle2.getKeys();  e.hasMoreElements();  ) {
      String key = e.nextElement();
      Resource r1 = (Resource)bundle1.resourceMap.get(key);
      Resource r2 = (Resource)bundle2.resourceMap.get(key);
      if (!r1.equals(r2)) {
        throw new InternalError("Resource mismatch on key '"+key+"'!");
      }
    }

    if (!Arrays.equals(result, bos.toByteArray())) {
      throw new InternalError("Output mismatch!");
    }
    
  }
}
