// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2010-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.i18n;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.lang.ref.WeakReference;
import java.util.Locale;
import java.util.function.Consumer;

/**
 * A wrapper for localization changes,
 * which wraps the original listener by holding it with a weak reference.
 * This allows garbage collection when the wrapped object is no longer referenced otherwise.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class WeakReferencedLocalizable
        implements Localizable
{
  /** The wrapped localizable. */
  @NotNull
  private final WeakReference<Localizable> ref;

  @Nullable
  private final Consumer<? super Localizable> remover;

  /**
   * Constructor.
   * This registers a remover which automatically removes this object
   * from the global localizatipn change list when the wrapped
   * localizable is garbage collected.
   * @param wrappedLocalizable localizable to wrap
   */
  public WeakReferencedLocalizable(@NotNull Localizable wrappedLocalizable)
  {
    this(wrappedLocalizable, I18n::removeLocalizationChangeListener);
  }

  /**
   * Weak localization wrapper.
   * @param wrappedLocalizable localizable to wrap
   * @param remover            remover which removes this object when its content is garbage collected
   */
  public WeakReferencedLocalizable(@NotNull Localizable wrappedLocalizable,
                                   @Nullable Consumer<? super Localizable> remover)
  {
    this.ref = new WeakReference<>(wrappedLocalizable);
    this.remover = remover;
  }

  @Nullable
  private Localizable getWrapped()
  {
    Localizable localizable = ref.get();
    if (localizable == null) {
      if (remover != null) {
        // remove this
        remover.accept(this);
      }
    }
    return localizable;
  }

  /**
   * Set the locale.
   *
   * @param l locale to set.
   */
  @Override
  public void setLocale(Locale l)
  {
    final Localizable localizable = getWrapped();
    if (localizable != null) {
      localizable.setLocale(l);
    }
  }

  /**
   * Get the locale.
   *
   * @return the locale
   */
  @Override
  public Locale getLocale()
  {
    final Localizable localizable = getWrapped();
    if (localizable != null) {
      return localizable.getLocale();
    }
    return null;
  }
}
