// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n;

import de.caff.annotation.NotNull;

import java.util.Locale;

/** 
 *  An internationalized runtime exception.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ResourcedRuntimeException
        extends RuntimeException
        implements ResourcedThrowable
{
  private static final long serialVersionUID = -35098071368240247L;

  /**
   * The resource tag.
   * @see I18n#getString
   * @see I18n#format
   */
  @NotNull
  protected final String   resTag;
  /**
   * The arguments for the resource.
   * @see I18n#format
   */
  protected  Object[] args  = null;
  /**
   *  Constructor for a simple I18n message.
   *  @param  tag  I18n resource tag of message
   *  @see de.caff.i18n.I18n#getString(String)
   */
  public ResourcedRuntimeException(@NotNull String tag)
  {
    resTag = tag;
  }

  /**
   *  Constructor for an I18n message with arguments.
   *  @param  tag   I18n resource tag for a message with arguments
   *  @param  args  arguments
   *  @see de.caff.i18n.I18n#format(String, Object...)
   */
  public ResourcedRuntimeException(@NotNull String tag, Object... args)
  {
    resTag    = tag;
    this.args = args;
  }

  /**
   *  Constructor for a simple I18n message after receiving another exception.
   *
   *  @param originalException wrapped exception
   *  @param tag   I18n resource tag for a message with arguments
   *  @see de.caff.i18n.I18n#getString(String)
   */
  public ResourcedRuntimeException(@NotNull Throwable originalException,
                                   @NotNull String tag)
  {
    this(tag);
    initCause(originalException);
  }

  /**
   *  Constructor for a I18n message with arguments after receiving another exception.
   *
   *  @param originalException wrapped exception
   *  @param tag               resource tag
   *  @param  args  arguments
   *  @see de.caff.i18n.I18n#format(String, Object...)
   */
  public ResourcedRuntimeException(@NotNull Throwable originalException,
                                   @NotNull String tag,
                                   Object... args)
  {
    this(tag, args);
    initCause(originalException);
  }

  /**
   *  Get the message with default locale.
   *  @return  message 
   */
  @Override
  public String getMessage()
  {
    return getMessage(null);
  }

  /**
   *  Get the message for a locale.
   *  @param l requested locale
   *  @return message for the given locale
   */
  @Override
  public String getMessage(Locale l)
  {
    if (args != null) {
      return I18n.format(l, resTag, args);
    }
    else {
      return I18n.getString(resTag, l);
    }
  }

  /**
   *  Get the exception from where this resourced exception originated.
   *  @return original exception or {@code null} if there is no original
   *          exception
   */
  @Override
  public Throwable getOriginalException()
  {
    return getCause();
  }

  /**
   *  Set the exception from where this resourced exception originated.
   *  @param originalException original exception
   */
  @Override
  public final void setOriginalException(Throwable originalException)
  {
    initCause(originalException);
  }
}
