// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n;

import de.caff.annotation.NotNull;

import java.util.*;

/**
 *  Handle a collection of resource bundles. This allows
 *  parts of an application (e.g. beans) to maintain
 *  their own resource bundle but access to all these bundles
 *  is handled as an unit.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ResourceBundleCollection
        extends ResourceBundle
{
  /** The collected resource bundles. */
  private final Collection<ResourceBundle>    resourceBundles = new LinkedList<>();
  /** The collected resources. */
  private Map<String, Object> resources       = new HashMap<>();
  /** The locale of this bundle. */
  private final Locale locale;

  /**
   * Constructor.
   * @param locale locale of this bundle
   */
  public ResourceBundleCollection(Locale locale)
  {
    this.locale = locale;
  }

  /**
   * Returns the locale of this resource bundle. This method can be used after a
   * call to getBundle() to determine whether the resource bundle returned really
   * corresponds to the requested locale or is a fallback.
   *
   * @return the locale of this resource bundle
   */
  @Override
  public Locale getLocale()
  {
    return locale;
  }

  /**
   *  Add an resource bundle.
   *  <em>If this bundle defines an already defined resource,
   *  the old resource will be overwritten!</em>
   *  @param  bundle   resource bundle to add
   */
  public void addResourceBundle(ResourceBundle bundle) {
    resourceBundles.add(bundle);
    addEntries(bundle);
  }

  /**
   *  Remove a resource bundle.
   *  @param  bundle   resource bundle to remove
   */
  public void removeResourceBundle(ResourceBundle bundle) {
    if (resourceBundles.remove(bundle)) {
      // really removed
      // --- now we need to create a new hash map because we cannot
      // --- easily just remove the key/value pairs due to correct 
      // --- handling of multiple entries
      resources = new HashMap<>();
      for (ResourceBundle resourceBundle : resourceBundles) {
        addEntries(resourceBundle);
      }
    }
  }

  /**
   *  Add the key/value pairs from a bundle to the collected list.
   *  @param  bundle   resource bundle for which the entries should be added
   */
  private void addEntries(ResourceBundle bundle) {
    for (Enumeration<String> e = bundle.getKeys();   e.hasMoreElements();  ) {
      String key = e.nextElement();
      if (I18n.DEBUG) {
	if (resources.get(key) != null) {
	  System.err.println("key '"+key+"'redefined in resource bundle "+bundle);
	}
      }
      resources.put(key, bundle.getObject(key));
    }
  }

  /**
   * Set the parent collection.
   * @param parent parent
   */
  void setParentBundle(ResourceBundle parent)
  {
    setParent(parent);
  }

  /**
   *  Must have method for getting the value for a key.
   *  @param  key  see description of java.util.ResourceBundle
   *  @return the value for the given key or {@code null}
   */
  @Override
  protected Object handleGetObject(@NotNull String key) {
    return resources.get(key);
  }

  /**
   *  Return an iterator of the keys.
   *  @return iterator of all keys from the collected tables.
   */
  @NotNull
  @Override
  public Enumeration<String> getKeys() {
    return Collections.enumeration(resources.keySet());
  }
}
