// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.Utility;
import de.caff.util.debug.Debug;

import javax.swing.*;
import java.awt.*;
import java.io.PrintStream;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 *  Application specific support for internationalisation.
 *  This allows support for several resource classes for
 *  one application but <b>has the big disadvantage that
 *  separate resource bundles may  not define the same
 *  keys</b>.  Beware of that! You can switch DEBUG on
 *  to see if this happens.<p>
 *  This design should be overworked when possible.
 *  <p>
 *  To allow for enhancements the previous static design
 *  is now switched to something more object-oriented.
 *  This allows to take advantages of the new possibilities
 *  introduced with Java 1.2.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public abstract class I18n
{
  /** Switch on to see whether there are key clashes. */
  static final boolean     DEBUG            = false;

  /** 
   *  A singleton of the real I18n class which manages the
   *  I18n handling.
   */
  private static I18n i18n;  // loaded later to avoid class loading deadlock

  private static final Object _LOCK = new Object();

  /** General suffix for action specific resources. */
  public static final String ACTION_SUFFIX = "[ACTION]";
  /** The i18n tag suffix added for a label/button text/menu text. */
  public static final String SUFFIX_TEXT        = "-NAME"+ACTION_SUFFIX;
  /** The i18n tag suffix added for a tooltip. */
  public static final String SUFFIX_TOOLTIP     = "-TTT"+ACTION_SUFFIX;
  /** The i18n tag suffix added for a description. */
  public static final String SUFFIX_DESCRIPTION = "-DESCR"+ACTION_SUFFIX;
  /** The i18n tag suffix added for an accelarator. */
  public static final String SUFFIX_ACCELERATOR = "-ACCEL"+ACTION_SUFFIX;
  /** The i18n tag suffix added for a mnemonic. */
  public static final String SUFFIX_MNEMONIC    = "-MNEMO"+ACTION_SUFFIX;
  /** The i18n tag suffix added for an icon path. */
  public static final String SUFFIX_ICON        = "-ICON"+ACTION_SUFFIX;
  /** The i18n tag suffix added for the path of an icon used for the disabled state. */
  public static final String SUFFIX_DISABLED_ICON  = "-ICON-DIS"+ACTION_SUFFIX;
  /** The i18n tag suffix added for the path of an icon used for the inactive state. */
  public static final String SUFFIX_INACTIVE_ICON  = "-ICON-INACT"+ACTION_SUFFIX;

  /**
   *  Add an application specific resource class base name.
   *  This should be called by any application/applet before
   *  any other i18n specific code is used.
   *  @param  base  base class name for resources
   *  @see java.util.ResourceBundle
   */
  public static void addAppResourceBase(@NotNull String base)
  {
    addAppResourceBase(base, false);
  }

  /**
   *  Add an application specific resource class base name.
   *  This should be called by any application/applet before
   *  any other i18n specific code is used.
   *  @param  base  base class name for resources
   *  @param  prepend prepend this resource base?
   *  @see java.util.ResourceBundle
   */
  public static void addAppResourceBase(@NotNull String base, boolean prepend)
  {
    if (i18n == null) {
      synchronized (_LOCK) {
        if (i18n == null) {
          i18n = new DefaultI18n();
        }
      }
    }
    i18n._addAppResourceBase(base, prepend);
  }

  /**
   *  Add an application specific resource class base name.
   *  This should be called by any application/applet before
   *  any other i18n specific code is used.
   *  @param  base  base class name for resources
   *  @param  prepend prepend this resource base?
   *  @see java.util.ResourceBundle
   */
  protected abstract void _addAppResourceBase(@NotNull String base, boolean prepend);

  /**
   *  Set the locale to be used as a default for the application.
   *  @param  l   locale to be used as default
   */
  public static void setDefaultLocale(Locale l)
  {
    i18n._setDefaultLocale(l);
  }

  /**
   *  Set the locale to be used as a default for the application.
   *  @param  l   locale to be used as default (system default if {@code null})
   */
  protected abstract void _setDefaultLocale(@Nullable Locale l);

  /**
   *  Get the locale to be used as a default for the application.
   *  @return   locale to be used as default
   */
  @NotNull
  public static Locale getDefaultLocale()
  {
    return i18n._getDefaultLocale();
  }

  /**
   *  Get the locale to be used as a default for the application.
   *  @return   locale to be used as default
   */
  @NotNull
  protected abstract Locale _getDefaultLocale();

  /**
   *  Get the fallback resource bundle.
   *  @return fallback bundle
   */
  protected abstract ResourceBundle _getFallbackBundle();

  /**
   *  Get a ResourceBundle for a locale.
   *  @param  l  locale
   *  @return ResourceBundle for that Locale
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  private static ResourceBundle getBundle(@Nullable Locale l)
    throws MissingResourceException 
  {
    return i18n._getBundle(l);
  }

  /**
   *  Get a ResourceBundle for a locale.
   *  @param  l  locale
   *  @return ResourceBundle for that Locale
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  @NotNull
  protected abstract ResourceBundle _getBundle(@Nullable Locale l)
    throws MissingResourceException;

  /**
   * Dump the complete currently known i18n resources for a given locale.
   * @param out    print stream where to dump to
   * @param locale locale for which to dump the resources
   */
  protected abstract void _dumpResources(@NotNull PrintStream out,
                                         @Nullable Locale locale);

  /**
   * Dump the complete currently known i18n resources for a given locale.
   * @param out    print stream where to dump to
   * @param locale locale for which to dump the resources
   */
  public static void dump(@NotNull PrintStream out,
                          @Nullable Locale locale)
  {
    i18n._dumpResources(out, locale);
  }

  /**
   *  Get a string for the default locale.
   *  @param   tag     resource tag
   *  @return  localized string
   *  @throws  MissingResourceException    when no appResourceBase is set
   */
  @NotNull
  public static String getString(@NotNull String tag)
          throws MissingResourceException
  {
    return getString(tag, null);
  }

  /**
   *  Get a String specified by a Locale.
   *  @param   tag     resource tag
   *  @param   l       Locale to be used
   *  @return  localized String
   *  @exception MissingResourceException  when no appResourceBase is set
   *  @see java.util.ResourceBundle#getString
   */
  @NotNull
  public static String getString(@NotNull String tag, @Nullable Locale l)
          throws MissingResourceException
  {
    // System.out.println("Tag = "+tag+", l = "+(l != null ? l.getDisplayName() :  "<unknown>")+", returning \'"+ getBundle(l).getString(tag)+"'");
    if (l == null) {
      l = i18n._getDefaultLocale(); 
    }
    try {
      return getBundle(l).getString(tag);
    } catch (MissingResourceException x) {
/*
      String variant = l.getVariant();
      if (variant != null  &&  variant.length() > 0) {
        return getString(tag, new Locale(l.getLanguage(), l.getCountry()));
      }
      String country = l.getCountry();
      if (country != null  &&  country.length() > 0) {
        return getString(tag, new Locale(l.getLanguage()));
      }
*/
      return i18n._getFallbackBundle().getString(tag);
    }
  }

  /**
   *  Compile a String with a format using the default locale.
   *  @see de.caff.util.Utility#compileString
   *  @param  tag   resource tag of format
   *  @param  args  arguments
   *  @return formatted string
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  @NotNull
  public static String format(@NotNull String tag, Object ... args)
    throws MissingResourceException 
  {
    return format(null, tag, args);
  }

  /**
   *  Compile a String with a format.
   *  @see de.caff.util.Utility#compileString
   *  @param  l     locale
   *  @param  tag   resource tag of format
   *  @param  args  arguments
   *  @return formatted string
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  @NotNull
  public static String format(@Nullable Locale l,
                              @NotNull String tag,
                              Object ... args)
    throws MissingResourceException
  {
    return Utility.compileString(tag, args, getBundle(l));
  }

  /**
   *  Compile a String with a format using the default locale.
   *  @see de.caff.util.Utility#compileString
   *  @param  tag   resource tag of format
   *  @param  args  arguments
   *  @return formatted string
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  @NotNull
  public static String getFormattedString(@NotNull String tag, Object ... args)
    throws MissingResourceException
  {
    return format(null, tag, args);
  }

  /**
   *  Compile a String with a format.
   *  @see de.caff.util.Utility#compileString
   *  @param  l     locale
   *  @param  tag   resource tag of format
   *  @param  args  arguments
   *  @return formatted string
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  @NotNull
  public static String getFormattedString(@Nullable Locale l, @NotNull String tag, Object ... args)
    throws MissingResourceException
  {
    return Utility.compileString(tag, args, getBundle(l));
  }

  /**
   *  Add a listener for localization changes.
   *  @param  localizable  listener for changes
   */
  public static void addLocalizationChangeListener(@NotNull Localizable localizable)
  {
    i18n._addLocalizationChangeListener(localizable);
  }

  /**
   *  Add a listener for localization changes.
   *  @param  localizable  listener for changes
   */
  protected abstract void _addLocalizationChangeListener(@NotNull Localizable localizable);

  /**
   *  Remove a listener for localization changes.
   *  @param  localizable  listener to be removed
   */
  public static void removeLocalizationChangeListener(@NotNull Localizable localizable)
  {
    i18n._removeLocalizationChangeListener(localizable);
  }

  /**
   *  Remove a listener for localization changes.
   *  @param  localizable  listener to be removed
   */
  protected abstract void _removeLocalizationChangeListener(@NotNull Localizable localizable);

  /**
   *  Tell all registered localizables of localization changes.
   *  @param  locale  new locale
   */
  protected abstract void _fireLocaleChanged(Locale locale);

  /**
   * Get an icon from a tag defining its location, using the default locale.
   * @param tag      i18n tag
   * @return icon or {@code null} if there is no icon defined for the tag
   *         or it cannot be loaded
   */
  @Nullable
  public static Icon getIcon(@NotNull String tag)
  {
    return getIcon(tag, null);
  }

  /**
   * Get an icon from a tag defining its location.
   * @param tag      i18n tag
   * @param locale   locale
   * @return icon or {@code null} if there is no icon defined for the tag
   *         or it cannot be loaded
   */
  @Nullable
  public static Icon getIcon(@NotNull String tag, @Nullable Locale locale)
  {
    try {
      String location = getString(tag, locale);
      Image image = Utility.loadImage(location);
      if (image != null) {
        return new ImageIcon(image);
      }
    } catch (MissingResourceException e) {
      Debug.warn(e);
    }
    return null;
  }
}

