// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Types;

import java.io.PrintStream;
import java.lang.ref.WeakReference;
import java.util.*;

/**
 *  Application specific support for internationalization,
 *  used for Java 1.2+. 
 *  This uses the weak reference feature to allow for easier garbage collection.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class DefaultI18n
        extends I18n
{
  /** Fallback locale. */
  private static final Locale FALLBACK_LOCALE = new Locale("", "", "");
  /** The resource bundles. */
  protected final Map<Locale,ResourceBundleCollection> resourceBundles  = new HashMap<>();
  /** The resource bases. */
  protected final List<String>      appResourceBases = new LinkedList<>();
  /** The default locale. */
  protected Locale    defaultLocale    = Locale.getDefault();

  /**
   *  Collection of known localizables, which have to be informed
   *  of localization changes.
   *  <p>
   *  The localizables are only weakly referenced here.
   */
  protected final List<WeakReference<Localizable>> localizables     = new ArrayList<>();

  /**
   * Add an application specific resource class base name.
   * This should be called by any application/applet before
   * any other i18n specific code is used.
   *
   * @param base    base class name for resources
   * @param prepend prepend this resource base?
   * @see java.util.ResourceBundle
   */
  @Override
  protected synchronized void _addAppResourceBase(@NotNull String base, boolean prepend)
  {
    if (!appResourceBases.contains(base)) {
      if (prepend) {
        appResourceBases.add(0, base);
      }
      else {
        appResourceBases.add(base);
      }
      resourceBundles.clear();
    }
  }

  /**
   *  This method is here because some JVM fail to load fallback resources.
   *  @param name full qualified resource class name
   *  @return fallback resources
   */
  private ResourceBundle getFallbackResourceBundle(String name)
  {
    return XmlResourceBundle.getResourceBundle(name, null, getClass().getClassLoader());
  }

  /**
   *  Set the locale to be used as a default for the application.
   *  @param  l   locale to be used as default
   */
  @Override
  protected synchronized void _setDefaultLocale(Locale l)
  {
    // System.out.println("Default locale: "+l);
    if (l == null) {
      l = Locale.getDefault();
    }
    defaultLocale = l;
    resourceBundles.clear();

    _fireLocaleChanged(l);
  }

  /**
   *  Get the locale to be used as a default for the application.
   *  @return   locale to be used as default
   */
  @NotNull
  @Override
  protected synchronized Locale _getDefaultLocale()
  {
    return defaultLocale;
  }

  /**
   *  Get a ResourceBundle for a locale.
   *  @param  l  locale
   *  @return ResourceBundle for that Locale
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  @NotNull
  @Override
  protected synchronized ResourceBundle _getBundle(@Nullable Locale l)
  {
    if (l == null) {
      l = defaultLocale;
    }
    ResourceBundle res = resourceBundles.get(l);
    if (res == null) {
      String language = l.getLanguage();
      String country = l.getCountry();
      String variant = l.getVariant();

      List<Locale> locales;
      if (!FALLBACK_LOCALE.equals(l)) {
        locales = new ArrayList<>(4);
        if (!variant.isEmpty()) {
          locales.add(l);
        }
        if (!country.isEmpty()) {
          locales.add((locales.isEmpty()) ?
                              l : new Locale(language, country, ""));
        }
        if (!language.isEmpty()) {
          locales.add((locales.isEmpty()) ?
                              l : new Locale(language, "", ""));
        }
        locales.add(FALLBACK_LOCALE);
      }
      else {
        locales = Types.asList(FALLBACK_LOCALE);
      }

      ResourceBundleCollection childCollection = null;

      for (Locale locale: locales) {
        ResourceBundle bundle = resourceBundles.get(locale);
        if (bundle == null) {
          // this resource is unknown yet
          if (appResourceBases.isEmpty()) {
            throw new MissingResourceException("No application specific resource base defined",
                                               "<unknown>",
                                               "");
          }


          ResourceBundleCollection collect = new ResourceBundleCollection(locale);
          if (childCollection != null) {
            childCollection.setParentBundle(collect);
          }

          for (String baseName : appResourceBases) {
            try {
              final ResourceBundle resourceBundle = XmlResourceBundle.getResourceBundle(baseName,
                                                                                        locale,
                                                                                        getClass().getClassLoader());
              if (locale.equals(resourceBundle.getLocale())) {
                collect.addResourceBundle(resourceBundle);
              }
            } catch (MissingResourceException e1) {
            }
          }

          resourceBundles.put(locale, collect);
          childCollection = collect;
        }
        else {
          break;
        }
      }
      res = resourceBundles.get(l);
    }
    return res;
  }

  /**
   * Get the fallback resource bundle.
   *
   * @return fallback bundle
   */
  @Override
  protected synchronized ResourceBundle _getFallbackBundle()
  {
    return _getBundle(FALLBACK_LOCALE);
  }

  /**
   *  Add a listener for localization changes.
   *  @param  localizable  listener for changes
   */
  @Override
  protected synchronized void _addLocalizationChangeListener(@NotNull Localizable localizable)
  {
    localizables.add(new WeakReference<>(localizable));
    localizable.setLocale(_getDefaultLocale());
  }

  /**
   *  Remove a listener for localization changes.
   *  @param  localizable  listener to be removed
   */
  @Override
  protected synchronized void _removeLocalizationChangeListener(@NotNull Localizable localizable)
  {
    for (ListIterator<WeakReference<Localizable>> iterator = localizables.listIterator(); iterator.hasNext();) {
      WeakReference<Localizable> next = iterator.next();
      if (localizable == next.get()) {
        iterator.remove();
        break;
      }
    }
  }

  /**
   *  Tell all registered localizables of localization changes.
   *  @param  locale  new locale
   */
  @Override
  protected synchronized void _fireLocaleChanged(Locale locale)
  {
    for (ListIterator<WeakReference<Localizable>> it = localizables.listIterator(); it.hasNext(); ) {
      WeakReference<Localizable> ref = it.next();
      Localizable loc = ref.get();
      if (loc != null) {
        loc.setLocale(locale);
      }
      else {
        // no longer referenced
        it.remove();
      }
    }
  }

  /**
   * Dump the complete currently known i18n resources for a given locale.
   * @param out    print stream where to dump to
   * @param locale locale for which to dump the resources
   */
  @Override
  protected synchronized void _dumpResources(@NotNull PrintStream out,
                                             @Nullable Locale locale)
  {
    ResourceBundle bundle = _getBundle(locale);
    List<String> keys = new LinkedList<>();
    for (Enumeration<String> k = bundle.getKeys();  k.hasMoreElements();  ) {
      String key = k.nextElement();
      keys.add(key);
    }
    Collections.sort(keys);
    out.println(String.format("Resource Dump for Locale [%s]", locale));
    for (String key: keys) {
      out.println(String.format("\"%s\":\t\"%s\"", key, bundle.getObject(key)));
    }
  }
}

