// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.ui;

import javax.swing.*;

/**
 *  Handler for UI Resource.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public abstract class UIResourceHandler
{
  /** The id of the ui resource. */
  private final String id;

  /**
   *  Constructor.
   *  @param id ID of resource
   */
  protected UIResourceHandler(String id)
  {
    this.id = id;
  }

  /**
   *  Get the id of this resource.
   *  @return the id
   */
  public String getID()
  {
    return id;
  }

  /**
   *  Get the string representation of the value of the resource.
   *  @param defaults the UI defaults from where the value is read
   *  @return the value in textual form
   */
  public String getValue(UIDefaults defaults)
  {
    return fromValue(defaults.get(id));
  }

  /**
   *  Extract the value from a string representation,
   *  @param uirc  UI resource collection
   *  @param value value in textual form
   *  @return {@code true} if it was possible to extract the value from the text,
   *          {@code false} otherwise
   */
  public boolean setValue(UIResourceCollection uirc, String value)
  {
    Object v = toValue(value);
    return v != null && uirc.setUIValue(getID(), v);
  }

  /**
   *  Convert a textual representation of a value to a value.
   *  @param text textual representation
   *  @return the value or {@code null} if the text cannot be converted
   *  @see #fromValue(Object)
   */
  protected abstract Object toValue(String text);

  /**
   * Get the string representation of a value.
   *
   * @param value the UI value
   * @return the value in textual form
   * @see #toValue(String)
   */
  protected String fromValue(Object value)
  {
    return value != null  ?  value.toString()  :  null;
  }

  /**
   *  Get the class handled by this resource handler.
   *  @return handled class
   */
  public abstract Class<?> getHandledClass();
}
